---
name: retro-p5
description: Transform videos and images into lo-fi pixel art using demoscene and permacomputing aesthetics. Effects include sparse pixels, ordered dithering, halftone dots, CRT phosphor simulation, crosshatching, and scanline blocks. Node.js + Sharp based, minimal dependencies. Outputs GIF or MP4.
---

# Retro P5 - Lo-Fi Pixel Processor

Lightweight video/image processor inspired by demoscene, permacomputing, and 8-bit aesthetics.

## Quick Start

```bash
cd /mnt/skills/user/retro-p5
npm install
node process.mjs input.mp4 output.gif -e sparse
```

## Effects

| Effect | Description | Best For |
|--------|-------------|----------|
| `sparse` | Gapped colored pixels with edge preservation | Stylized portraits, retro art |
| `scanline` | Horizontal run-length blocks | 1-bit bitmap aesthetic |
| `dither` | Ordered Bayer dithering (adjustable) | Game Boy, classic PC look |
| `halftone` | Circular/square dots by luminance | Newspaper, pop art |
| `crt` | RGB subpixels with scanline gaps | CRT monitor simulation |
| `edge` | Edge detection only | Line art, technical |
| `crosshatch` | Diagonal line hatching by darkness | Pen & ink drawing |
| `sparse-dither` | Dither then sparse (combined) | Extra texture |

## Palettes

| Palette | Colors | Style |
|---------|--------|-------|
| `sparse` | Gray, black, teal, coral, white | Default reference style |
| `gameboy` | 4 greens | Classic Game Boy |
| `mono` | Black, white | High contrast |
| `cga` | Black, cyan, magenta, white | IBM CGA |
| `amber` | Black, amber | Amber monochrome monitor |
| `green` | Black, green | Green phosphor monitor |
| `pastel` | Soft pastels | Gentle, modern |
| `vapor` | Hot pink, cyan, purple, green | Vaporwave |
| `sepia` | Brown tones | Vintage photo |
| `ink` | Paper white, ink black | Pen drawing |

## Usage Examples

```bash
# Basic sparse effect (matches reference images)
node process.mjs video.mp4 output.gif -e sparse

# Strong Game Boy dithering
node process.mjs video.mp4 output.gif -e dither -p gameboy --strength 0.8

# Maximum dither (8x8 matrix, full strength)
node process.mjs video.mp4 output.gif -e dither --matrix-size 8 --strength 1.0

# Halftone newspaper dots
node process.mjs video.mp4 output.gif -e halftone -p mono --dot-size 6

# CRT phosphor RGB subpixels
node process.mjs video.mp4 output.gif -e crt

# Pen & ink crosshatch
node process.mjs video.mp4 output.gif -e crosshatch -p ink

# Vaporwave sparse+dither combo
node process.mjs video.mp4 output.gif -e sparse-dither -p vapor

# Process single image
node process.mjs photo.jpg output.png -e sparse -p pastel

# Before/after comparison GIF (3s original, 3s transformed)
node process.mjs video.mp4 compare.gif -e sparse --compare
```

## Options Reference

```
--effect, -e      Effect to apply (default: sparse)
--palette, -p     Color palette (default: sparse)
--cell-size       Grid cell size for sparse/halftone (default: 10)
--pixel-size      Drawn pixel size within cell (default: 6)
--dot-size        Dot size for halftone (default: 8)
--strength, -s    Dither strength 0.0-1.0 (default: 0.5)
--matrix-size     Dither matrix: 2, 4, or 8 (default: 4)
--fps             Output framerate (default: 10)
--max-duration    Max video duration in seconds (default: 10)
--scale           Scale factor before processing (default: 1)
--compare         Generate before/after comparison GIF
--compare-duration  Duration per segment in seconds (default: 3)
```

## Effect Details

### Sparse Pixels
The signature effect. Creates gapped colored pixels on a neutral background.

Key features:
- Edge detection preserves contours (hair, faces)
- Only skips pixels similar to background (not dark areas)
- Contrast boost for better color separation
- Adjustable cell/pixel size for density control

```bash
# Denser pixels (smaller gaps)
node process.mjs input.mp4 output.gif -e sparse --cell-size 8 --pixel-size 6

# Sparser pixels (larger gaps)
node process.mjs input.mp4 output.gif -e sparse --cell-size 14 --pixel-size 8
```

### Dithering
Ordered Bayer dithering with adjustable intensity.

```bash
# Subtle dither
node process.mjs input.mp4 output.gif -e dither --strength 0.3

# Heavy dither pattern
node process.mjs input.mp4 output.gif -e dither --strength 1.0 --matrix-size 8
```

### Halftone
Variable-size dots based on image luminance.

```bash
# Small dots (more detail)
node process.mjs input.mp4 output.gif -e halftone --dot-size 4

# Large dots (pop art)
node process.mjs input.mp4 output.gif -e halftone --dot-size 12
```

## Post-Processing with FFmpeg/Gifsicle

Optimize output or create variations:

```bash
# Reduce colors + optimize
gifsicle -O3 --colors 32 output.gif -o output_opt.gif

# Add extra dithering
gifsicle --colors 16 --dither output.gif -o output_dither.gif

# Speed up
gifsicle --delay=5 output.gif -o output_fast.gif

# Scale down for smaller file
ffmpeg -i output.gif -vf "scale=iw/2:-1:flags=neighbor" output_small.gif
```

## Dependencies

- Node.js 18+
- Sharp (auto-installed via npm)
- FFmpeg (for video I/O)

```bash
npm install  # installs sharp
```

## File Size Comparison

| Effect | Typical Size | Notes |
|--------|--------------|-------|
| sparse | 400-800K | Gaps reduce entropy |
| scanline | 200-400K | Very compressible |
| dither | 1-2M | Noise increases size |
| halftone | 300-600K | Clean edges compress well |
| crt | 800K-1.5M | RGB detail adds size |

## Architecture

```
input video/image
    ↓
[ffmpeg extract frames]
    ↓
[sharp: load PNG buffer]
    ↓
[effect function: pixel manipulation]
    ↓
[sharp: output PNG buffer]
    ↓
[ffmpeg: compile to GIF/MP4]
    ↓
output
```

~400 lines of code. No opencv, no scikit-image, no heavy ML dependencies.

## References

- [128kb.eu](https://128kb.eu/) - Small file size art
- [Permacomputing](https://permacomputing.net/) - Sustainable computing
- [Demoscene](https://en.wikipedia.org/wiki/Demoscene) - Size-constrained digital art
