#!/usr/bin/env node
/**
 * Sparse Pixel Retro Processor v2
 * 
 * Lo-fi pixel aesthetics inspired by demoscene and permacomputing.
 * Uses Sharp for image processing, ffmpeg for video I/O.
 */

import sharp from 'sharp';
import { readFileSync, writeFileSync, mkdirSync, readdirSync, rmSync, copyFileSync } from 'fs';
import { execSync } from 'child_process';
import { join, extname } from 'path';
import { parseArgs } from 'util';

// ============================================================================
// PALETTES
// ============================================================================

const PALETTES = {
  sparse: [
    [200, 200, 200], // gray background
    [0, 0, 0],       // black
    [72, 185, 178],  // teal
    [205, 112, 100], // coral
    [255, 255, 255], // white
  ],
  gameboy: [
    [15, 56, 15],
    [48, 98, 48],
    [139, 172, 15],
    [155, 188, 15],
  ],
  mono: [
    [0, 0, 0],
    [255, 255, 255],
  ],
  cga: [
    [0, 0, 0],
    [85, 255, 255],
    [255, 85, 255],
    [255, 255, 255],
  ],
  amber: [
    [0, 0, 0],
    [255, 176, 0],
  ],
  green: [
    [0, 0, 0],
    [0, 255, 65],
  ],
  pastel: [
    [245, 245, 240], // off-white bg
    [255, 182, 193], // pink
    [255, 218, 185], // peach
    [186, 255, 201], // mint
    [186, 225, 255], // sky
    [221, 160, 221], // plum
  ],
  vapor: [
    [20, 20, 40],    // dark blue bg
    [255, 113, 206], // hot pink
    [1, 205, 254],   // cyan
    [185, 103, 255], // purple
    [5, 255, 161],   // green
  ],
  sepia: [
    [44, 33, 24],
    [112, 87, 63],
    [181, 148, 105],
    [235, 220, 178],
  ],
  ink: [
    [250, 250, 245], // paper
    [20, 20, 30],    // ink
    [80, 60, 50],    // brown ink
  ],
};

// Bayer matrices for dithering
const BAYER2 = [
  [0, 2],
  [3, 1],
].map(row => row.map(v => v / 4));

const BAYER4 = [
  [0, 8, 2, 10],
  [12, 4, 14, 6],
  [3, 11, 1, 9],
  [15, 7, 13, 5],
].map(row => row.map(v => v / 16));

const BAYER8 = [
  [0, 32, 8, 40, 2, 34, 10, 42],
  [48, 16, 56, 24, 50, 18, 58, 26],
  [12, 44, 4, 36, 14, 46, 6, 38],
  [60, 28, 52, 20, 62, 30, 54, 22],
  [3, 35, 11, 43, 1, 33, 9, 41],
  [51, 19, 59, 27, 49, 17, 57, 25],
  [15, 47, 7, 39, 13, 45, 5, 37],
  [63, 31, 55, 23, 61, 29, 53, 21],
].map(row => row.map(v => v / 64));

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

function nearestColor(r, g, b, palette) {
  let minDist = Infinity;
  let nearest = palette[0];
  for (const [pr, pg, pb] of palette) {
    const dist = (r - pr) ** 2 + (g - pg) ** 2 + (b - pb) ** 2;
    if (dist < minDist) {
      minDist = dist;
      nearest = [pr, pg, pb];
    }
  }
  return nearest;
}

function clamp(val, min, max) {
  return Math.max(min, Math.min(max, val));
}

function luminance(r, g, b) {
  return (r * 0.299 + g * 0.587 + b * 0.114) / 255;
}

// Simple edge detection using Sobel-like operator
function detectEdges(data, width, height, channels) {
  const edges = new Float32Array(width * height);
  
  for (let y = 1; y < height - 1; y++) {
    for (let x = 1; x < width - 1; x++) {
      const idx = (y * width + x) * channels;
      
      const getL = (dx, dy) => {
        const i = ((y + dy) * width + (x + dx)) * channels;
        return luminance(data[i], data[i + 1], data[i + 2]);
      };
      
      const gx = -getL(-1, -1) + getL(1, -1) - 2 * getL(-1, 0) + 2 * getL(1, 0) - getL(-1, 1) + getL(1, 1);
      const gy = -getL(-1, -1) - 2 * getL(0, -1) - getL(1, -1) + getL(-1, 1) + 2 * getL(0, 1) + getL(1, 1);
      
      edges[y * width + x] = Math.sqrt(gx * gx + gy * gy);
    }
  }
  
  return edges;
}

// ============================================================================
// EFFECTS
// ============================================================================

/**
 * Sparse pixel effect v2 - with edge preservation
 * Now properly handles dark areas like hair
 */
async function sparsePixels(inputBuffer, options = {}) {
  const {
    cellSize = 10,
    pixelSize = 6,
    palette = PALETTES.sparse,
    bgColor = palette[0],
    edgeBoost = true,
    contrastBoost = 1.2,
    skipBgOnly = true,
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const edges = edgeBoost ? detectEdges(data, info.width, info.height, info.channels) : null;

  const outWidth = Math.ceil(width / cellSize) * cellSize;
  const outHeight = Math.ceil(height / cellSize) * cellSize;
  const output = Buffer.alloc(outWidth * outHeight * 4);

  // Fill with background
  for (let i = 0; i < output.length; i += 4) {
    output[i] = bgColor[0];
    output[i + 1] = bgColor[1];
    output[i + 2] = bgColor[2];
    output[i + 3] = 255;
  }

  const margin = Math.floor((cellSize - pixelSize) / 2);
  const searchPalette = palette.slice(1);

  for (let cy = 0; cy < height; cy += cellSize) {
    for (let cx = 0; cx < width; cx += cellSize) {
      const sx = Math.min(cx + Math.floor(cellSize / 2), info.width - 1);
      const sy = Math.min(cy + Math.floor(cellSize / 2), info.height - 1);
      const idx = (sy * info.width + sx) * info.channels;

      let r = data[idx];
      let g = data[idx + 1];
      let b = data[idx + 2];

      if (contrastBoost !== 1) {
        r = clamp((r - 128) * contrastBoost + 128, 0, 255);
        g = clamp((g - 128) * contrastBoost + 128, 0, 255);
        b = clamp((b - 128) * contrastBoost + 128, 0, 255);
      }

      if (skipBgOnly) {
        const bgDist = Math.sqrt(
          (r - bgColor[0]) ** 2 + (g - bgColor[1]) ** 2 + (b - bgColor[2]) ** 2
        );
        const edgeStrength = edges ? edges[sy * info.width + sx] : 0;
        if (bgDist < 40 && edgeStrength < 0.15) {
          continue;
        }
      }

      const [nr, ng, nb] = nearestColor(r, g, b, searchPalette);

      for (let py = 0; py < pixelSize; py++) {
        for (let px = 0; px < pixelSize; px++) {
          const outX = cx + margin + px;
          const outY = cy + margin + py;
          if (outX < outWidth && outY < outHeight) {
            const outIdx = (outY * outWidth + outX) * 4;
            output[outIdx] = nr;
            output[outIdx + 1] = ng;
            output[outIdx + 2] = nb;
            output[outIdx + 3] = 255;
          }
        }
      }
    }
  }

  return sharp(output, {
    raw: { width: outWidth, height: outHeight, channels: 4 }
  }).png().toBuffer();
}

/**
 * Scanline effect - horizontal run-length blocks
 */
async function scanlineBlocks(inputBuffer, options = {}) {
  const {
    blockHeight = 4,
    minRunLength = 4,
    palette = PALETTES.mono,
    threshold = 128,
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data } = await image.grayscale().raw().toBuffer({ resolveWithObject: true });

  const output = Buffer.alloc(width * height * 4);

  for (let y = 0; y < height; y++) {
    let runStart = 0;
    let runValue = data[y * width] > threshold ? 1 : 0;

    for (let x = 0; x <= width; x++) {
      const currentValue = x < width ? (data[y * width + x] > threshold ? 1 : 0) : -1;

      if (currentValue !== runValue || x === width) {
        const runLength = x - runStart;
        const colorIdx = runLength >= minRunLength ? runValue : 1 - runValue;
        const color = palette[colorIdx] || palette[0];

        for (let rx = runStart; rx < x; rx++) {
          const idx = (y * width + rx) * 4;
          output[idx] = color[0];
          output[idx + 1] = color[1];
          output[idx + 2] = color[2];
          output[idx + 3] = 255;
        }

        runStart = x;
        runValue = currentValue;
      }
    }
  }

  return sharp(output, { raw: { width, height, channels: 4 } }).png().toBuffer();
}

/**
 * Ordered Bayer dithering with adjustable strength
 */
async function bayerDither(inputBuffer, options = {}) {
  const {
    palette = PALETTES.mono,
    matrixSize = 4,
    strength = 0.5,
    pixelScale = 1,
  } = options;

  const matrix = matrixSize === 2 ? BAYER2 : matrixSize === 8 ? BAYER8 : BAYER4;
  const mSize = matrix.length;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const output = Buffer.alloc(width * height * 4);

  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const idx = (y * info.width + x) * info.channels;
      const r = data[idx];
      const g = data[idx + 1];
      const b = data[idx + 2];

      const bx = Math.floor(x / pixelScale) % mSize;
      const by = Math.floor(y / pixelScale) % mSize;
      const threshold = matrix[by][bx];

      const offset = (threshold - 0.5) * strength;
      const adjustedR = clamp((r / 255 + offset) * 255, 0, 255);
      const adjustedG = clamp((g / 255 + offset) * 255, 0, 255);
      const adjustedB = clamp((b / 255 + offset) * 255, 0, 255);

      const [nr, ng, nb] = nearestColor(adjustedR, adjustedG, adjustedB, palette);

      const outIdx = (y * width + x) * 4;
      output[outIdx] = nr;
      output[outIdx + 1] = ng;
      output[outIdx + 2] = nb;
      output[outIdx + 3] = 255;
    }
  }

  return sharp(output, { raw: { width, height, channels: 4 } }).png().toBuffer();
}

/**
 * Halftone effect - circular dots based on luminance
 */
async function halftone(inputBuffer, options = {}) {
  const {
    dotSize = 8,
    palette = PALETTES.mono,
    angle = 45,
    shape = 'circle',
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const output = Buffer.alloc(width * height * 4);
  const bgColor = palette[0];
  const fgColor = palette[palette.length - 1];

  for (let i = 0; i < output.length; i += 4) {
    output[i] = bgColor[0];
    output[i + 1] = bgColor[1];
    output[i + 2] = bgColor[2];
    output[i + 3] = 255;
  }

  const angleRad = (angle * Math.PI) / 180;
  const cos = Math.cos(angleRad);
  const sin = Math.sin(angleRad);

  for (let cy = 0; cy < height; cy += dotSize) {
    for (let cx = 0; cx < width; cx += dotSize) {
      let totalLum = 0;
      let count = 0;
      for (let dy = 0; dy < dotSize && cy + dy < height; dy++) {
        for (let dx = 0; dx < dotSize && cx + dx < width; dx++) {
          const idx = ((cy + dy) * info.width + (cx + dx)) * info.channels;
          totalLum += luminance(data[idx], data[idx + 1], data[idx + 2]);
          count++;
        }
      }
      const avgLum = totalLum / count;
      const maxRadius = dotSize / 2;
      const radius = maxRadius * (1 - avgLum);

      const centerX = cx + dotSize / 2;
      const centerY = cy + dotSize / 2;

      for (let dy = 0; dy < dotSize && cy + dy < height; dy++) {
        for (let dx = 0; dx < dotSize && cx + dx < width; dx++) {
          const px = cx + dx;
          const py = cy + dy;

          let relX = px - centerX;
          let relY = py - centerY;
          const rotX = relX * cos - relY * sin;
          const rotY = relX * sin + relY * cos;

          let dist;
          if (shape === 'square') {
            dist = Math.max(Math.abs(rotX), Math.abs(rotY));
          } else if (shape === 'diamond') {
            dist = Math.abs(rotX) + Math.abs(rotY);
          } else {
            dist = Math.sqrt(rotX * rotX + rotY * rotY);
          }

          if (dist <= radius) {
            const outIdx = (py * width + px) * 4;
            output[outIdx] = fgColor[0];
            output[outIdx + 1] = fgColor[1];
            output[outIdx + 2] = fgColor[2];
            output[outIdx + 3] = 255;
          }
        }
      }
    }
  }

  return sharp(output, { raw: { width, height, channels: 4 } }).png().toBuffer();
}

/**
 * CRT phosphor simulation - RGB subpixels with bloom
 */
async function crtPhosphor(inputBuffer, options = {}) {
  const {
    pixelSize = 6,
    scanlineGap = 2,
    brightness = 1.2,
    subpixelWidth = 2,
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const cellHeight = pixelSize + scanlineGap;
  const cellWidth = subpixelWidth * 3;
  
  const outWidth = Math.ceil(width / cellWidth) * cellWidth;
  const outHeight = Math.ceil(height / cellHeight) * cellHeight;
  const output = Buffer.alloc(outWidth * outHeight * 4);

  for (let i = 0; i < output.length; i += 4) {
    output[i] = 0;
    output[i + 1] = 0;
    output[i + 2] = 0;
    output[i + 3] = 255;
  }

  for (let cy = 0; cy < height; cy += cellHeight) {
    for (let cx = 0; cx < width; cx += cellWidth) {
      const sx = Math.min(cx, info.width - 1);
      const sy = Math.min(cy, info.height - 1);
      const idx = (sy * info.width + sx) * info.channels;

      const r = clamp(data[idx] * brightness, 0, 255);
      const g = clamp(data[idx + 1] * brightness, 0, 255);
      const b = clamp(data[idx + 2] * brightness, 0, 255);

      for (let py = 0; py < pixelSize && cy + py < outHeight; py++) {
        for (let px = 0; px < subpixelWidth && cx + px < outWidth; px++) {
          const outIdx = ((cy + py) * outWidth + cx + px) * 4;
          output[outIdx] = r;
        }
        for (let px = 0; px < subpixelWidth && cx + subpixelWidth + px < outWidth; px++) {
          const outIdx = ((cy + py) * outWidth + cx + subpixelWidth + px) * 4;
          output[outIdx + 1] = g;
        }
        for (let px = 0; px < subpixelWidth && cx + subpixelWidth * 2 + px < outWidth; px++) {
          const outIdx = ((cy + py) * outWidth + cx + subpixelWidth * 2 + px) * 4;
          output[outIdx + 2] = b;
        }
      }
    }
  }

  return sharp(output, { raw: { width: outWidth, height: outHeight, channels: 4 } }).png().toBuffer();
}

/**
 * Edge-only effect
 */
async function edgeOnly(inputBuffer, options = {}) {
  const {
    threshold = 0.2,
    palette = PALETTES.mono,
    invert = false,
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const edges = detectEdges(data, info.width, info.height, info.channels);
  const output = Buffer.alloc(width * height * 4);

  const bgColor = invert ? palette[palette.length - 1] : palette[0];
  const fgColor = invert ? palette[0] : palette[palette.length - 1];

  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const edgeVal = edges[y * width + x];
      const isEdge = edgeVal > threshold;
      const color = isEdge ? fgColor : bgColor;
      const idx = (y * width + x) * 4;
      output[idx] = color[0];
      output[idx + 1] = color[1];
      output[idx + 2] = color[2];
      output[idx + 3] = 255;
    }
  }

  return sharp(output, { raw: { width, height, channels: 4 } }).png().toBuffer();
}

/**
 * Crosshatch effect - diagonal lines based on luminance
 */
async function crosshatch(inputBuffer, options = {}) {
  const {
    lineSpacing = 6,
    palette = PALETTES.ink,
    levels = 4,
  } = options;

  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const output = Buffer.alloc(width * height * 4);
  const bgColor = palette[0];
  const lineColor = palette[palette.length - 1];

  for (let i = 0; i < output.length; i += 4) {
    output[i] = bgColor[0];
    output[i + 1] = bgColor[1];
    output[i + 2] = bgColor[2];
    output[i + 3] = 255;
  }

  for (let y = 0; y < height; y++) {
    for (let x = 0; x < width; x++) {
      const idx = (y * info.width + x) * info.channels;
      const lum = luminance(data[idx], data[idx + 1], data[idx + 2]);
      const darkness = 1 - lum;
      const level = Math.floor(darkness * levels);

      let drawLine = false;

      if (level >= 1 && (x + y) % lineSpacing === 0) drawLine = true;
      if (level >= 2 && (x - y + 1000) % lineSpacing === 0) drawLine = true;
      if (level >= 3 && y % Math.floor(lineSpacing / 2) === 0) drawLine = true;
      if (level >= 4 && x % Math.floor(lineSpacing / 2) === 0) drawLine = true;

      if (drawLine) {
        const outIdx = (y * width + x) * 4;
        output[outIdx] = lineColor[0];
        output[outIdx + 1] = lineColor[1];
        output[outIdx + 2] = lineColor[2];
      }
    }
  }

  return sharp(output, { raw: { width, height, channels: 4 } }).png().toBuffer();
}

/**
 * Sparse + Dither combination
 */
async function sparseDither(inputBuffer, options = {}) {
  // First apply dither
  const dithered = await bayerDither(inputBuffer, {
    ...options,
    strength: options.strength || 0.3,
  });
  // Then apply sparse
  return sparsePixels(dithered, options);
}

/**
 * Edge + Sparse combination - edges highlighted
 */
async function edgeSparse(inputBuffer, options = {}) {
  const image = sharp(inputBuffer);
  const metadata = await image.metadata();
  const { width, height } = metadata;

  const { data, info } = await image.raw().toBuffer({ resolveWithObject: true });

  const {
    cellSize = 10,
    pixelSize = 6,
    palette = PALETTES.sparse,
    bgColor = palette[0],
    edgeColor = palette[1], // Use darkest for edges
  } = options;

  const edges = detectEdges(data, info.width, info.height, info.channels);

  const outWidth = Math.ceil(width / cellSize) * cellSize;
  const outHeight = Math.ceil(height / cellSize) * cellSize;
  const output = Buffer.alloc(outWidth * outHeight * 4);

  // Fill background
  for (let i = 0; i < output.length; i += 4) {
    output[i] = bgColor[0];
    output[i + 1] = bgColor[1];
    output[i + 2] = bgColor[2];
    output[i + 3] = 255;
  }

  const margin = Math.floor((cellSize - pixelSize) / 2);
  const searchPalette = palette.slice(1);

  for (let cy = 0; cy < height; cy += cellSize) {
    for (let cx = 0; cx < width; cx += cellSize) {
      const sx = Math.min(cx + Math.floor(cellSize / 2), info.width - 1);
      const sy = Math.min(cy + Math.floor(cellSize / 2), info.height - 1);
      const idx = (sy * info.width + sx) * info.channels;

      const r = data[idx];
      const g = data[idx + 1];
      const b = data[idx + 2];

      const bgDist = Math.sqrt(
        (r - bgColor[0]) ** 2 + (g - bgColor[1]) ** 2 + (b - bgColor[2]) ** 2
      );
      const edgeStrength = edges[sy * info.width + sx];

      if (bgDist < 40 && edgeStrength < 0.15) continue;

      // Use edge color for strong edges, otherwise find nearest
      let nr, ng, nb;
      if (edgeStrength > 0.3) {
        [nr, ng, nb] = edgeColor;
      } else {
        [nr, ng, nb] = nearestColor(r, g, b, searchPalette);
      }

      for (let py = 0; py < pixelSize; py++) {
        for (let px = 0; px < pixelSize; px++) {
          const outX = cx + margin + px;
          const outY = cy + margin + py;
          if (outX < outWidth && outY < outHeight) {
            const outIdx = (outY * outWidth + outX) * 4;
            output[outIdx] = nr;
            output[outIdx + 1] = ng;
            output[outIdx + 2] = nb;
            output[outIdx + 3] = 255;
          }
        }
      }
    }
  }

  return sharp(output, {
    raw: { width: outWidth, height: outHeight, channels: 4 }
  }).png().toBuffer();
}

const EFFECTS = {
  sparse: sparsePixels,
  scanline: scanlineBlocks,
  dither: bayerDither,
  halftone,
  crt: crtPhosphor,
  edge: edgeOnly,
  crosshatch,
  'sparse-dither': sparseDither,
};

// ============================================================================
// VIDEO PROCESSING
// ============================================================================

/**
 * Compare mode - creates before/after GIF (N seconds original, N seconds transformed)
 */
async function processVideoCompare(inputPath, outputPath, effect, options = {}) {
  const { fps = 10, compareDuration = 3, scale = 1 } = options;

  const tmpDir = '/tmp/retro-frames';
  const inFrames = join(tmpDir, 'in');
  const outFrames = join(tmpDir, 'out');
  const compareFrames = join(tmpDir, 'compare');

  try { rmSync(tmpDir, { recursive: true }); } catch {}
  mkdirSync(inFrames, { recursive: true });
  mkdirSync(outFrames, { recursive: true });
  mkdirSync(compareFrames, { recursive: true });

  console.log(`Extracting ${compareDuration}s from ${inputPath}...`);

  const scaleFilter = scale !== 1 ? `,scale=iw*${scale}:ih*${scale}:flags=neighbor` : '';
  execSync(
    `ffmpeg -y -i "${inputPath}" -t ${compareDuration} -vf "fps=${fps}${scaleFilter}" "${inFrames}/%04d.png"`,
    { stdio: 'pipe' }
  );

  const frameFiles = readdirSync(inFrames).filter(f => f.endsWith('.png')).sort();
  console.log(`Processing ${frameFiles.length} frames with "${effect}" effect...`);

  const effectFn = EFFECTS[effect] || sparsePixels;

  for (let i = 0; i < frameFiles.length; i++) {
    const inPath = join(inFrames, frameFiles[i]);
    const outPath = join(outFrames, frameFiles[i]);

    const inputBuffer = readFileSync(inPath);
    const outputBuffer = await effectFn(inputBuffer, options);
    writeFileSync(outPath, outputBuffer);

    if ((i + 1) % 10 === 0 || i === frameFiles.length - 1) {
      process.stdout.write(`\r  ${i + 1}/${frameFiles.length} frames`);
    }
  }
  console.log('');

  console.log('Creating comparison sequence...');
  let frameNum = 1;

  // Get original frame dimensions
  const firstFrame = readFileSync(join(inFrames, frameFiles[0]));
  const { width: origWidth, height: origHeight } = await sharp(firstFrame).metadata();

  // Copy original frames first
  for (const file of frameFiles) {
    const src = join(inFrames, file);
    const dst = join(compareFrames, String(frameNum).padStart(4, '0') + '.png');
    copyFileSync(src, dst);
    frameNum++;
  }

  // Then copy transformed frames (resized to match original dimensions, flatten to RGB)
  for (const file of frameFiles) {
    const src = join(outFrames, file);
    const dst = join(compareFrames, String(frameNum).padStart(4, '0') + '.png');
    const transformed = readFileSync(src);
    const resized = await sharp(transformed)
      .resize(origWidth, origHeight)
      .flatten({ background: { r: 200, g: 200, b: 200 } })
      .png()
      .toBuffer();
    writeFileSync(dst, resized);
    frameNum++;
  }

  console.log(`Compiling ${frameNum - 1} frames to ${outputPath}...`);

  execSync(
    `ffmpeg -y -framerate ${fps} -i "${compareFrames}/%04d.png" -vf "palettegen=max_colors=64" "${tmpDir}/palette.png"`,
    { stdio: 'pipe' }
  );
  execSync(
    `ffmpeg -y -framerate ${fps} -i "${compareFrames}/%04d.png" -i "${tmpDir}/palette.png" -filter_complex "[0:v][1:v]paletteuse=dither=bayer" "${outputPath}"`,
    { stdio: 'pipe' }
  );

  rmSync(tmpDir, { recursive: true });
  console.log(`✓ Compare GIF saved: ${outputPath} (${compareDuration}s original + ${compareDuration}s transformed)`);
}

async function processVideo(inputPath, outputPath, effect, options = {}) {
  const { fps = 10, maxDuration = 10, scale = 1 } = options;

  const tmpDir = '/tmp/retro-frames';
  const inFrames = join(tmpDir, 'in');
  const outFrames = join(tmpDir, 'out');

  try { rmSync(tmpDir, { recursive: true }); } catch {}
  mkdirSync(inFrames, { recursive: true });
  mkdirSync(outFrames, { recursive: true });

  console.log(`Extracting frames from ${inputPath}...`);

  const scaleFilter = scale !== 1 ? `,scale=iw*${scale}:ih*${scale}:flags=neighbor` : '';
  execSync(
    `ffmpeg -y -i "${inputPath}" -t ${maxDuration} -vf "fps=${fps}${scaleFilter}" "${inFrames}/%04d.png"`,
    { stdio: 'pipe' }
  );

  const frameFiles = readdirSync(inFrames).filter(f => f.endsWith('.png')).sort();
  console.log(`Processing ${frameFiles.length} frames with "${effect}" effect...`);

  const effectFn = EFFECTS[effect] || sparsePixels;

  for (let i = 0; i < frameFiles.length; i++) {
    const inPath = join(inFrames, frameFiles[i]);
    const outPath = join(outFrames, frameFiles[i]);

    const inputBuffer = readFileSync(inPath);
    const outputBuffer = await effectFn(inputBuffer, options);
    writeFileSync(outPath, outputBuffer);

    if ((i + 1) % 10 === 0 || i === frameFiles.length - 1) {
      process.stdout.write(`\r  ${i + 1}/${frameFiles.length} frames`);
    }
  }
  console.log('');

  console.log(`Compiling to ${outputPath}...`);

  const ext = extname(outputPath).toLowerCase();
  if (ext === '.gif') {
    execSync(
      `ffmpeg -y -framerate ${fps} -i "${outFrames}/%04d.png" -vf "palettegen=max_colors=64" "${tmpDir}/palette.png"`,
      { stdio: 'pipe' }
    );
    execSync(
      `ffmpeg -y -framerate ${fps} -i "${outFrames}/%04d.png" -i "${tmpDir}/palette.png" -filter_complex "[0:v][1:v]paletteuse=dither=bayer" "${outputPath}"`,
      { stdio: 'pipe' }
    );
  } else {
    execSync(
      `ffmpeg -y -framerate ${fps} -i "${outFrames}/%04d.png" -c:v libx264 -pix_fmt yuv420p "${outputPath}"`,
      { stdio: 'pipe' }
    );
  }

  rmSync(tmpDir, { recursive: true });
  console.log(`✓ Output saved to: ${outputPath}`);
}

async function processImage(inputPath, outputPath, effect, options = {}) {
  const effectFn = EFFECTS[effect] || sparsePixels;
  const inputBuffer = readFileSync(inputPath);
  const outputBuffer = await effectFn(inputBuffer, options);
  writeFileSync(outputPath, outputBuffer);
  console.log(`✓ Output saved to: ${outputPath}`);
}

// ============================================================================
// CLI
// ============================================================================

const { values, positionals } = parseArgs({
  allowPositionals: true,
  options: {
    effect: { type: 'string', short: 'e', default: 'sparse' },
    palette: { type: 'string', short: 'p', default: 'sparse' },
    'cell-size': { type: 'string', default: '10' },
    'pixel-size': { type: 'string', default: '6' },
    'dot-size': { type: 'string', default: '8' },
    strength: { type: 'string', short: 's', default: '0.5' },
    fps: { type: 'string', default: '10' },
    'max-duration': { type: 'string', default: '10' },
    scale: { type: 'string', default: '1' },
    'matrix-size': { type: 'string', default: '4' },
    compare: { type: 'boolean', default: false },
    'compare-duration': { type: 'string', default: '3' },
    help: { type: 'boolean', short: 'h' },
  },
});

if (values.help || positionals.length < 2) {
  console.log(`
Sparse Pixel Retro Processor v2

Usage: node process.mjs <input> <output> [options]

EFFECTS (-e, --effect):
  sparse        Gapped colored pixels with edge preservation (default)
  scanline      Horizontal run-length blocks  
  dither        Ordered Bayer dithering (adjustable strength)
  halftone      Circular/square dots based on luminance
  crt           CRT phosphor RGB subpixels with scanlines
  edge          Edge detection only
  crosshatch    Diagonal line hatching
  sparse-dither Dither then sparse (combined effect)

PALETTES (-p, --palette):
  sparse   Gray, black, teal, coral, white (default)
  gameboy  Classic Game Boy green
  mono     Black and white
  cga      CGA magenta/cyan
  amber    Amber monochrome
  green    Green monochrome  
  pastel   Soft pastels
  vapor    Vaporwave colors
  sepia    Sepia tones
  ink      Paper and ink

OPTIONS:
  --cell-size N        Grid cell size for sparse (default: 10)
  --pixel-size N       Drawn pixel size for sparse (default: 6)
  --dot-size N         Dot size for halftone (default: 8)
  --strength N         Dither strength 0-1 (default: 0.5)
  --matrix-size N      Dither matrix: 2, 4, or 8 (default: 4)
  --fps N              Output framerate (default: 10)
  --max-duration N     Max seconds to process (default: 10)
  --scale N            Scale factor (default: 1)
  --compare            Generate before/after comparison GIF
  --compare-duration N Duration per segment in seconds (default: 3)

EXAMPLES:
  # Sparse pixels (fixed hair/dark areas)
  node process.mjs video.mp4 output.gif -e sparse
  
  # Strong dithering with Game Boy palette
  node process.mjs video.mp4 output.gif -e dither -p gameboy --strength 0.8
  
  # Maximum dither effect (8x8 matrix, high strength)
  node process.mjs video.mp4 output.gif -e dither --matrix-size 8 --strength 1.0
  
  # Halftone newspaper effect
  node process.mjs video.mp4 output.gif -e halftone -p mono --dot-size 6
  
  # CRT phosphor simulation
  node process.mjs video.mp4 output.gif -e crt
  
  # Combined: dither + sparse
  node process.mjs video.mp4 output.gif -e sparse-dither -p vapor

  # Before/after comparison GIF (3s original, 3s transformed)
  node process.mjs video.mp4 compare.gif -e sparse --compare

  # Custom compare duration (5s each segment)
  node process.mjs video.mp4 compare.gif -e dither --compare --compare-duration 5
`);
  process.exit(0);
}

const [input, output] = positionals;
const effect = values.effect;
const options = {
  palette: PALETTES[values.palette] || PALETTES.sparse,
  cellSize: parseInt(values['cell-size']),
  pixelSize: parseInt(values['pixel-size']),
  dotSize: parseInt(values['dot-size']),
  strength: parseFloat(values.strength),
  matrixSize: parseInt(values['matrix-size']),
  fps: parseInt(values.fps),
  maxDuration: parseInt(values['max-duration']),
  scale: parseFloat(values.scale),
  compareDuration: parseInt(values['compare-duration']),
};

const inputExt = extname(input).toLowerCase();
const isVideo = ['.mp4', '.mov', '.avi', '.webm', '.gif'].includes(inputExt);

if (isVideo) {
  if (values.compare) {
    processVideoCompare(input, output, effect, options);
  } else {
    processVideo(input, output, effect, options);
  }
} else {
  processImage(input, output, effect, options);
}
