<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 15.08.2016
 * @time 11:09
 */

namespace SicoCreditPlus\Components;

use Exception;
use Psr\Container\ContainerInterface;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Checkout\Cart\LineItem\LineItemCollection;
use Shopware\Core\Checkout\Order\Aggregate\OrderLineItem\OrderLineItemEntity;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionDefinition;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;
use Shopware\Core\Checkout\Order\OrderDefinition;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Content\Mail\Service\AbstractMailService;
use Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateTranslation\MailTemplateTranslationEntity;
use Shopware\Core\Content\MailTemplate\MailTemplateEntity;
use Shopware\Core\Content\Mail\Service\MailService;

use Shopware\Core\Content\Product\Aggregate\ProductPrice\ProductPriceEntity;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Pricing\Price;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Aggregation\Bucket\TermsAggregation;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Aggregation\Metric\CountAggregation;
use Shopware\Core\Framework\DataAbstractionLayer\Search\AggregationResult\Bucket\TermsResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\AndFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Validation\DataBag\DataBag;
use Shopware\Core\Framework\Validation\Exception\ConstraintViolationException;
use Shopware\Core\System\Currency\CurrencyEntity;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateCollection;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateEntity;
use Shopware\Core\System\StateMachine\StateMachineRegistry;
use Shopware\Core\System\StateMachine\Transition;
use Shopware\Storefront\Framework\Routing\Router;
use SicoCreditPlus\Core\Content\CreditPlusPayment\CreditPlusPaymentEntity;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionCollection;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionEntity;
use SicoCreditPlus\Lib\Controller\CreditPlusWebshopAPI;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopContract;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopCurrency;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopFinanceArticle;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopRateTableMonthRow;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopVoucher;

use Shopware\Core\System\SystemConfig\SystemConfigService;

use Shopware\Core\Content\MailTemplate\Exception\SalesChannelNotFoundException;
use SicoCreditPlus\Service\CreditPlusPayment;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;


//use Symfony\Component\DependencyInjection\Container;

class SicoCreditPlusHelper {

	public const MAIL_SENT = 0;
	public const MAIL_NO_STATUS = 1;
	public const MAIL_NO_TEMPLATE = 2;
	public const MAIL_NO_RECEIVER = 3;
	public const MAIL_DELIVERY_ERROR = 4;

	/**
	 * Local static storage for requested Contract Data
	 *
	 * @var array $_oContractData
	 */
	private static $_oContractData = array();
	/** @var WebshopFinanceArticle $_oFinancingArticleReference */
	private $_oFinancingArticleReference = null;
	/** @var string $_sFinancingMode */
	private $_sFinancingMode = null;
	/** @var CreditPlusWebshopAPI $_oWSApi */
	private static $_oWSApi = null;
	private static $_sSalesChannelForApi = '';

	/** @var array $_aCreditCancelStates All Credit States in which a cancel action is possible */
	private $_aCreditCancelStates = array(
		24,
		20,
		21,
		22,
		23,
		25,
	);

	/** @var array $_aCreditCancelStates All Credit States in which a return action is possible */
	private $_aCreditReturnStates = array(
		24,
		99,
		32
	);


	/** @var StateMachineStateEntity[] $_aPaymentStatesCache All Payment States which have once been requested */
	private static $_aPaymentStatesCache = array();

	/** @var StateMachineStateEntity[] $_aOrderStatesCache */
	private static $_aOrderStatesCache = array();


	/** @var SystemConfigService */
	protected $_oSystemConfigService;

	/**
	 * @var EntityRepository
	 */
	protected $_oSicoOfferedOptions;

	/** @var SalesChannelContext */
	protected $salesChannelContext;

	/** @var Context */
	protected $context = null;

	/** @var SicoCreditPlusLogger */
	protected $oSicoCreditPlusLogger;

	/** @var EntityRepository */
	protected $oStateMachineStateRepo;

	/** @var EntityRepository */
	protected $oOrderTransactionRepo;

	/**
	 * @var StateMachineRegistry
	 */
	private $stateMachineRegistry;

	/** @var MailService  */
	protected $oMailService;

	/** @var EntityRepository  */
	protected $oMailTemplateRepo;
	/** @var EntityRepository */
	protected $oMailTemplateTranslationRepo;

	/** @var EntityRepository  */
	protected $oProductRepo;

	/** @var EntityRepository  */
	protected $oCurrencyRepo;

	/** @var EntityRepository  */
	protected $oProdGroupRepo;

	/** @var Session */
	protected $oSession;

	/** @var EntityRepository */
	protected $oCpPayment;

	/**
	 * @var ContainerInterface
	 */
	protected $container;

	/** @var EntityRepository */
	protected $oPaymentMethodRepo;

	/** @var TranslatorInterface */
	protected $oTranslator;

	/** @var Router */
	protected $oRouter;

	private $bCacheFiles = false;

	/**
	 * @internal
	 * @required
	 */
	public function setContainer(ContainerInterface $container): ?ContainerInterface
	{
		$previous = $this->container;
		$this->container = $container;

		return $previous;
	}

	public function __construct(
		SystemConfigService $systemConfigService,
		EntityRepository $sicoOfferedOptions,
		SicoCreditPlusLogger $sicoCreditPlusLogger,
		EntityRepository $oStateMachineStateRepo,
		EntityRepository $oOrderTransactionRepo,
		StateMachineRegistry $stateMachineRegistry,
		AbstractMailService $mailService,
		EntityRepository $oMailTemplateRepo,
		EntityRepository $oMailTemplateTranslationRepo,
		EntityRepository $oProductRepo,
		EntityRepository $oCurrencyRepo,
		EntityRepository $oProdGroupRepo,
		EntityRepository $oCpPayment,
		EntityRepository $oPaymentMethodRepo,
		TranslatorInterface $oTranslator,
		Router $oRouter)
	{
		$this->_oSystemConfigService = $systemConfigService;
		$this->_oSicoOfferedOptions = $sicoOfferedOptions;
		$this->oSicoCreditPlusLogger = $sicoCreditPlusLogger;
		$this->oStateMachineStateRepo = $oStateMachineStateRepo;
		$this->oOrderTransactionRepo = $oOrderTransactionRepo;
		$this->stateMachineRegistry = $stateMachineRegistry;
		$this->oMailService = $mailService;
		$this->oMailTemplateRepo = $oMailTemplateRepo;
		$this->oMailTemplateTranslationRepo = $oMailTemplateTranslationRepo;
		$this->oProductRepo = $oProductRepo;
		$this->oCurrencyRepo = $oCurrencyRepo;
		$this->oProdGroupRepo = $oProdGroupRepo;
		$this->oCpPayment = $oCpPayment;
		$this->oPaymentMethodRepo = $oPaymentMethodRepo;
		$this->oTranslator = $oTranslator;
		$this->oRouter = $oRouter;
	}

	public function setSalesChannelContext(SalesChannelContext $salesChannelContext){
		$this->salesChannelContext = $salesChannelContext;
	}
	public function setContext(Context $context){
		$this->context = $context;
	}

	public function setCacheFiles(bool $bCacheFiles): void
	{
		$this->bCacheFiles = $bCacheFiles;
	}

	public function resetWebshopAPI(): void
	{
		self::$_sSalesChannelForApi = '';
	}

	/**
	 * @return CreditPlusWebshopAPI
	 */
	public function getWebshopAPI(string $salesChannelId) {
		if ( self::$_oWSApi !== null && ($salesChannelId === self::$_sSalesChannelForApi) ) {
			return self::$_oWSApi;
		}

		$soapUser = $this->_oSystemConfigService->get('SicoCreditPlus.config.sSoapUser',$salesChannelId);
		$soapPass = $this->_oSystemConfigService->get('SicoCreditPlus.config.sSoapPass',$salesChannelId);
		$soapType = $this->_oSystemConfigService->get('SicoCreditPlus.config.sSoapType',$salesChannelId);
		$wsdl = $this->_oSystemConfigService->get('SicoCreditPlus.config.sWSDL',$salesChannelId);
		$bLogActive = $this->_oSystemConfigService->get('SicoCreditPlus.config.bLogActive',$salesChannelId);
		$aSoapParams = array(
			'soap-user' => $soapUser,
			'soap-pass' => $soapPass,
			'soap-type' => $soapType,
			'wsdl' => $wsdl,
			'cache' => $this->bCacheFiles,
		);
		$oWSApi = new CreditPlusWebshopAPI($aSoapParams);
		$oWSApi->setDebugState(true, false);

		if ( $bLogActive ) {
			$oWSApi->setLogger($this->oSicoCreditPlusLogger);
		}

		self::$_sSalesChannelForApi = $salesChannelId;
		return self::$_oWSApi = $oWSApi;
	}
	/**
	 * Takes general info about the currency available in the shop and returns an object as required for the API
	 * @param CurrencyEntity $oCurrency
	 * @return WebshopCurrency
	 */
	public function getWebshopCurrency($oCurrency) {
		//$oCurrency = Shopware()->Shop()->getCurrency();
		if($oCurrency == null){
			$oCurrency = $this->salesChannelContext->getCurrency();
		}
		//
		$aWSCurrency = array(
			'decimals' => 2,
			'decimalSeparator' => ',',
			'thousandSeparator' => '.',
			'side' => (($oCurrency->getPosition() == 0)?'left':'right'),//(($oCurrency->getSymbolPosition()==0)?'left':'right'),
			'sign' => $oCurrency->getSymbol(),
			'name' => $oCurrency->getName()
		);
		return WebshopCurrency::fromArray($aWSCurrency);
	}

	/**
	 * @param CurrencyEntity $oCurrency
	 * @param ProductEntity $oArticleDetails
	 * @param float $dMinRate
	 * @param float $dOrderPrice
	 *
	 * @return WebshopRateTableMonthRow[]
	 */
	public function getFinancingMonths( SalesChannelContext $context, $oCurrency, $oArticleDetails = null, $dMinRate = null, $dOrderPrice = null ) {
		/** @var WebshopRateTableMonthRow[] $aReturn */
		$aReturn = array();
		if ( $oArticleDetails == null ) { return $aReturn; }
		if ( $this->isPaymentExcludedByMissingApi($context) ) {
			$this->oSicoCreditPlusLogger->log(true, 'API is not reachable: Returning empty rate table.', ['function' => __METHOD__, 'line' => __LINE__, 'articleNumber' => $oArticleDetails->getProductNumber(), 'salesChannel' => $context->getSalesChannel()->getName()], $context->getSalesChannelId());
			return $aReturn;
		}
		$oWSCurrency = $this->getWebshopCurrency($oCurrency);

		if ( $dOrderPrice === null ) {
			$aPrices = $oArticleDetails->getPrices();
			if ( $aPrices ) {
				foreach ( $aPrices as $oPrice ) {
					if ( $oPrice->getQuantityStart() <= 1 ) {
						if ( !$oPrice->getQuantityEnd() || ($oPrice->getQuantityEnd() >= 1) ) {
							$dOrderPrice = $oPrice->getPrice()->get($oCurrency->getId())->getGross();
							//$dOrderPrice = $dOrderPrice * (1+$oArticleDetails->getTax()->getTaxRate()/100);
						}
					}
				}
			}
			if ( $dOrderPrice === null ) {
				$aPrices = $oArticleDetails->getPrice();
				foreach ( $aPrices as $oPrice ) {
					$dOrderPrice = $oPrice->getGross();
				}
			}
		}
		$dMinBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',$context->getSalesChannelId()));
		$dMaxBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMaxBasketPrice',$context->getSalesChannelId()));

		//Special case: If min rate is 0.01, we are checking for fictional rates, skip the other absolute rules
		$bSkipSanityCheck = $dMinRate === 0.01;
		if ( $dMinRate === null ) {
			//$dMinRate = floatval($oConfig->getByNamespace('SicoCreditPlus','sMinRate'));
			$dMinRate = $this->_oSystemConfigService->get('SicoCreditPlus.config.sMinRate',$context->getSalesChannelId());
		}
		if ( !$bSkipSanityCheck ) {
			// Product or basket is cheaper than min price => no financing
			if ( $dMinBasketPrice > $dOrderPrice ) {
				return $aReturn;
			}
			// Product or basket is more expensive than max price => no financing
			if ( $dMaxBasketPrice < $dOrderPrice ) {
				return $aReturn;
			}
		}


		try{
			$oWSApi = $this->getWebshopAPI($context->getSalesChannelId());
			$aReturn = $this->getProductSpecificFinancingMonths($oArticleDetails, $dMinRate, $dOrderPrice, $aReturn, $oWSApi);
			$aReturn = $this->getUnspecificFinancingMonths($dMinRate, $dOrderPrice, $oWSApi, $aReturn);
			$aReturn = $this->getCalculatedFinancingMonths($aReturn, $dOrderPrice, $oWSCurrency);
		}catch( Exception $e){
			$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),['FILE' => $e->getFile(),'LINE' => $e->getLine(),'Trace' => $e->getTraceAsString()]);
		}
		return $aReturn;
	}


	/**
	 * @param ProductEntity $oArticleDetails
	 * @param float $dMinRate
	 * @param float $dOrderPrice
	 * @param WebshopRateTableMonthRow[] $aReturn
	 * @param CreditPlusWebshopAPI $oWSApi
	 * @return WebshopRateTableMonthRow[] $aReturn + cheaper or new options
	 * @throws Exception
	 */
	private function getProductSpecificFinancingMonths( $oArticleDetails, $dMinRate, $dOrderPrice, $aReturn, $oWSApi) {
		$criteria = new Criteria();
		$criteria = $criteria->addFilter(
			new EqualsFilter('sicoActive',1),
			new EqualsFilter('sicoProdgroup.product.id',$oArticleDetails->getId()),
			new EqualsFilter('sicoProdgroup.prodgroupSalesChannel.salesChannelId',$this->salesChannelContext->getSalesChannelId())
		);

		$criteria->addAssociations([
			'sicoProdgroup',
		]);
		/** @var SicoOfferedOptionCollection $oRes */
		$oRes = $this->_oSicoOfferedOptions->search($criteria,$this->salesChannelContext->getContext())->getEntities();

		return $this->getOptionsFromResultSet($oRes, $dMinRate, $dOrderPrice, $aReturn, $oWSApi, true);
	}

	/**
	 * @param float $dMinRate
	 * @param float $dOrderPrice
	 * @param WebshopRateTableMonthRow[] $aReturn
	 * @param CreditPlusWebshopAPI $oWSApi
	 * @return WebshopRateTableMonthRow[] $aReturn + cheaper or new options
	 * @throws Exception
	 */
	private function getUnspecificFinancingMonths( $dMinRate, $dOrderPrice, $oWSApi, $aReturn = array()) {
		try {

			$oProdGroupCriteria = new Criteria();
			$oProdGroupCriteria->addFilter(
				new EqualsFilter('prodgroupSalesChannel.salesChannelId', $this->salesChannelContext->getSalesChannelId())
			);
			$oProdGroupCriteria->addAggregation(
				new TermsAggregation(
					'per-group',
					'id',
					null,
					null,
					new CountAggregation('productCount', 'product.id')
				)
			);
			$oProdGroupsResult = $this->oProdGroupRepo->search($oProdGroupCriteria, $this->salesChannelContext->getContext());
			/** @var TermsResult $oGroupAggregations */
			$oGroupAggregations = $oProdGroupsResult->getAggregations()->get('per-group');
			$aGoodGroups = [];
			foreach ( $oGroupAggregations->getBuckets() as $oBucket ) {
				// $oBucket->getCount() returns 1 if $oBucket->getResult()->getCount() returns 0;
				// $oBucket->getCount() returns whatever $oBucket->getResult()->getCount() returns, if greater than 0;
				// Logic is a bit flawed, but results can at least be found.
				$iCount = $oBucket->getResult()->getCount();
				if ( $iCount === 0 ) {
					$aGoodGroups[] = $oBucket->getKey();
				}
			}
			if ( count($aGoodGroups) === 0 ) {
				// No unspecific product groups available
				return $aReturn;
			}

			$oCriteria = new Criteria();
			$oCriteria = $oCriteria->addFilter(
				new EqualsFilter('sicoActive',1),
				new EqualsAnyFilter('sicoProdgroup.id', $aGoodGroups)
			);
			$oCriteria->addAssociations([
				'sicoProdgroup',
			]);

			/** @var SicoOfferedOptionCollection $oRes */
			$oRes = $this->_oSicoOfferedOptions->search($oCriteria,$this->salesChannelContext->getContext())->getEntities();

			return $this->getOptionsFromResultSet($oRes, $dMinRate, $dOrderPrice, $aReturn, $oWSApi);
		} catch (Exception $e) {
			throw $e;
		}

	}

	/**
	 * @param SicoOfferedOptionCollection $oRes
	 * @param float $dMinRate
	 * @param float $dOrderPrice
	 * @param WebshopRateTableMonthRow[] $aReturn
	 * @param CreditPlusWebshopAPI $oWSApi
	 * @param bool $bIsProductSpecific
	 * @return WebshopRateTableMonthRow[] $aReturn + cheaper or new options
	 */

	private function getOptionsFromResultSet( $oRes, $dMinRate, $dOrderPrice, $aReturn, $oWSApi, $bIsProductSpecific = false ) {

		if(count($oRes)){
			/** @var SicoOfferedOptionEntity $oOption */
			foreach($oRes as $oOption){
				$aOption = array(
					'months' => $oOption->getSicoMonths(),
					'interestRate' => $oOption->getSicoInterest(),
					'nominalInterestRate' => $oWSApi->getNominalInterestFromEffectiveInterest($oOption->getSicoInterest()),
					'productCode' => $oOption->getSicoProdcode(),
					'productTypeID' => $oOption->getSicoProdgroup()->getSicoProductTypeId(),
					'productClassID' => $oOption->getSicoProdgroup()->getSicoProductClassId(),
					'productGroupName' => $oOption->getSicoProdgroup()->getSicoName(),
					'isProductSpecific' => $bIsProductSpecific
				);
				$dRateFactor = floatval($oOption->getSicoRatefactor());
				if ( $dRateFactor === -1.0 ) {
					$dRateFactor = false;
				}
				$aOption['monthlyRate'] = $oWSApi->getMonthRateByPriceMonthsAndInterest($dOrderPrice,$aOption['months'],$aOption['interestRate'],$dRateFactor);
				if ( $aOption['monthlyRate'] < $dMinRate ) {
					// Skip rates, which are too low
					continue;
				}
				if ( !isset($aReturn[$aOption['months']]) ) {
					// If not yet set, just add
					$aReturn[$aOption['months']] = new WebshopRateTableMonthRow($aOption);
				} elseif ( $aReturn[$aOption['months']]->interestRate > $aOption['interestRate'] ) {
					// If general is cheaper, replace
					$aReturn[$aOption['months']] = new WebshopRateTableMonthRow($aOption);
				}
			}
		}
		return $aReturn;
	}

	/**
	 * @param WebshopRateTableMonthRow[] $aReturn
	 * @param float $dOrderPrice
	 * @param WebshopCurrency $oCurrency
	 *
	 * @return WebshopRateTableMonthRow[]
	 */
	public function getCalculatedFinancingMonths( $aReturn, $dOrderPrice, $oCurrency ) {
		// Prepare values for output (Currency sign, number format)
		foreach ( $aReturn as &$oWSOption ) {
			$oWSOption->totalAmount = $oWSOption->monthlyRate*$oWSOption->months;
			// If interest is 0.00 or above
			if ( $oWSOption->interestRate >= 0.00 ) {
				// If total price is lower than base price
				// or if price is higher than base price with 0.00 interest,
				// something must be off on the single payments
				// which will be corrected on the final payment anyway
				if ( ($oWSOption->totalAmount < $dOrderPrice) || (($oWSOption->totalAmount > $dOrderPrice) && ($oWSOption->interestRate == 0.00)) ) {
					$oWSOption->totalAmount = $dOrderPrice;
				}
			}
			$oWSOption->interest = $oWSOption->totalAmount-$dOrderPrice;
			$oWSOption->interestRate = number_format($oWSOption->interestRate, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator).'%';
			$oWSOption->nominalInterestRate = number_format($oWSOption->nominalInterestRate, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator).'%';
			if ( $oCurrency->side && ($oCurrency->side == 'left') ) {
				$oWSOption->interest = $oCurrency->sign.' '.number_format($oWSOption->interest, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator);
				$oWSOption->totalAmount = $oCurrency->sign.' '.number_format($oWSOption->totalAmount, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator);
				$oWSOption->monthlyRate = $oCurrency->sign.' '.number_format($oWSOption->monthlyRate, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator);
			} else {
				$oWSOption->interest = number_format($oWSOption->interest, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator).' '.$oCurrency->sign;
				$oWSOption->totalAmount = number_format($oWSOption->totalAmount, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator).' '.$oCurrency->sign;
				$oWSOption->monthlyRate = number_format($oWSOption->monthlyRate, $oCurrency->decimals, $oCurrency->decimalSeparator, $oCurrency->thousandSeparator).' '.$oCurrency->sign;
			}
		}
		// Sorty by months ascending
		ksort($aReturn);
		return $aReturn;
	}

	/**
	 * @param SalesChannelContext $context Context used for determining shop which is starting the financing
	 * @param OrderEntity|Cart $oOrder Order or Cart object for the total sums
	 * @param LineItemCollection|null $aArticles LineItems to be analyzed
	 * @param float|null $dBasketPrice If predetermined value should be used for calculation
	 * @return WebshopFinanceArticle
	 * @throws \Exception
	 */
	public function getFinancingArticleReference(SalesChannelContext $context, $oOrder = null, ?LineItemCollection $aArticles = null, ?float $dBasketPrice = null) {
		if ( !$oOrder && !$aArticles ) {
			return null;
		}

		$dMinRate = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMinRate',$context->getSalesChannelId()));
		$sBasketStrategy = $this->_oSystemConfigService->get('SicoCreditPlus.config.sBasketFinancingMode',$context->getSalesChannelId());

		if ( ($this->_sFinancingMode === $sBasketStrategy) && ($this->_oFinancingArticleReference !== null) ) {
			return $this->_oFinancingArticleReference;
		}
		if($oOrder != null){
			if($oOrder instanceof OrderEntity){
				$oCurrency = $oOrder->getCurrency();
			} else {
				$oCurrency = $oOrder->getData()->get('currency');
			}

		}else {
			$oCurrency = $this->salesChannelContext->getCurrency();
		}
		$oWSCurrency = $this->getWebshopCurrency($oCurrency);
		$oWSApi = $this->getWebshopAPI($context->getSalesChannelId());
		$aWebshopArticles = array();

		$oCollection = $aArticles;
		if($oCollection === null){
			$oCollection = $oOrder->getLineItems();
		}

		if ( $dBasketPrice !== null ) {
			$dOrderPrice = $dBasketPrice;
		} else {
			if ( $oOrder ) {
				if ( $oOrder instanceof OrderEntity ) {
					$dOrderPrice = $oOrder->getAmountTotal();
				} else {
					$dOrderPrice = $oOrder->getPrice()->getTotalPrice();
				}
			} else {
				$dOrderPrice = $aArticles->getPrices()->sum()->getTotalPrice();
			}
		}

		/** @var OrderLineItemEntity|LineItem $lineItem */
		foreach($oCollection as $lineItem){
			if ( $lineItem->getType() !== LineItem::PRODUCT_LINE_ITEM_TYPE ) {
				continue;
			}
			if($lineItem instanceof OrderLineItemEntity){
				$dPrice = $lineItem->getTotalPrice();
				$dAmount = $lineItem->getQuantity();
			}
			else{
				$dPrice = $lineItem->getPrice()->getTotalPrice();
				$dAmount = $lineItem->getQuantity();
			}

			if($oOrder){
				if ( $oOrder instanceof OrderEntity ) {
					$iKey = $lineItem->getProduct()->getId();
					$oProduct = $lineItem->getProduct();
				} else {
					$iKey = $lineItem->getReferencedId();
					if($iKey){
						$oProduct = $this->oProductRepo->search(new Criteria([$iKey]),$context->getContext())->get($iKey);
					} else {//should never happen
						throw new \Exception('Line Item is not a product');
					}
				}
			} else {
				$iKey = $lineItem->getReferencedId();
				$oProduct = $this->oProductRepo->search(new Criteria([$iKey]),$context->getContext())->get($iKey);
			}
			$aWebshopArticle = array(
				'id' => $iKey,
				'amount' => $dAmount,
				'unitprice' => $dPrice,
				'isProductSpecific' => false
			);
			$aMonths = $this->getFinancingMonths($context, $oCurrency, $oProduct, $dMinRate, $dOrderPrice);

			if ( !$aMonths ) {
				// One of the articles has no financing options, so this will not be offered for the order
				// Return null and handle it in the calling function
				return null;
			}
			$aWebshopArticle['financingMonths'] = $aMonths;
			foreach ( $aMonths as $oMonthRow ) {
				$aWebshopArticle['isProductSpecific'] = $aWebshopArticle['isProductSpecific'] || $oMonthRow->isProductSpecific;
				$dInterestRate = $oWSApi->retrieveFloatFromFormattedInterestRate($oMonthRow->interestRate, $oWSCurrency);
				if ( !isset($aWebshopArticle['cheapestInterestRate']) || ($dInterestRate < $aWebshopArticle['cheapestInterestRate']) ) {
					$aWebshopArticle['cheapestInterestRate'] = $dInterestRate;
					$aWebshopArticle['productTypeID'] = $oMonthRow->productTypeID;
					$aWebshopArticle['productClassID'] = $oMonthRow->productClassID;
				}
				if ( !isset($aWebshopArticle['mostExpensiveInterestRate']) || ($dInterestRate > $aWebshopArticle['mostExpensiveInterestRate']) ) {
					$aWebshopArticle['mostExpensiveInterestRate'] = $dInterestRate;
				}
				if ( $oMonthRow->isProductSpecific && ( $aWebshopArticle['cheapestInterestRate'] == $dInterestRate ) ) {
					$aWebshopArticle['productTypeID'] = $oMonthRow->productTypeID;
					$aWebshopArticle['productClassID'] = $oMonthRow->productClassID;
				}
				$aWebshopArticle['aMonthRows'][(int)$oMonthRow->months] = $dInterestRate;
			}
			$oWebshopArticle = new WebshopFinanceArticle($aWebshopArticle);
			$aWebshopArticles[$iKey] = $oWebshopArticle;
		}

		$oWebshopArticle = $oWSApi->getFinancingArticleReference($aWebshopArticles, $sBasketStrategy);
		$this->_oFinancingArticleReference = $oWebshopArticle;
		return $this->_oFinancingArticleReference;
	}

	/**
	 * @param WebshopRateTableMonthRow[] $aMonths
	 * @return float|null
	 */

	public function getCheapestInterestRate( $aMonths ): ?float {
		$dCheapestInterestRate = null;
		foreach ( $aMonths as $oMonth ) {
			$dEffInterest = floatval(str_replace(array(',','%'),array('.',''),$oMonth->interestRate));
			if ( ( $dCheapestInterestRate === null ) || ($dEffInterest < $dCheapestInterestRate) ) {
				$dCheapestInterestRate = $dEffInterest;
			}
		}
		return $dCheapestInterestRate;
	}


	/**
	 * @param WebshopRateTableMonthRow[] $aMonths
	 * @return WebshopRateTableMonthRow
	 */

	public function getCheapestRate( $aMonths ): ?WebshopRateTableMonthRow {
		/** @var WebshopRateTableMonthRow $aCheapestRate */
		$aCheapestRate = null;
		foreach ( $aMonths as $oMonth ) {
			$dRate = $this->getNumberFromFromattedValue($oMonth->monthlyRate);
			if ( ( $aCheapestRate === null ) || ($dRate < $this->getNumberFromFromattedValue($aCheapestRate->monthlyRate)) ) {
				$aCheapestRate = $oMonth;
			}
		}
		return $aCheapestRate;
	}

	/**
	 * Filter out non-numeric values like " " or €
	 * Converting € 40.412,41 to floatval(40412.41)
	 * @param string $sRate
	 * @return float
	 */
	public function getNumberFromFromattedValue( $sRate ) {
		$sMonthRate = '';
		// Filter out non-numeric values like " " or €
		// € 40.412,41 is converted to 40.412,41
		for ( $iPos = 0 ; $iPos < strlen($sRate) ; $iPos++ ) {
			$sSubString = substr($sRate,$iPos,1);
			if ( preg_match('/[0-9,.]/',$sSubString) ) {
				$sMonthRate .= $sSubString;
			}
		}
		$aRate = explode(',', $sMonthRate);

		// 40.412,41 is converted to 40412.41
		if ( (count($aRate) == 2) && (strlen($aRate[count($aRate)-1]) <= 2) ) {
			$aBigNumber = explode('.',$aRate[0]);
			$sMonthRate = implode('',$aBigNumber).'.'.$aRate[1];
			// 40,412.41 is converted to 40412.41
		} else {
			$aRate = explode('.', $sRate);
			$sMonthRate = implode('',explode(',',$aRate[0])).'.'.$aRate[1];
		}
		// And now we have a number...
		$dRate = floatval($sMonthRate);
		return $dRate;
	}

	/**
	 * Requests payment data from CreditPlus and returns $oOrder with dynamically declared cpContractData
	 *
	 * @param string $sSaleChannelId
	 * @param OrderEntity|OrderTransactionEntity|Cart $oOrder Order object
	 * @return WebshopContract|null Data saved at CreditPlus
	 */
	public function getContractData(string $sSaleChannelId, $oOrder ): ?WebshopContract {
		$sDealerOrderNumber = '';
		$sOrderNumber = '';
		$sPaymentName = '';
		$sTransactionNumber = null;
		$sSalesChannelId = $sSaleChannelId;
		if($oOrder instanceof OrderEntity){
			$sDealerOrderNumber = $oOrder->getTransactions()->first()->getId();
			$sOrderNumber = $oOrder->getOrderNumber();
			$oPaymentMethod = $oOrder->getTransactions()->first()->getPaymentMethod();
			if($oPaymentMethod != null){
				$sPaymentName = $oOrder->getTransactions()->first()->getPaymentMethod()->getHandlerIdentifier();
			} else {
				/** @var OrderTransactionEntity $oTransaction */
				$oTransaction = $this->oOrderTransactionRepo->search((new Criteria([$sDealerOrderNumber]))->addAssociation('paymentMethod'),$this->context)->first();
				$sPaymentName = $oTransaction->getPaymentMethod()->getHandlerIdentifier();
			}

			$aOrderTransactionFields = $oOrder->getCustomFields();
			if(isset($aOrderTransactionFields['cpDealerOrderNumber'])){
				$sTransactionNumber = $aOrderTransactionFields['cpDealerOrderNumber'];
			}

			if($this->context && !$sTransactionNumber){
				$oCpPaymentCollection = $this->oCpPayment->search((new Criteria())->addFilter(new EqualsFilter('orderId',$oOrder->getId())),$this->context);
				/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
				$oCpPaymentEntity = $oCpPaymentCollection->first();
				if($oCpPaymentEntity){
					$sTransactionNumber = $oCpPaymentEntity->getOrderNumber();
				} else {
					$sTransactionNumber = $oOrder->getOrderNumber();
				}
			}
			$sSalesChannelId = $oOrder->getSalesChannelId();
		} else if($oOrder instanceof OrderTransactionEntity){
			$sDealerOrderNumber = $oOrder->getId();
			$oOrderEntity = $oOrder->getOrder();
			$sOrderNumber = $oOrderEntity->getOrderNumber();
			$sTransactionNumber = $oOrderEntity->getCustomFieldsValue('cpDealerOrderNumber');
			$sPaymentName = $oOrder->getPaymentMethod()->getHandlerIdentifier();
			$sSalesChannelId = $oOrderEntity->getSalesChannelId();
		} else if($oOrder instanceof Cart) {
			$sDealerOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber');
			$sOrderNumber = $this->oSession->get('cpDealerOrderNumber');
			$sTransactionNumber = $oOrder->getData()->get('cpDealerOrderNumber');
			$sPaymentName = CreditPlusPayment::class;
		}

		$this->oSicoCreditPlusLogger->log(false,'getContract',['dealerOrderNumber' => $sDealerOrderNumber,'orderNumber' => $sOrderNumber, 'contractData' => self::$_oContractData]);
		if ( isset(self::$_oContractData[$sDealerOrderNumber]) && self::$_oContractData[$sDealerOrderNumber] !== null ) {
			self::$_oContractData[$sDealerOrderNumber] = $this->getState(self::$_oContractData[$sDealerOrderNumber], $sSalesChannelId);
			return self::$_oContractData[$sDealerOrderNumber];
		}
		if ( isset(self::$_oContractData[$sOrderNumber]) && self::$_oContractData[$sOrderNumber] !== null ) {
			self::$_oContractData[$sOrderNumber] = $this->getState(self::$_oContractData[$sOrderNumber], $sSalesChannelId);
			return self::$_oContractData[$sOrderNumber];
		}
		if($sTransactionNumber != null){
			if ( isset(self::$_oContractData[$sTransactionNumber]) && self::$_oContractData[$sTransactionNumber] !== null ) {
				self::$_oContractData[$sTransactionNumber] = $this->getState(self::$_oContractData[$sTransactionNumber], $sSalesChannelId);
				return self::$_oContractData[$sTransactionNumber];
			}
		}

		if( ($sPaymentName !== CreditPlusPayment::class) || ($sDealerOrderNumber === '')){
			return self::$_oContractData[$sDealerOrderNumber] = new WebshopContract();
		}

		// Take the multi getter and use only the expected result
		$aWebshopContracts = $this->getContractDataMultiple($sSalesChannelId,array($sDealerOrderNumber, $sOrderNumber,$sTransactionNumber));

		if(isset($aWebshopContracts[$sDealerOrderNumber])){
			$aWebshopContracts[$sDealerOrderNumber] = $this->getState($aWebshopContracts[$sDealerOrderNumber], $sSalesChannelId);
			return $aWebshopContracts[$sDealerOrderNumber];
		}
		else if(isset($aWebshopContracts[$sOrderNumber])){
			$aWebshopContracts[$sOrderNumber] = $this->getState($aWebshopContracts[$sOrderNumber], $sSalesChannelId);
			return $aWebshopContracts[$sOrderNumber];
		} else if(isset($aWebshopContracts[$sTransactionNumber])){
			$aWebshopContracts[$sTransactionNumber] = $this->getState($aWebshopContracts[$sTransactionNumber], $sSalesChannelId);
			return $aWebshopContracts[$sTransactionNumber];
		}
		else{
			return null;
		}
	}

	/**
	 * Method to set specific test status on specific orders if testmode is active.
	 *
	 * @param WebshopContract $oContract
	 * @return WebshopContract
	 */
	private function getState($oContract, $sSalesChannelId = null){

		$bTest = $this->_oSystemConfigService->get('SicoCreditPlus.config.bTestMode', $sSalesChannelId);
		if ( $bTest ) {
			$sSimulatedOrderNumber = $this->_oSystemConfigService->get('SicoCreditPlus.config.sTestOrderNumber', $sSalesChannelId);
			$sReturnValue = $this->_oSystemConfigService->get('SicoCreditPlus.config.sReturnValue', $sSalesChannelId);
			if ( ($oContract->dealerOrderNumber === $sSimulatedOrderNumber) && !empty($sReturnValue) ) {
				$iReturnValue = 10;
				$sRemainingText = '';
				switch ( $sReturnValue ) {
					case '10': // Antrag liegt vor
					case '20': // Ausgefüllt
					case '24n': // Genehmigt
					case '24a': // Lieferfreigabe
					case '24d': // Ausgeliefert
					case '32d': // Auszahlung
					case '92': // Abgelehnt weich
					case '93': // Abgelehnt hart
					case '94': // Fehler
					case '95': // Storniert
					case '99d': // Ausbezahlt
						$iReturnValue = (int)$sReturnValue;
						$sRemainingText = str_replace($iReturnValue.'', '', $sReturnValue);
				}
				$oContract->state = $iReturnValue;
				// a = approved
				if ( $sRemainingText === 'a' ) {
					$oContract->finallyApproved = true;
				}
				// d = dispatched
				if ( $sRemainingText === 'd' ) {
					$oContract->finallyApproved = true;
					$oContract->deliveryDone = true;
				}
			}
		}
		return $oContract;
	}

	/**
	 * Requests multiple WebshopContracts by sending the transaction IDs through the API
	 * Utilizes local caching per item if same item is requested more than once.
	 *
	 * @param string[] $aTransactionIDs
	 * @return WebshopContract[] Array of WebshopContracts with TransactionID as its index
	 */
	public function getContractDataMultiple(string $sSalesChannelId, $aTransactionIDs ) {
		$aReturn = array();
		if ( !$aTransactionIDs ) {
			return array();
		}
		$aKeys = array();
		// Don't refetch already fetched data
		// Empty TransactionID == empty WebshopContract
		foreach ( $aTransactionIDs as $iKey => $sTransactionID ) {
			if ( $sTransactionID === '' ) {
				self::$_oContractData[$sTransactionID] = new WebshopContract();
				$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				unset($aTransactionIDs[$iKey]);
			} elseif ( $sTransactionID === null ) {
				// Rare case, where one provided ID is null just removes it from the request
				unset($aTransactionIDs[$iKey]);
			} elseif ( isset(self::$_oContractData[$sTransactionID]) && self::$_oContractData[$sTransactionID] ) {
				$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				unset($aTransactionIDs[$iKey]);
			} else {
				$aKeys[$sTransactionID] = $iKey;
			}
		}
		// If requests needs to be made, do it
		if ( $aTransactionIDs ) {
			//$oConfig = Shopware()->Config();
			$sCPDealer = $this->_oSystemConfigService->get('SicoCreditPlus.config.sCPDealer',$sSalesChannelId);
			$oWSApi = $this->getWebshopAPI($sSalesChannelId);
			$oCreditOfferResponse = null;
			$return = [];
			if ( $oWSApi->isApiCallable() ) {
				$oCreditOfferResponse = $oWSApi->getContractsCPWebshop( array('dealerOrderNumber' => $aTransactionIDs, 'dealerNumber' => $sCPDealer ) );
			} else {
				$this->oSicoCreditPlusLogger->log(true,'Api is not call able',[]);
			}
			if(!$oCreditOfferResponse){
				$return = array( 0 => false );
			}
			else {
				if(is_object($oCreditOfferResponse)){
					$oReflectionObj = new \ReflectionObject($oCreditOfferResponse);
					if($oReflectionObj->hasProperty('return')){
						$return = $oCreditOfferResponse->return;
					}
				}

				if ( !is_object($oCreditOfferResponse) ) {
					$return = array( 0 => false );
				}
			}


			if (!is_array($return) ) {
				$return = array($return);
			}
			// Cache contract data by TransactionID
			foreach ( $return as $oCreditOfferResponseSingle ) {
				if(!is_object($oCreditOfferResponseSingle)){
					continue;
				}

				$sTransactionID = $oCreditOfferResponseSingle->dealerOrderNumber.'';
				self::$_oContractData[$sTransactionID] = WebshopContract::fromObject($oCreditOfferResponseSingle);
				// This is only provided in state 24
				if ( (int)self::$_oContractData[$sTransactionID]->state === 24 ) {
					$oCompleteUrl = $oWSApi->completeOrderUrlCPWebshop($sTransactionID);
					if ( is_object($oCompleteUrl) && $oCompleteUrl && property_exists($oCompleteUrl,'completeOrderUrl') && $oCompleteUrl->completeOrderUrl ) {
						self::$_oContractData[$sTransactionID]->completeOrderUrl = $oCompleteUrl->completeOrderUrl.'';
					}
				}
				$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				unset($aTransactionIDs[$aKeys[$sTransactionID]]);
			}
		}
		// Return WebshopContract[] (using TransactionID as index)
		return $aReturn;
	}


	/**
	 * @param OrderEntity $oOrder The order being cancelled
	 * @param string $sUserName The User cancelling the order (E-Mail Address)
	 * @return array Array with errors or null if none
	 */

	public function cancelOrder( $oOrder, $sUserName = 'User requested' ) {

		$oOrderTransaction = $oOrder->getTransactions()->last();
		$payment = $oOrderTransaction->getPaymentMethod()->getHandlerIdentifier();
		// Not my payment? Not my problem! :)
		if ( $payment != 'SicoCreditPlus\Service\CreditPlusPayment' ) {
			return null;
		}
		$aError = array();
		//$oLang = Shopware()->Snippets()->getNamespace('frontend/plugins/payment/creditplus');
		// Only orders, which are not yet cancelled are eligible for API Cancel.
		// So we are going through the uncancelled items and refund them as one call.
		/*
		$iOrderStateCancelled = 4;
		if ( defined('Shopware\Models\Order\Status::ORDER_STATE_CANCELLED_REJECTED') ) {
			$iOrderStateCancelled = Status::ORDER_STATE_CANCELLED_REJECTED;
		}
		*/

		if ( $oOrderTransaction->getStateId() !== md5('creditplus_cancelled') ) {
			//$oConfig = Shopware()->Config();
			$oWSApi = $this->getWebshopAPI($oOrder->getSalesChannelId());

			$oContractData = $this->getContractData($oOrder->getSalesChannelId(),$oOrder);
			if(!is_object($oContractData)){
				$aError = array(
					'sError' => (400),
					'sErrorMessage' => 'credit-plus-error.not-found-remote-cancelled',
					'sErrorMessagePlain' => 'Die Bestellung konnte nicht bei CreditPlus gefunden werden.<br />Die Bestellung im Shop wurde jedoch auf den Status storniert gesetzt.'//$oLang->get('ErrorCancelAlreadyCancelled')
				);
				$this->setOrderStatus($oOrder,'cancel');
				$this->setPaymentStatus($oOrder, 'creditplus_cancelled');
				$this->resetInstockProducts($oOrder);
				return $aError;
			}
			$sDealerOrderNumber = $oContractData->dealerOrderNumber;
			// If delivery is not yet done, and the order is in state "pending", cancel is possible
			if ( !$oContractData->deliveryDone && (in_array($oContractData->state,$this->_aCreditCancelStates)) ) {

				$aCancelData = array(
					'dealerOrderNumber' => $sDealerOrderNumber,
					'cancelationFrom' => $sUserName,
					'dealerNumber' => $this->_oSystemConfigService->get('SicoCreditPlus.config.sCPDealer',$oOrder->getSalesChannelId())//$oConfig->getByNamespace('SicoCreditPlus', 'sCPDealer')
				);

				$oResponse = $oWSApi->cancelOrderCPWebshop($aCancelData);
				$this->resetInstockProducts($oOrder);

				if ( is_object($oResponse) && ( property_exists($oResponse,'confirmation') )  && ( property_exists($oResponse->confirmation,'confirmationItems') )  && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
					$iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
					$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
					$aError = array(
						'sError' => (500+$iErrorCode),
						'sErrorMessage' => 'credit-plus-error.cancel-code-'.$iErrorCode,
						'sErrorMessagePlain' => $sErrorMessage
					);
					// 501 == Kein Auftrag gefunden
					// 502 == Pflichtfelder fehlen, kann durch Code heraus nicht passieren
					// 506 == Storno im Status "Bezahlt" kann nicht stattfinden
					// 514 == Irgendein Pflichtfeld sprengt sein Format/Länge
					// dealerOrderNumber > 40
					// cancelationFrom > 27
					// 515 == Auftragsstatus = Unbekannt
					// 516 == Auftragsstatus = Abgelehnt
					// 517 == Auftragsstatus = Storniert
					// 518 == Auftragsstatus = Wird bezahlt
					// Anderer Wert == Keine Ahnung, was passiert ist
				}


				// If delivery has happened, then only refunds are available. Try cancel first though, then use refunds.
			} elseif ( $oContractData->deliveryDone && (in_array($oContractData->state,$this->_aCreditReturnStates)) ) {


				$aCancelData = array(
					'dealerOrderNumber' => $sDealerOrderNumber,
					'cancelationFrom' => $sUserName,
					'dealerNumber' => $this->_oSystemConfigService->get('SicoCreditPlus.config.sCPDealer',$oOrder->getSalesChannelId())
				);

				$oResponse = $oWSApi->cancelOrderCPWebshop($aCancelData);
				$this->oSicoCreditPlusLogger->log(false,var_export($oResponse,true),['file' => __FILE__,'line' => __LINE__]);

				if ( is_object($oResponse) && ( property_exists($oResponse,'confirmation') )  && ( property_exists($oResponse->confirmation,'confirmationItems') )  && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
					$iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
					$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
					$aError = array(
						'sError' => (500+$iErrorCode),
						'sErrorMessage' => 'credit-plus-error.status-code-'.$iErrorCode,
						'sErrorMessagePlain' => $sErrorMessage
					);
					// 501 ==  Kein Auftrag gefunden
					// 502 == Pflichtfelder fehlen, kann durch Code heraus nicht passieren
					// 514 == Irgendein Pflichtfeld sprengt sein Format/Länge
					// dealerOrderNumber > 40
					// cancelationFrom > 27
					// 516 == Auftragsstatus = Abgelehnt
					// 517 == Auftragsstatus = Storniert
					// Sonstige Werte == Keine Ahnung, was passiert ist
					if ( $iErrorCode == 6 ) {
						// 506 == Storno im Status "Bezahlt" kann nicht stattfinden
						$aError = $this->returnOrderAfterError($oWSApi, $oContractData, $sUserName, $sDealerOrderNumber);
					} elseif ( $iErrorCode == 15 ) {
						// 515 == Auftragsstatus = Unbekannt
						// oder Auftragsstatus = in Bearbeitung:
						// Hier landet die Bestellung auch, wenn diese endgültig genehmigt ist und bereits versendet wurde.
						$aError = $this->returnOrderAfterError($oWSApi, $oContractData, $sUserName, $sDealerOrderNumber);
					} elseif ( $iErrorCode == 18 ) {
						// 518 == Auftragsstatus = Wird bezahlt
						$aError = $this->returnOrderAfterError($oWSApi, $oContractData, $sUserName, $sDealerOrderNumber);
					}
				}
			} elseif ( ($oContractData->state === '') || ($oContractData->state === 92) || ($oContractData->state === 93) || ($oContractData->state === 95) ) {
				// If it isn't finished on CreditPlus, it is ok to cancel anyway.
				// If it is forbidden to pay with CreditPlus, cancel is also ok.
				$aError = array(
					'sError' => (401),
					'sErrorMessage' => 'credit-plus-error.invalid-state',
					'sErrorMessagePlain' => 'Die Bestellung ist aufgrund des Status bei CreditPlus nicht bei CreditPlus stornierbar.<br />Die Bestellung im Shop wurde jedoch auf den Status storniert gesetzt.'//$oLang->get('ErrorCancelNotAvailable')
				);
			} else {
				// Any other state needs no special treatment
			}
		} else {
			$aError = array(
				'sError' => (400),
				'sErrorMessage' => 'credit-plus-error.already-cancelled',
				'sErrorMessagePlain' => 'Die Bestellung ist aufgrund des Status bei CreditPlus nicht bei CreditPlus stornierbar.<br />Die Bestellung im Shop wurde jedoch auf den Status storniert gesetzt.'//$oLang->get('ErrorCancelAlreadyCancelled')
			);
		}
		if ( !$aError || ($aError['sError'] == 401) ) {
			// If no error or unimportant error
			$this->setOrderStatus($oOrder,'cancel');
		}
		return $aError;
	}

	/**
	 * @param OrderEntity $oOrder
	 */
	private function resetInstockProducts($oOrder){
		$aLineItems = $oOrder->getLineItems();
		if($aLineItems === null){
			/** @var EntityRepository $oOrderRepo */
			$oOrderRepo = $this->container->get(OrderDefinition::ENTITY_NAME.'.repository');
			$oOrderCriteria = new Criteria([$oOrder->getId()]);
			$oOrderCriteria->addAssociation('lineItems.product');
			/** @var OrderEntity $oOrder */
			$oOrder = $oOrderRepo->search($oOrderCriteria,$this->getContext())->first();
			$aLineItems = $oOrder->getLineItems();
		}

		foreach($aLineItems as $lineItem){
			$oProduct = $lineItem->getProduct();
			$oQuantity = $lineItem->getQuantity();
			if($oProduct === null){
				$oProduct = $this->oProductRepo->search(new Criteria([$lineItem->getProductId()]),$this->getContext())->first();
			}
			$productInstock = $oProduct->getStock();
			$productAvailableInstock = $oProduct->getAvailableStock();
			$update = ['id' => $oProduct->getId()];
			if($productAvailableInstock !== null){
				$productAvailableInstockUpdate = $productAvailableInstock + $oQuantity;
				$update['available_stock'] = $productAvailableInstockUpdate;
			}
			$this->oProductRepo->update([$update],$this->getContext());
		}
	}

	/**
	 * @param CreditPlusWebshopAPI $oWSApi The Webshop API Object
	 * @param WebshopContract $oContractData The WebshopContract to be cancelled
	 * @param string $sUserName Backend username (for documentation purposes)
	 * @param string $sDealerOrderNumber DealerOrderNumber (CP2016...)
	 *
	 * @return array array(sError => error_code, sErrorMessage => error_text)
	 */
	protected function returnOrderAfterError($oWSApi, $oContractData, $sUserName, $sDealerOrderNumber) {
		// Collect returnable amount (= price - already returned items)
		$dReturnAmount = floatval($oContractData->price);
		$sDate = date('c');
		foreach ( $oContractData->voucher as $oVoucher ) {
			$dReturnAmount -= floatval($oVoucher->value);
		}
		$oVoucher = new WebshopVoucher($dReturnAmount,'Restbetrag',$sDate,$sUserName,false,$oContractData->id,$sDealerOrderNumber,$oContractData->dealerNumber);
		$oWSApi->addReturnProduct($oVoucher);

		$oResponse = $oWSApi->returnProductCPWebshop();

		// Fehlerhandling
		if ( is_object($oResponse) && ( property_exists($oResponse,'confirmation') ) && ( property_exists($oResponse->confirmation,'confirmationItems') )  && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
			$iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
			$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
			$aError = array(
				'sError' => (500+$iErrorCode),
				'sErrorMessage' => 'credit-plus-error.confirmation-error-code-'.$iErrorCode,
				'sErrorMessagePlain' => $sErrorMessage
			);

			// 501 == Kein Auftrag gefunden
			// 502 == Pflichtfelder fehlen, kann durch Code heraus nicht passieren
			// 507 == Retoure im Status "Nicht bezahlt" kann nicht stattfinden.
			// Nicht möglich, da dieser Aufruf nur im Status "Bezahlt" gemacht wird.
			// 512 == Summe der Retouren übersteigt Restwert des Kreditvertrags
			// 514 == Irgendein Pflichtfeld sprengt sein Format/Länge
			// dealerOrderNumber > 40
			// cancelationFrom > 27
			// Sonstige Werte == Keine Ahnung, was passiert ist
		} else {
			$aError = array();
		}
		return $aError;
	}


	/**
	 * Sends order.
	 * Copied from order_main, extended by commitDeliveryCPWebshop
	 * @param OrderEntity $oOrder
	 * @return array
	 * @see Shopware_Controllers_Backend_SicoCreditPlusOrder::sendOrderAction()
	 * @see CreditPlusWebshopAPI::commitDeliveryCPWebshop()
	 */
	public function sendOrder($oOrder)
	{
		$aError = array();

		$payment = $oOrder->getTransactions()->first()->getPaymentMethod()->getHandlerIdentifier();
		// Only do this for our payment type
		if ( $payment == 'SicoCreditPlus\Service\CreditPlusPayment' ) {
			//$oConfig = Shopware()->Config();
			$oWSApi = $this->getWebshopAPI($oOrder->getSalesChannelId());
			$oContractData = $this->getContractData($oOrder->getSalesChannelId(),$oOrder);
			$aData = array(
				'dealerNumber' => $this->_oSystemConfigService->get('SicoCreditPlus.config.sCPDealer',$oOrder->getSalesChannelId()),
				'dealerOrderNumber' => $oContractData->dealerOrderNumber,
				'invoiceNumber' => $oOrder->getOrderNumber(),
				'invoicePrice' => $oOrder->getPrice()->getTotalPrice(),
				'deliveryDate' => date('c')
			);
			$oResponse = $oWSApi->commitDeliveryCPWebshop( $aData );
			if ( is_string($oResponse) ) {
				$aError['sError'] = 500;
				$aError['sErrorMessage'] = 'credit-plus-error.string-response';
				$aError['sErrorMessagePlain'] = $oResponse;
			} else {
				if ( (property_exists($oResponse,'confirmation')) && (property_exists($oResponse->confirmation,'isSucceed')) && ( $oResponse->confirmation->isSucceed ) ) {
					// No error
					// $aError['sError'] = 200;
					// $aError['sErrorMessage'] = 'Status geändert';
				} else {
					if ( ( property_exists($oResponse,'confirmation') ) && ( property_exists($oResponse->confirmation,'confirmationItems') ) && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
						$iErrorCode = intval($oResponse->confirmation->confirmationItems->errorCode);
						$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
						$aError = array(
							'sError' => (500+$iErrorCode),
							'sErrorMessage' => 'credit-plus-error.send-order-code-'.$iErrorCode,
							'sErrorMessagePlain' => $sErrorMessage
						);
						// 501 == Kein Auftrag gefunden.
						// 502 == Pflichtfelder fehlen, kann durch Code heraus nicht passieren.
						// 503 == Rechnungssumme > Kreditsumme.
						// 504 == Lieferdatum in der Zukunft.
						// 505 == Vertrag ist nicht genehmigt.
						// 506 == Vertrag ist bereits ausbezahlt.
						// 508 == Vertrag ist in falschem Zustand.
						// 510 == Vertrag ist nicht finallyApproved.
						// 511 == Vertrag steht bereits auf ausgeliefert.
						// 514 == Irgendein Pflichtfeld sprengt sein Format/Länge.
						// dealerOrderNumber > 40
						// invoiceNumber > 40
						// 523 == Rechnungssumme < Kreditsumme.
						// Sonstige Werte == Keine Ahnung, was passiert ist
					} else {
						$aError['sError'] = 4.99;
						$aError['sErrorMessage'] = 'credit-plus-error.xzibited';
						$aError['sErrorMessagePlain'] = 'An Error has been Xzibited (An Error in an Error has occurred)';
					}
				}
			}

			if ( !$aError ) {
				$this->setPaymentStatus($oOrder,'creditplus_approved_and_dispatched');
				//Setzt den Bestellstatus auf Komplett
				$this->setOrderStatus($oOrder,'process');
				$this->setOrderStatus($oOrder,'complete');

			}
		}
		return $aError;
	}


	/**
	 * @param SalesChannelContext $context
	 * @param Cart $cart
	 * @param WebshopRateTableMonthRow[] $aMonths Financing months, if already provided
	 * @return bool true if Payment is excluded (either by Session or Basket)
	 */
	public function isPaymentExcluded(SalesChannelContext $context, Cart $cart, array $aMonths = array()): bool {
		$bSccpExcluded = $this->isPaymentExcludedByMissingApi($context);
		$bSccpExcluded = $bSccpExcluded || $this->isPaymentExcludedByBasket($context, $cart);
		$bSccpExcluded = $bSccpExcluded || $this->isPaymentExcludedByArticlesInBasket($context, $cart, $aMonths);
		return $bSccpExcluded;
	}


	/**
	 * If the API is not callable, the payment type should not be offered
	 * @param SalesChannelContext $context
	 * @return bool
	 */
	public function isPaymentExcludedByMissingApi(SalesChannelContext $context): bool {
		$oWSAPI = $this->getWebshopAPI($context->getSalesChannelId());
		if ( !$oWSAPI->isApiCallable() ) {
			return true;
		}
		return false;
	}


	/**
	 * Check Basket price for "excluded"
	 * @param SalesChannelContext $context
	 * @param Cart $cart
	 * @return bool
	 */

	public function isPaymentExcludedByBasket(SalesChannelContext $context, Cart $cart): bool {
		// Exclude by price
		$dBasketPrice = $cart->getPrice()->getTotalPrice();
		$dMaxBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMaxBasketPrice',$context->getSalesChannelId()));
		$dMinBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',$context->getSalesChannelId()));
		return (($dBasketPrice < $dMinBasketPrice) || ($dBasketPrice > $dMaxBasketPrice));
	}

	/**
	 * Checks the current basket articles for availability of sccp_financing
	 *
	 * @param SalesChannelContext $context
	 * @param Cart $oCart
	 * @param WebshopRateTableMonthRow[] $aMonths Financing months if provided
	 * @return bool
	 */
	public function isPaymentExcludedByArticlesInBasket(SalesChannelContext $context, Cart $oCart, array $aMonths = array()): bool {
		$dAmount = $oCart->getPrice()->getTotalPrice();
		foreach ( $oCart->getLineItems() as $oBasketArticle ) {
			// Skip basket contents which are not normal products
			if ( !$oBasketArticle->isGood() ) {
				continue;
			}
			if($oBasketArticle->getReferencedId() == null){
				// We have a Problem the Referenced Id which should be the ProductId is null
				return true;
			}
			/** @var ProductEntity $oProductEntity */
			$oProductEntity = $this->oProductRepo->search(
				new Criteria([$oBasketArticle->getReferencedId()]),$context->getContext()
			)->get($oBasketArticle->getReferencedId());

			// Product can not be identified, assume blocked for basket
			if ( $oProductEntity == null ) {
				return true;
			}
			/** @var CurrencyEntity $oCurrency */
			$oCurrency = $this->oCurrencyRepo->search(new Criteria([$context->getCurrencyId()]),$context->getContext())->get($context->getCurrencyId());

			if ( !$aMonths ) {
				$aMonths = $this->getFinancingMonths($context,$oCurrency, $oProductEntity, null, $dAmount);
				// No months == product excluded
				if ( !$aMonths ) {
					return true;
				}
			}
		}

		return false;
	}


	/**
	 * Returns all CreditPlus payment states
	 *
	 * @return array CreditPlus payment states as array
	 */
	public function getCreditPlusPaymentStates() {
		$aStates = array(
			'creditplus_referred' => array(
				// Krebes 20
				'name' => 'creditplus_referred',
				'description' => 'Antrag ausgefüllt',
				'group' => 'payment',
				'position' => 61,
				'mail' => 0
			),
			'creditplus_accepted' => array(
				// Krebes 24
				'name' => 'creditplus_accepted',
				'description' => 'Antrag angenommen',
				'group' => 'payment',
				'position' => 62,
				'mail' => 0
			),
			'creditplus_declined_soft' => array(
				// Krebes 92
				'name' => 'creditplus_declined_soft',
				'description' => 'Weich abgelehnt',
				'group' => 'payment',
				'position' => 71,
				'mail' => 0
			),
			'creditplus_declined_hard' => array(
				// Krebes 93
				'name' => 'creditplus_declined_hard',
				'description' => 'Abgelehnt',
				'group' => 'payment',
				'position' => 72,
				'mail' => 0
			),
			'creditplus_approved' => array(
				// Krebes 24, Approved
				'name' => 'creditplus_approved',
				'description' => 'Lieferfreigabe',
				'group' => 'payment',
				'position' => 73,
				'mail' => 0
			),
			'creditplus_approved_and_dispatched' => array(
				// Krebes 24, Approved, Delivered
				'name' => 'creditplus_approved_and_dispatched',
				'description' => 'Ausgeliefert',
				'group' => 'payment',
				'position' => 74,
				'mail' => 0
			),
			'creditplus_processing_payment' => array(
				// Krebes 32
				'name' => 'creditplus_processing_payment',
				'description' => 'Auszahlung',
				'group' => 'payment',
				'position' => 75,
				'mail' => 0
			),
			'creditplus_paid' => array(
				// Krebes 99
				'name' => 'creditplus_paid',
				'description' => 'Ausgezahlt',
				'group' => 'payment',
				'position' => 76,
				'mail' => 0
			),
			'creditplus_cancelled' => array(
				// Krebes 95
				'name' => 'creditplus_cancelled',
				'description' => 'Storniert',
				'group' => 'payment',
				'position' => 77,
				'mail' => 0
			),
			'creditplus_error' => array(
				// Krebes 94
				'name' => 'creditplus_error',
				'description' => 'Fehler',
				'group' => 'payment',
				'position' => 78,
				'mail' => 0
			),
			'creditplus_docs_received' => array(
				// Krebes 21, No information request
				'name' => 'creditplus_docs_received',
				'description' => 'Warten auf Unterlagen',
				'group' => 'payment',
				'position' => 79,
				'mail' => 0
			),
		);

		return $aStates;
	}

	/**
	 * @param int|string $iStateNumber State in Krebes
	 * @param bool|int $bFinallyApproved Whether this request has been approved by CreditPlus
	 * @param bool|int $bDeliveryDone Whether the shop has already sent out the goods
	 * @param bool|int $bRequestInformation True when additional information is requested
	 *
	 * @return string Name of the state for the given parameters
	 */

	public function getCreditPlusPaymentState( $iStateNumber, $bFinallyApproved = false, $bDeliveryDone = false, $bRequestInformation = false ): string {
		$sStatus = 'review_necessary';
		if ( $iStateNumber == 10 ) {
			$sStatus = 'borrowing_request_shown';
		} elseif ( $iStateNumber == 20 ) {
			$sStatus = 'creditplus_referred';
		} elseif ( $iStateNumber == 24 ) {
			if ( $bFinallyApproved == false ) {
				// $bDeliveryDone = not important because CreditPlus approval is missing
				$sStatus = 'creditplus_accepted';
			} elseif ( $bDeliveryDone == true ) {
				// $bFinallyApproved = true && $bDeliveryDone = true - Done and sent
				$sStatus = 'creditplus_approved_and_dispatched';
			} else {
				// $bFinallyApproved = true && $bDeliveryDone = false - Ready for dispatch
				$sStatus = 'creditplus_approved';
			}
		} elseif ( $iStateNumber == 21 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 22 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 23 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 25 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 32 ) {
			$sStatus = 'creditplus_processing_payment';
		} elseif ( $iStateNumber == 92 ) {
			$sStatus = 'creditplus_declined_soft';
		} elseif ( $iStateNumber == 93 ) {
			$sStatus = 'creditplus_declined_hard';
		} elseif ( $iStateNumber == 99 ) {
			$sStatus = 'creditplus_paid';
		} elseif ( $iStateNumber == 95 ) {
			$sStatus = 'creditplus_cancelled';
		} elseif ( $iStateNumber == 94 ) {
			$sStatus = 'creditplus_error';
		}
		return $sStatus;
	}

	/**
	 * @param string $sPaymentSate
	 *
	 * @return StateMachineStateEntity|null
	 */
	public function getPaymentStateByName( string $sPaymentSate ): ?StateMachineStateEntity {
		if ( !isset(self::$_aPaymentStatesCache[$sPaymentSate]) ) {
			/** @var StateMachineStateCollection $oStateMachineStateCollection */
			$oStateMachineStateCollection = $this->oStateMachineStateRepo->search(
				(new Criteria())->addFilter(new EqualsFilter('technicalName', $sPaymentSate),new EqualsFilter('stateMachine.technicalName','order_transaction.state')),
				$this->salesChannelContext->getContext()
			)->getEntities();
			if($oStateMachineStateCollection->count() > 0){
				/* The Db Search should only get one result*/
				self::$_aPaymentStatesCache[$sPaymentSate] = $oStateMachineStateCollection->first();
			}else{
				self::$_aPaymentStatesCache[$sPaymentSate] = null;
			}
		}
		return self::$_aPaymentStatesCache[$sPaymentSate];
	}

	/**
	 * @param string $sOrderState
	 * @return StateMachineStateEntity|null
	 */
	public function getOrderStateByName( string $sOrderState): ?StateMachineStateEntity {
		if(!isset(self::$_aOrderStatesCache[$sOrderState])){
			$oStatMachineStateCollection = $this->oStateMachineStateRepo->search(
				(new Criteria())->addFilter(new EqualsFilter('technicalName',$sOrderState),new EqualsFilter('stateMachine.technicalName','order.state')),
				$this->salesChannelContext->getContext()
			)->getEntities();
			if($oStatMachineStateCollection->count() > 0){
				self::$_aOrderStatesCache[$sOrderState] = $oStatMachineStateCollection->first();
			} else {
				self::$_aOrderStatesCache[$sOrderState] = null;
			}
		}
		return self::$_aOrderStatesCache[$sOrderState];
	}

	/**
	 * Triggers Change Mail like in Bootstrap
	 * The code here is taken mostly from Shopware_Controllers_Backend_Order
	 *
	 * @see Shopware_Plugins_Frontend_SicoCreditPlus_Bootstrap::onSicoCreditPlusFinancingStatusChanged()
	 * @param OrderEntity|Cart $oOrder
	 *
	 * @throws Exception If a query builder function can not be found
	 */
	public function checkForUpdateMail($sSalesChannelId, $oOrder) {
		$oContractData = $this->getContractData($sSalesChannelId,$oOrder);
		if(!is_object($oContractData)){
			return;
		}
		$sNewState = $this->getCreditPlusPaymentState($oContractData->state, $oContractData->finallyApproved, $oContractData->deliveryDone, $oContractData->informationRequest);

		$oState = $this->getPaymentStateByName($sNewState);
		if($oOrder instanceof Cart){
			return;
		}
		$sStateTechnicalName = $oOrder->getTransactions()->first()->getStateMachineState()->getTechnicalName();
		$this->oSicoCreditPlusLogger->log(false,'check for Update Mail',['sStateTechnicalName' => $sStateTechnicalName,'otherStateName' => $oState->getTechnicalName(),'FILE' => __FILE__,'LINE' => __LINE__]);
		if($oState && ($sStateTechnicalName != $oState->getTechnicalName())){
			if ( $sNewState != 'review_necessary' ) {
				$oOrder->getTransactions()->first()->setStateId(md5($oState->getTechnicalName()));
				$oOrder->getTransactions()->first()->setStateMachineState($oState);

				$this->setPaymentStatus($oOrder,$sNewState);
			}
		}
	}


	/**
	 * @param OrderEntity $oOrder
	 * @param string $sNewState
	 */
	public function setOrderStatus(OrderEntity $oOrder, string $sNewState){
		$context = null;
		if($this->context == null){
			$context = $this->salesChannelContext->getContext();
		}
		else {
			$context = $this->context;
		}
		if($sNewState != 'reopen' && ($oOrder->getStateMachineState()->getTechnicalName() == 'cancelled')){
			// reopen should reset cancelled to open again
			return;
		}
		try{
			$this->stateMachineRegistry->transition(new Transition(
				OrderDefinition::ENTITY_NAME,
				$oOrder->getId(),
				$sNewState,
				'stateId'
			),$context);
		} catch( Exception $e){
			$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),['file' => $e->getFile(),'line' => $e->getLine()]);
		}
	}

	/**
	 * @param OrderEntity|OrderTransactionEntity|Cart $oOrder
	 * @param string $sPaymentstate
	 * @throws Exception
	 */
	public function setPaymentStatus($oOrder, string $sPaymentstate){
		$state2TransitionsMapping = [
			'borrowing_request_shown' => 'cp_borrowing_request_shown',
			'creditplus_referred' => 'cp_duly_completed',
			'creditplus_accepted' => 'cp_contract_accepted',
			'creditplus_approved' => 'cp_contract_approved',
			'creditplus_approved_and_dispatched' => 'cp_delivered',
			'creditplus_processing_payment' => 'cp_pay',
			'creditplus_paid' => 'cp_paid',
			'creditplus_declined_soft' => 'cp_contract_declined_soft',
			'creditplus_declined_hard' => 'cp_contract_declined_hard',
			'creditplus_cancelled' => 'cp_cancelled',
			'review_necessary' => 'cp_reviewNecessary'
		];

		if($sPaymentstate == 'review_necessary'){
			return;
		}
		//if the state is unknown then the Order is probably not a CreditPlus Order
		if(!isset($state2TransitionsMapping[$sPaymentstate])){
			$this->oSicoCreditPlusLogger->log(true,'Transition not found',['file' => __FILE__,'line' => __LINE__,'paymentstate' => $sPaymentstate]);
			return;
		}
		$sTransitionName = $state2TransitionsMapping[$sPaymentstate];
		if($sTransitionName != null){
			if($oOrder instanceof OrderEntity){
				/** @var OrderTransactionEntity $oOrderTransaction */
				$oOrderTransaction = $this->oOrderTransactionRepo->search(
					(new Criteria())->addFilter(new AndFilter([
						new EqualsFilter('orderId', $oOrder->getId()),
						new EqualsFilter('versionId',$oOrder->getVersionId())
					])),
					$this->getContext()
				)->first();

				if($oOrderTransaction){
					$oOrderTransactionState = $oOrderTransaction->getStateMachineState();
					if($oOrderTransactionState){
						if($oOrderTransactionState->getTechnicalName() == 'open'){
							try {
								$this->stateMachineRegistry->transition(new Transition(
									OrderTransactionDefinition::ENTITY_NAME,
									$oOrderTransaction->getId(),
									'cp_borrowing_request_shown',
									'stateId'
								), $this->getContext());
							}catch(\Exception $e){
								$this->oSicoCreditPlusLogger->log('true',$e->getMessage(),['file' => __FILE__,'line' => __LINE__]);
							}
						}
					} else {
						$oOpenStateEntity = $this->getPaymentStateByName('open');
						if($oOrderTransactionState->getStateMachineId() == $oOpenStateEntity->getId()){
							try{
								$this->stateMachineRegistry->transition(new Transition(
									OrderTransactionDefinition::ENTITY_NAME,
									$oOrderTransaction->getId(),
									'cp_borrowing_request_shown',
									'stateId'
								),$this->getContext());
							} catch(\Exception $e){
								$this->oSicoCreditPlusLogger->log('true',$e->getMessage(),['file' => __FILE__,'line' => __LINE__]);
							}
						}
					}
				}
			}
			else if($oOrder instanceof Cart){
				//Mail for declined Status came from creditPlus
				if($sPaymentstate == 'creditplus_declined_hard' || $sPaymentstate == 'creditplus_declined_soft'){
					return;
				}
				$context = $this->getContext();
				$iMailSent = $this->sendMail($context,$oOrder,$sPaymentstate);
				if ( $iMailSent === self::MAIL_SENT ) {
					$this->oSicoCreditPlusLogger->log(false,'e-mail sent', ['Transition Name' => $sTransitionName]);
				}
				return;
			}
			else{
				$oOrderTransaction = $oOrder;
			}

			$oOrderTransactionId = $oOrderTransaction->getId();

			try{
				$context = $this->getContext();
				$this->stateMachineRegistry->transition(new Transition(
					OrderTransactionDefinition::ENTITY_NAME,
					$oOrderTransactionId,
					$sTransitionName,
					'stateId'
				),$context);

				if($sPaymentstate == 'creditplus_cancelled' && $oOrder instanceof OrderEntity){
					//$this->resetInstockProducts($oOrder);
					$this->setOrderStatus($oOrder,'cancel');
				}
				//Mail for declined Status came from creditPlus
				if($sPaymentstate == 'creditplus_declined_hard' || $sPaymentstate == 'creditplus_declined_soft'){
					return;
				}
				$iMailSent = $this->sendMail($context,$oOrder,$sPaymentstate);
				if ( $iMailSent === self::MAIL_SENT ) {
					$this->oSicoCreditPlusLogger->log(false,'e-mail sent',['Transition Name' => $sTransitionName]);
				}
			} catch( Exception $e){

				$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),['Transition Name' => $sTransitionName,'OrderTransactionId' => $oOrderTransactionId,'file' => $e->getFile(),'line' => $e->getLine(),'file2' => __FILE__,'line2' => __LINE__]);
			}
		}
		else{
			$this->oSicoCreditPlusLogger->log(true,'Cannot find transitionName to state',['paymentState' => $sPaymentstate]);
		}
	}

	private function getContext(){
		$context = null;
		if($this->context == null){
			$context = $this->salesChannelContext->getContext();
		}
		else {
			$context = $this->context;
		}
		return $context;
	}

	/**
	 * @param Context $context
	 * @param OrderEntity|OrderTransactionEntity|Cart $oOrder
	 * @param string $paymentState
	 * @return int
	 * @throws Exception
	 */
	public function sendMail(Context $context, $oOrder, string $paymentState) {
		$paymentState2StateMapping = [
			'creditplus_referred' => '20',
			'creditplus_accepted' => '24n',
			'creditplus_approved' => '24a',
			'creditplus_paid' => '99',
			'creditplus_declined_soft' => '92',
			'creditplus_declined_hard' => '92'//I Mapped it to 92 because I don't need a new email Template
		];
		//Der Status ist nicht in der Liste, daher muss keine Mail verschickt werden
		if(!isset($paymentState2StateMapping[$paymentState])){
			$aLogInfo = ['FILE' => __FILE__,'LINE' => __LINE__,'paymentState' => $paymentState];
			$sSalesChannelId = null;
			if($oOrder instanceof OrderEntity){
				$sSalesChannelId = $oOrder->getSalesChannelId();
				$aLogInfo['orderNumber'] = $oOrder->getOrderNumber();
				$oContract = $this->getContractData($sSalesChannelId, $oOrder);
				$aLogInfo['transactionNumber'] = isset($oContract)?$oContract->dealerOrderNumber:'';
			} else if($oOrder instanceof OrderTransactionEntity){
				$oOrderObject = $oOrder->getOrder();
				$sSalesChannelId = $oOrderObject->getSalesChannelId();
				$aLogInfo['orderNumber'] = $oOrderObject->getOrderNumber();
				$oContract = $this->getContractData($sSalesChannelId, $oOrderObject);
				$aLogInfo['transactionNumber'] = isset($oContract)?$oContract->dealerOrderNumber:'';
			} else if($oOrder instanceof Cart){
				$sSalesChannelId = $this->salesChannelContext->getSalesChannel()->getId();
				$sDealerOrderNumber = $this->oSession->get('cpDealerOrderNumber');
				if($sDealerOrderNumber === null){
					$sDealerOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber');
				}
				$aLogInfo['orderNumber'] = '';
				$aLogInfo['transactionNumber'] = $sDealerOrderNumber;
			}
			$this->oSicoCreditPlusLogger->log(false,'Status not found for E-Mail', $aLogInfo, $sSalesChannelId);
			return self::MAIL_NO_STATUS;
		}
		$templateName = 'sCreditPlusState'.$paymentState2StateMapping[$paymentState];

		/** @var StateMachineStateEntity $oTargetStateMachineState */
		$oTargetStateMachineState = $this->oStateMachineStateRepo->search(
			(new Criteria())->addFilter(new EqualsFilter('technicalName',$paymentState)),$context
		)->first();
		$sSalesChannelId = '';
		if($oOrder instanceof OrderEntity){
			$sSalesChannelId = $oOrder->getSalesChannelId();
		} else if($oOrder instanceof OrderTransactionEntity){
			$sSalesChannelId = $oOrder->getOrder()->getSalesChannelId();
		} else if($oOrder instanceof Cart){
			$sSalesChannelId = $this->salesChannelContext->getSalesChannel()->getId();
		}

		$aVariables = [];
		$address = $this->_oSystemConfigService->get('SicoCreditPlus.config.sStatusMailAdress',$sSalesChannelId);
		if($oOrder instanceof OrderEntity){
			$oOrderTransaction = $oOrder->getTransactions()->first();
			$oContract = $this->getContractData($sSalesChannelId, $oOrder);
			$aVariables = [
				'order'=> $oOrder,
				'orderTransaction' => $oOrderTransaction,
				'newState' => $oTargetStateMachineState->getTechnicalName(),
				'address' => $address,
				'orderTransactionNumber' => $oContract?$oContract->dealerOrderNumber:'',
				'webshopContract' => $oContract,
			];
		} else if($oOrder instanceof OrderTransactionEntity){
			$oOrderTransaction = $oOrder;
			$oOrder = $oOrder->getOrder();
			$oContract = $this->getContractData($sSalesChannelId, $oOrder);
			$aVariables = [
				'order'=> $oOrder,
				'orderTransaction' => $oOrderTransaction,
				'newState' => $oTargetStateMachineState->getTechnicalName(),
				'address' => $address,
				'orderTransactionNumber' => $oContract?$oContract->dealerOrderNumber:'',
				'webshopContract' => $oContract,
			];
		} else if($oOrder instanceof Cart){
			$oOrderTransactionNumber = $this->oSession->get('cpDealerOrderNumber');
			if($oOrderTransactionNumber === null){
				$oOrderTransactionNumber = $oOrder->getData()->get('cpDealerOrderNumber');
			}
			$oCustomer = $this->salesChannelContext->getCustomer();
			/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
			$oCpPaymentEntity  = $this->oCpPayment->search((new Criteria())->addFilter(new EqualsFilter('orderNumber',$oOrderTransactionNumber)),$context)->first();
			$oContract = $this->getContractData($sSalesChannelId, $oOrder);
			$aVariables = [
				'firstName'=> $oCustomer->getFirstName(),
				'lastName' => $oCustomer->getLastName(),
				'orderDateTime' => $oCpPaymentEntity->getGenTimestamp(),
				'newState' => $oTargetStateMachineState->getTechnicalName(),
				'address' => $address,
				'orderTransactionNumber' => $oOrderTransactionNumber,
				'webshopContract' => $oContract,
			];
			$templateName = 'sCreditPlusStateBeforeOrder'.$paymentState2StateMapping[$paymentState];

		}

		$id = md5($templateName);
		$criteria = new Criteria([$id]);
		/** @var MailTemplateEntity|MailTemplateTranslationEntity $oMailTemplateEntity */
		$oMailTemplateEntity = $this->oMailTemplateRepo->search($criteria,$context)->get($id);

		if($oMailTemplateEntity === null){
			$this->oSicoCreditPlusLogger->log(true,'Mail template not found',['FILE' => __FILE__,'LINE' => __LINE__,'mail template id ' => $id]);
			return self::MAIL_NO_TEMPLATE;
		}

		if(empty($oMailTemplateEntity->getSubject()) ||
			empty($oMailTemplateEntity->getSenderName()) ||
			empty($oMailTemplateEntity->getContentHtml()) ||
			empty($oMailTemplateEntity->getContentPlain())){
			$oMailTemplateEntity = $this->oMailTemplateTranslationRepo->search((new Criteria())->addFilter(new EqualsFilter('mailTemplateId',$id)),$context)->first();
			if($oMailTemplateEntity === null){
				$this->oSicoCreditPlusLogger->log(true,'Mail Template is not defined',[
					'FILE' => __FILE__,
					'LINE' => __LINE__,
					'Mail_template_id' => $id,
					'mail_template_name' => $templateName,
					'MailTemplateEntity' => $oMailTemplateEntity,
				]);
				return self::MAIL_NO_TEMPLATE;
			}
		}


		$email = $this->getStatusEmailReceiver($paymentState,$sSalesChannelId);
		if(!empty($email)){
			$data = new DataBag();
			$data->set('salesChannelId',$sSalesChannelId);
			$data->set('recipients', [$email => $email]);
			$data->set('subject', $oMailTemplateEntity->getSubject());
			$data->set('senderName', $oMailTemplateEntity->getSenderName());
			$data->set('contentHtml', $oMailTemplateEntity->getContentHtml());
			$data->set('contentPlain', $oMailTemplateEntity->getContentPlain());

			$aVariables['newState'] = $this->oTranslator->trans('sicoCreditplus.states.'.$aVariables['newState']);
			// $this->oSicoCreditPlusLogger->log(false,'Status Mail Send',['senderName' => $oMailTemplateEntity->getSenderName(),'recv-email'=>$email,'subject'=>$oMailTemplateEntity->getSubject(),'contentHTML' => $oMailTemplateEntity->getContentHtml()]);
			try{
				$this->oMailService->send($data->all(),$context,$aVariables);
			}catch(SalesChannelNotFoundException $e){
				$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),['FILE' => $e->getFile(),'LINE' => $e->getLine()]);
			}
			catch(ConstraintViolationException $e){
				$violations = $e->getViolations();
				$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),[
					'FILE' => $e->getFile(),
					'LINE' => $e->getLine(),
					'VIOLATIONS' => $violations,
					'data' => var_export($data->all(),true)]);
				throw $e;
			}
			catch(\Exception $e){
				$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),[
					'FILE' => $e->getFile(),
					'LINE' => $e->getLine(),
					'data' => var_export($data->all(),true)]);
				throw $e;
			}
		} else {
			return self::MAIL_NO_RECEIVER;
		}
		return self::MAIL_SENT;
	}

	public function sendActivationFailedMail(OrderEntity $oOrder)
	{
		$sSalesChannelId = $oOrder->getSalesChannelId();
		$id = md5('sCreditPlusOrderActivationFailed');
		/** @var MailTemplateEntity|MailTemplateTranslationEntity $oMailTemplateEntity */
		$oMailTemplateEntity = $this->oMailTemplateRepo->search(new Criteria([$id]),$this->salesChannelContext->getContext())->get($id);
		if($oMailTemplateEntity == null){
			$this->oSicoCreditPlusLogger->log(true,'Could Not find sCreditPlusOrderActivationFailed',['id' => $id]);
			return self::MAIL_NO_TEMPLATE;
		}

		if($oMailTemplateEntity->getSubject() == null ||
			$oMailTemplateEntity->getSenderName() == null ||
			$oMailTemplateEntity->getContentHtml() == null ||
			$oMailTemplateEntity->getContentPlain() == null){
			$oMailTemplateEntity = $this->oMailTemplateTranslationRepo->search((new Criteria())->addFilter(new EqualsFilter('mailTemplateId',$id)),$this->salesChannelContext->getContext())->first();
			if($oMailTemplateEntity == null){
				$this->oSicoCreditPlusLogger->log(true,'Could not send activation failed Mail because Mail Template not found',[
					'FILE' => __FILE__,
					'LINE' => __LINE__,
					'Mail_template_id' => $id,
					'MailTemplateEntity' => $oMailTemplateEntity,
				]);
				return self::MAIL_NO_TEMPLATE;
			}
		}

		$oOrderTransaction = $oOrder->getTransactions()->first();
		$oContract = $this->getContractData($sSalesChannelId, $oOrder);
		$aVariables = [
			'order'=> $oOrder,
			'aOrders' => [$oOrder],
			'orderTransaction' => $oOrderTransaction,
			'orderTransactionNumber' => $oContract?$oContract->dealerOrderNumber:'',
			'webshopContract' => $oContract,
		];

		$data = new DataBag();
		$data->set('salesChannelId',$sSalesChannelId);
		$email = $this->_oSystemConfigService->get('SicoCreditPlus.config.sEmailActivationFailed',$sSalesChannelId);
		$aRecipients = [];
		if ( strpos($email, ',') !== false ) {
			$aEmails = explode(',', $email);
			foreach($aEmails as $sEmail){
				$aRecipients[$sEmail] = $sEmail;
			}
		} elseif ( $email ) {
			$aRecipients[$email] = $email;
		}
		if ( !$aRecipients ) {
			$this->oSicoCreditPlusLogger->log(true,'No recipients found for activation failed mail',['email' => $email]);
			return self::MAIL_NO_RECEIVER;
		}
		$data->set('recipients', $aRecipients);
		$data->set('subject', $oMailTemplateEntity->getSubject());
		$data->set('senderName', $oMailTemplateEntity->getSenderName());
		$data->set('contentHtml', $oMailTemplateEntity->getContentHtml());
		$data->set('contentPlain', $oMailTemplateEntity->getContentPlain());
		try{
			$this->oMailService->send($data->all(),$this->salesChannelContext->getContext(),$aVariables);
		} catch (SalesChannelNotFoundException $e) {
			$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),['FILE' => $e->getFile(),'LINE' => $e->getLine()]);
			return self::MAIL_DELIVERY_ERROR;
		} catch (ConstraintViolationException $e) {
			$violations = $e->getViolations();
			$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),[
				'FILE' => $e->getFile(),
				'LINE' => $e->getLine(),
				'VIOLATIONS' => $violations,
				'data' => var_export($data->all(),true)]);
			return self::MAIL_DELIVERY_ERROR;
		} catch (\Exception $e) {
			$this->oSicoCreditPlusLogger->log(true,$e->getMessage(),[
				'FILE' => $e->getFile(),
				'LINE' => $e->getLine(),
				'data' => var_export($data->all(),true)]);
			return self::MAIL_DELIVERY_ERROR;
		} catch (\Throwable $e) {
			$this->oSicoCreditPlusLogger->log(true, $e->getMessage(), [
				'FILE' => $e->getFile(),
				'LINE' => $e->getLine(),
				'data' => var_export($data->all(), true)
			]);
			return self::MAIL_DELIVERY_ERROR;
		}
		return self::MAIL_SENT;
	}

	protected function getStatusEmailReceiver($paymentState, string $sSaleChannelId){
		$state2configMapping = [

			'creditplus_referred' => 'sEmailState20',
			'creditplus_accepted' => 'sEmailState24n',
			'creditplus_approved' => 'sEmailState24a',
			'creditplus_approved_and_dispatched' => 'sEmailState24a',
			'creditplus_processing_payment' => 'sEmailState99',
			'creditplus_paid' => 'sEmailState99',
			'creditplus_declined_soft' => 'sEmailState92',
			'creditplus_declined_hard' => 'sEmailState92',
			'creditplus_cancelled' => 'sEmailState95'
		];
		return $this->_oSystemConfigService->get('SicoCreditPlus.config.'.$state2configMapping[$paymentState],$sSaleChannelId);
	}

	public function sendAdditionalMessageMail(SalesChannelContext $context,OrderEntity $oOrder,string $sTargetUrl){
		try{
			$oWebshopContract = $this->getContractData($oOrder->getSalesChannelId(),$oOrder); // Schlägt fehl, wenn keine Transactions dabei sind

			if($this->_oSystemConfigService->get('SicoCreditPlus.config.sTransactionMode',$context->getSalesChannelId()) == 'inorder'){
				if( in_array(((int)$oWebshopContract->state), array(20,92,93)) ){
					$id = md5('sCreditplusOrderAdditionalMessage20');
				} else {
					$id = md5('sCreditplusOrderAdditionalMessage');
				}
			} else {
				$id = md5('sCreditplusOrderAdditionalMessage');
			}

			/** @var MailTemplateEntity|MailTemplateTranslationEntity $oMailTemplateEntity */
			$oMailTemplateEntity = $this->oMailTemplateRepo->search(new Criteria([$id]),$context->getContext())->get($id);
			if($oMailTemplateEntity == null){
				$this->oSicoCreditPlusLogger->log(true,'Could Not find sCreditplusOrderAdditionalMessage',['id' => $id]);
				return;
			}

			if($oMailTemplateEntity->getSubject() == null ||
			$oMailTemplateEntity->getSenderName() == null ||
			$oMailTemplateEntity->getContentHtml() == null ||
			$oMailTemplateEntity->getContentPlain() == null){
				$oMailTemplateEntity = $this->oMailTemplateTranslationRepo->search((new Criteria())->addFilter(new EqualsFilter('mailTemplateId',$id)),$context->getContext())->first();
				if($oMailTemplateEntity == null){
					$this->oSicoCreditPlusLogger->log(true,'Could not send additional Message because Mail Template not found',[
						'FILE' => __FILE__,
						'LINE' => __LINE__,
						'Mail_template_id' => $id,
						'MailTemplateEntity' => $oMailTemplateEntity,
					]);
					return;
				}
			}

			$data = new DataBag();
			$data->set('salesChannelId',$oOrder->getSalesChannelId());
			$email = $oOrder->getOrderCustomer()->getEmail();
			$name = $oOrder->getOrderCustomer()->getFirstName().' '.$oOrder->getOrderCustomer()->getLastName();
			$data->set('recipients', [$email => $name]);
			$data->set('subject', $oMailTemplateEntity->getSubject());
			$data->set('senderName', $oMailTemplateEntity->getSenderName());
			$data->set('contentHtml', $oMailTemplateEntity->getContentHtml());
			$data->set('contentPlain', $oMailTemplateEntity->getContentPlain());
			// $this->oSicoCreditPlusLogger->log(false,'Additional Message Send',['senderName' => $oMailTemplateEntity->getSenderName(),'recv-email'=>$email,'subject'=>$oMailTemplateEntity->getSubject(),'contentHTML' => $oMailTemplateEntity->getContentHtml()]);
			$this->oMailService->send($data->all(),$context->getContext(),[
				'order'=> $oOrder,
				'sTargetURL' => $sTargetUrl
			]);
		} catch(SalesChannelNotFoundException $e){
			$this->oSicoCreditPlusLogger->log(true,'SalesChannelNotFoundException:'.$e->getMessage(),['FILE' => $e->getFile(),'LINE' => $e->getLine()]);
		} catch(InconsistentCriteriaIdsException $e) {
			$this->oSicoCreditPlusLogger->log(true, 'InconsistentCriteriaIdsException:' . $e->getMessage(), ['FILE' => $e->getFile(), 'LINE' => $e->getLine()]);
		} catch(ConstraintViolationException $e){
			$this->oSicoCreditPlusLogger->log(true, 'ConstraintViolationException:' . $e->getMessage(), ['FILE' => $e->getFile(), 'LINE' => $e->getLine()]);
			$violations = $e->getErrors();
			foreach($violations as $violation){
				$this->oSicoCreditPlusLogger->log(true, 'Violation:' . $violation['detail'], [
					'FILE' => $e->getFile(),
					'LINE' => $e->getLine(),
					'PARAMETERS' => $violation['meta']['parameters'],
					'POINTER' => $violation['source']['pointer']]);
			}
		} catch( Exception $e){
			$this->oSicoCreditPlusLogger->log(true,'Exception:'.$e->getMessage(),['FILE' => $e->getFile(),'LINE' => $e->getLine()]);
		}
	}
	public function getTargetURLForFinishingPayment(string $sOrderTransactionId, string $sSalesChannelId, $oPathDetail = UrlGeneratorInterface::ABSOLUTE_URL) {
		$sReturnUrl = $this->oRouter->generate('frontend.creditplus.trigger',['statusText' => 30307]);
		$sReturnUrl = base64_encode($sReturnUrl);
		$selectedView = $this->_oSystemConfigService->get('SicoCreditPlus.config.sCPShownAs', $sSalesChannelId);
		switch ( $selectedView ) {
			case 'popup':
				$url = $this->oRouter->generate('frontend.creditplus.trigger.popup', ['orderTransactionId' => $sOrderTransactionId, 'returnUrl' => $sReturnUrl], $oPathDetail);
				break;
			case 'iframe':
				$url = $this->oRouter->generate('frontend.creditplus.trigger.iframe', ['orderTransactionId' => $sOrderTransactionId, 'returnUrl' => $sReturnUrl], $oPathDetail);
				break;
			case 'new_window':
				$url = $this->oRouter->generate('frontend.creditplus.trigger.new-window', ['orderTransactionId' => $sOrderTransactionId, 'returnUrl' => $sReturnUrl], $oPathDetail);
				break;
			default:
				$url = $this->oRouter->generate('frontend.creditplus.trigger.popup', ['orderTransactionId' => $sOrderTransactionId, 'returnUrl' => $sReturnUrl], $oPathDetail);
		}
		return $url;
	}

	public function getCreditPlusPaymentMethodId(Context $context):?string{
		$sPaymentMethodId = $this->oPaymentMethodRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('handlerIdentifier',CreditPlusPayment::class)),$context)->firstId();
		return $sPaymentMethodId;
	}

	public function hasProductFinancing(ProductEntity $oProduct,SalesChannelContext $context):bool{
		$dMinBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',$context->getSalesChannelId()));
		$dMaxBasketPrice = floatval($this->_oSystemConfigService->get('SicoCreditPlus.config.sMaxBasketPrice',$context->getSalesChannelId()));
		$aPrices = $oProduct->getPrices();
		$sCurrencyId = $context->getCurrencyId();

		if($aPrices == null){
			$aPrices = $oProduct->getPrice();
			if($aPrices == null){
				return false;
			}
		}
		$dOrderPrice = 0;
		foreach ( $aPrices as $oPrice ) {
			if($oPrice instanceof ProductPriceEntity){
				if($oPrice->getQuantityStart() <= 1 && $oPrice->getQuantityEnd() > 1){
					$dOrderPrice = $oPrice->getPrice()->get($sCurrencyId)->getGross();
				}
			}

			if($oPrice instanceof Price){
				$dOrderPrice = $oPrice->getGross();
			}
		}

		// Product or basket is cheaper than min price => no financing
		if ( $dMinBasketPrice > $dOrderPrice ) {
			return false;
		}
		// Product or basket is more expensive than max price => no financing
		if ( $dMaxBasketPrice < $dOrderPrice ) {
			return false;
		}
		$dMinRate = $this->_oSystemConfigService->get('SicoCreditPlus.config.sMinRate',$context->getSalesChannelId());
		$oWSCurrency = $this->getWebshopCurrency($context->getCurrency());
		$aReturn = [];
		try {
			$oWSApi = $this->getWebshopAPI($context->getSalesChannelId());
			$aReturn = $this->getProductSpecificFinancingMonths($oProduct, $dMinRate, $dOrderPrice, $aReturn, $oWSApi);
			$aReturn = $this->getUnspecificFinancingMonths($dMinRate, $dOrderPrice, $oWSApi, $aReturn);
			$aReturn = $this->getCalculatedFinancingMonths($aReturn, $dOrderPrice, $oWSCurrency);
		} catch(\Exception $e){
			return false;
		}
		if(empty($aReturn)){
			return false;
		}
		return true;
	}
	public function setSession(SessionInterface $session){
		$this->oSession = $session;
	}
}
