<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 11.12.2018
 * @time 11:31
 */

namespace SicoCreditPlus\Components;

use Doctrine\DBAL\Connection;
use Monolog\Logger;
use Psr\Log\LoggerInterface;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use SicoCreditPlus\Lib\Controller\WSSoapClient;
use SicoCreditPlus\Lib\CreditPlusHelper\AbstractShopLogger;

class SicoCreditPlusLogger extends AbstractShopLogger {

	/** @var LoggerInterface|Logger|SicoCreditPlusPsrLogger */
	protected $oLogger;

	/**
	 * @var SystemConfigService
	 */
	protected $oSystemConfigService;

	/**
	 * @var Connection
	 */
	protected $oConnection;

	public function __construct(LoggerInterface $logger, $oSystemConfigService, $oConnection){
		$this->oLogger = $logger;
		$this->oSystemConfigService = $oSystemConfigService;
		$this->oConnection = $oConnection;
	}

	/**
	 * Write SoapClient Request-Info
	 * to Log-File
	 *
	 * @param string $sToday Timestamp for log file entry
	 * @param bool $bError Whether or not this file is prefixed with error
	 * @param string $sFunctionName Which function has been called
	 * @param string $sContent Content to log
	 */
	public function writeLastRequestInfo( $sToday = '', $bError = false, $sFunctionName = '', $sContent = '' ) {
		$oLogger = $this->oLogger;
		$sContent = $this->cleanUpContent($sContent);
		if ( $bError ) {
			$oLogger->error($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		} else {
			$oLogger->info($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		}
	}

	/**
	 * Write SoapClient Request-Info
	 * to Log-File
	 *
	 * @param string $sToday Timestamp for log file entry
	 * @param bool $bError Whether or not to use error prefix in file name
	 * @param string $sFunctionName Which function was called
	 * @param string $sContent Content to log
	 */
	public function writeLastResponseInfo( $sToday = '', $bError = false, $sFunctionName = '', $sContent = '' ) {
		$oLogger = $this->oLogger;
		$sContent = $this->cleanUpContent($sContent);
		if ( $bError ) {
			$oLogger->error($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		} else {
			$oLogger->info($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		}
	}

	/**
	 * Write Request-Information-Data
	 * to Log-File
	 *
	 * @param string $sToday Timestamp for log file entry
	 * @param string $sFunctionName The called API function name (e.g. getContracts)
	 * @param string[] $aArgumentData The arguments passed to $sFunctionName
	 * @param bool $bError true, if logged event is an error
	 */
	public function writeRequestInformation( $sToday = '', $bError = false, $sFunctionName = '', $aArgumentData = array() ) {
		$oLogger = $this->oLogger;
		// Do a var_export, filter out new lines and replace them with exactly one space character
		if ( isset($aArgumentData) && isset($aArgumentData[1]) && ($aArgumentData[1] instanceof WSSoapClient) ) {
			$aArgumentData[1] = $this->dumpSoapClient($aArgumentData[1]);
		}

		$sArgumentData = var_export($aArgumentData, true);
		//$sArgumentData = var_export($aArgumentData[0],true);
		if( strlen($sArgumentData) > 5000000 ) { return; }
		$sArgumentData = $this->cleanUpContent($sArgumentData);
		$sContent = $sFunctionName.' '.$sArgumentData;

		if ( $bError ) {
			$oLogger->error($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		} else {
			$oLogger->info($sContent, array('function' => $sFunctionName, 'date' => $sToday, 'plugin' => 'SicoCreditPlus'));
		}
	}

	private function dumpSoapClient(WSSoapClient $oSoapClient){
		$ret = [];
		//$ret['Cookies'] = $oSoapClient->__getCookies();
		$ret['lastRequest'] = $oSoapClient->__getLastRequest();
		$ret['lastRequestHeaders'] = $oSoapClient->__getLastRequestHeaders();
		$ret['lastResponse'] = $oSoapClient->__getLastResponse();
		$ret['lastResponseHeaders'] = $oSoapClient->__getLastResponseHeaders();
		return $ret;
	}

	/**
	 * @param bool $bError
	 * @param string $message
	 * @param array $params
	 */
	public function log(bool $bError, string $message,array $params,?string $sSalesChannelId = null){
		$bLogActive = $this->isLogActive($sSalesChannelId);
		$iLogErrorIntoDb = (int)$this->oSystemConfigService->get('SicoCreditPlus.config.iLogErrorIntoDb',$sSalesChannelId);
		if(!$bLogActive){
			return;
		}
		if($bError){
			$this->oLogger->error($message,$params);
			if($iLogErrorIntoDb > 0){
				$sParams = json_encode($params);
				$envelope = [
					'id' => Uuid::randomBytes(),
					'message' => substr($message,0,254),
					//For Values of the log level
					// ./vendor/shopware/platform/src/Administration/Resources/app/administration/src/module/sw-settings-logging/page/sw-settings-logging-list/index.js
					'level' => 400,
					'channel' => 'credit_plus',
					'context' => $sParams,
					'extra' => '[]',
					'updated_at' => null,
					'created_at' => (new \DateTime())->format(Defaults::STORAGE_DATE_TIME_FORMAT),
				];
				try {
					$this->oConnection->insert('log_entry', $envelope);
				}catch(\Throwable $e){
					$envelope['context'] = json_encode([]);
					$envelope['extra'] = json_encode([]);
					$this->oConnection->insert('log_entry', $envelope);
				}
			}
		}
		else{
			$this->oLogger->info($message,$params);
			if($iLogErrorIntoDb > 1){
				$sParams = json_encode($params);
				$envelope = [
					'id' => Uuid::randomBytes(),
					'message' => substr($message,0,254),
					//For Values of the log level
					// ./vendor/shopware/platform/src/Administration/Resources/app/administration/src/module/sw-settings-logging/page/sw-settings-logging-list/index.js
					'level' => 200,
					'channel' => 'credit_plus',
					'context' => $sParams,
					'extra' => '[]',
					'updated_at' => null,
					'created_at' => (new \DateTime())->format(Defaults::STORAGE_DATE_TIME_FORMAT),
				];
				try {
					$this->oConnection->insert('log_entry', $envelope);
				}catch(\Throwable $e){
					$envelope['context'] = json_encode([]);
					$envelope['extra'] = json_encode([]);
					$this->oConnection->insert('log_entry', $envelope);
				}
			}
		}
	}

	/**
	 * @param string|null $sSalesChannelId
	 * @return bool
	 */
	private function isLogActive(?string $sSalesChannelId = null): bool
	{
		return (bool)$this->oSystemConfigService->get('SicoCreditPlus.config.bLogActive', $sSalesChannelId);
	}

	public function debug($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->debug($message, $context);
		}
	}
	public function info($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->info($message, $context);
		}

	}
	public function notice($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->notice($message, $context);
		}
	}
	public function warning($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->warning($message, $context);
		}
	}
	public function alert($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->alert($message, $context);
		}
	}
	public function error($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->error($message, $context);
		}
	}
	public function emergency($message, $context, ?string $sSalesChannelId = null): void
	{
		if ( $this->isLogActive($sSalesChannelId) ) {
			$this->oLogger->emergency($message, $context);
		}
	}
}
