<?php
/**
 * Created by PhpStorm.
 * User: Philipp.Holzmann
 * Date: 24.01.2020
 * Time: 11:01
 */
namespace SicoCreditPlus\Controller;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionDefinition;
use Shopware\Core\Checkout\Order\OrderCollection;
use Shopware\Core\Checkout\Order\OrderDefinition;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Order\SalesChannel\OrderService;
use Shopware\Core\Checkout\Payment\PaymentMethodDefinition;
use Shopware\Core\Checkout\Payment\PaymentMethodEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\RangeFilter;
use Shopware\Core\Framework\Context;
use Shopware\Core\System\SalesChannel\SalesChannelDefinition;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use SicoCreditPlus\Core\Content\CreditPlusPayment\CreditPlusPaymentDefinition;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopContract;
use SicoCreditPlus\Service\CreditPlusPayment;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * @Route(defaults={"_routeScope"={"api"}})
 * Class CpContractController
 */
class CpContractController extends AbstractController{

	protected $creditPlusHelper;
	protected $creditPlusLogger;
	protected $orderService;

	public function __construct(SicoCreditPlusHelper $creditPlusHelper, SicoCreditPlusLogger $logger, OrderService $orderService = null){
		$this->creditPlusHelper = $creditPlusHelper;
		$this->creditPlusLogger = $logger;
		$this->orderService = $orderService;
	}

	protected function getOrder($orderId,$context){
		$orderRepo = $this->container->get(OrderDefinition::ENTITY_NAME . '.repository');
		$oCriteria = new Criteria([$orderId]);
		$oCriteria->addAssociations([
			'transactions',
			'transactions.paymentMethod',
			'price',
			'state'
		]);
		/** @var OrderEntity $orderEntity */
		$orderEntity = $orderRepo->search($oCriteria,$context)->get($orderId);
		return $orderEntity;
	}

	/**
	 * @param Request $request
	 * @param Context $context
	 * @param string $orderId
	 * @return JsonResponse
	 * @Route("/api/sico/contract-data/{orderId}", name="api.action.sico.contract-data",methods={"GET"})
	 */
	public function getContractData(Request $request, Context $context,string $orderId){
		//$this->creditPlusHelper->setSalesChannelContext($cont);
		$this->creditPlusHelper->setContext($context);
		/** @var OrderEntity $orderEntity */
		$orderEntity = $this->getOrder($orderId,$context);

		if($orderEntity->getTransactions()->first()->getPaymentMethod()->getHandlerIdentifier() != CreditPlusPayment::class){
			return new JsonResponse(['order is not a creditPlus order: '.$orderId],404);
		}

		$contract = $this->creditPlusHelper->getContractData($orderEntity->getSalesChannelId(),$orderEntity);
		//Kunde hat bestellt aber die Antragsstrecke nicht durchlaufen.
		if($contract == null){
			$defaultWebshopContract = new WebshopContract();
			return new JsonResponse($defaultWebshopContract);
			//return new JsonResponse(['could not find contract with id: '.$orderId],404);
		}
		$sState = $this->creditPlusHelper->getCreditPlusPaymentState($contract->state,$contract->finallyApproved,$contract->deliveryDone);
		$transaction = $orderEntity->getTransactions()->first();
		if($transaction->getStateMachineState()->getTechnicalName() != $sState){
			$this->creditPlusHelper->setPaymentStatus($orderEntity,$sState);
		}
        $sStatus = $this->creditPlusHelper->getCreditPlusPaymentState($contract->state,$contract->finallyApproved,$contract->deliveryDone);

        $contract->stateText = 'sicoCreditplus.states.'.$sStatus;

		return new JsonResponse($contract);
	}
	/**
	 * @param Request $request
	 * @param Context $context
	 * @param string $orderId
	 * @return JsonResponse
	 * @Route("/api/sico/send-order/{orderId}", name="api.action.sico.send-order",methods={"PATCH"})
	 */
	public function sendOrder(Request $request, Context $context,string $orderId){
		$this->creditPlusHelper->setContext($context);
		$aReturn = array('success' => true);
		$oOrder = $this->getOrder($orderId,$context);
		$aError = $this->creditPlusHelper->sendOrder($oOrder);
		if($aError) {
			return new JsonResponse($aError,200);
		}
		return new JsonResponse($aReturn);
	}

	/**
	 * @param Request $request
	 * @param Context $context
	 * @param string $orderId
	 * @return JsonResponse
	 * @Route("/api/sico/cancel-order/{orderId}", name="api.action.sico.cancel-order",methods={"PATCH"})
	 */
	public function cancelOrder(Request $request, Context $context,string $orderId) {
		$this->creditPlusHelper->setContext($context);
		$aReturn = array('success' => true);
		$oOrder = $this->getOrder($orderId,$context);
		$aError = $this->creditPlusHelper->cancelOrder($oOrder);
		if($aError) {
			return new JsonResponse($aError,200);
		}
		return new JsonResponse($aReturn);
	}

	/**
	 * @param Request $request
	 * @param Context $context
	 * @param string $orderId
	 * @param string $newPaymentMethodId
	 * @return JsonResponse
	 * @Route("/api/sico/change-payment-method-order/{orderId}/{newPaymentMethodId}", name="api.action.sico.change-payment-method-order",methods={"PATCH"})
	 */
	public function changePaymentMethodOrder(Request $request, Context $context, string $orderId, string $newPaymentMethodId) {
		$this->creditPlusHelper->setContext($context);
		$aReturn = array('success' => true);

		$oOrder = $this->getOrder($orderId, $context);
		if ( $aTransactions = $oOrder->getTransactions() ) {
			foreach ( $aTransactions as $oTransaction ) {
				$sTransactionPaymentMethodId = $oTransaction->getPaymentMethodId();
				if ( $sTransactionPaymentMethodId == $newPaymentMethodId ) {
					// Usually another payment method is selected. Break on error though.
					return new JsonResponse(['error' => 'credit-plus-error.payment-unchanged'], 400);
				}
			}
		}
		$this->creditPlusLogger->log(false, 'Order ' . $oOrder->getId() . ' will be cancelled at CreditPlus.', [], $oOrder->getSalesChannelId());
		$aError = $this->creditPlusHelper->cancelOrder($oOrder);
		if($aError && ($aError['sError'] !== 400)  && ($aError['sError'] !== 401) ) {
			return new JsonResponse($aError, 500);
		}
		$oTransactions = $oOrder->getTransactions();

		// Move State Machine to open via changePaymentMethod
		if ( $oTransactions ) {
			/** @var EntityRepository $oTransactionRepo */
			$oTransactionRepo = $this->container->get(OrderTransactionDefinition::ENTITY_NAME.'.repository');
			foreach ( $oTransactions as $oTransaction ) {
				$this->creditPlusLogger->log(false, 'Order ' . $oOrder->getId() . ' payment method will be changed to ' . $newPaymentMethodId . '.', [], $oOrder->getSalesChannelId());
				$oTransactionRepo->update([[
					'id' => $oTransaction->getId(),
					'paymentMethodId' => $newPaymentMethodId
				]], $context);
			}
			try {
				$oParameterBag = new ParameterBag(['stateFieldName' => 'stateId']);
				$this->creditPlusLogger->log(false, 'Order ' . $oOrder->getId() . ' state will be marked as open.', [], $oOrder->getSalesChannelId());
				$this->creditPlusHelper->setOrderStatus($oOrder, 'reopen');
				$this->creditPlusLogger->log(false, 'Order ' . $oOrder->getId() . ' payment state will be marked as open.', [], $oOrder->getSalesChannelId());
				$this->orderService->orderTransactionStateTransition($oTransactions->first()->getId(), 'cp_changePaymentMethod', $oParameterBag, $context);
			} catch ( \Exception $e ) {
				// If the transition fails, it probably is open anyway.
				// Or the user can move it by hand.
				$this->creditPlusLogger->log(true, 'Order ' . $oOrder->getId() . ' could not be marked as open.', [], $oOrder->getSalesChannelId());
			}
		}

		return new JsonResponse($aReturn);
	}

	/**
	 * @param Request $request
	 * @param Context $context
	 * @Route("/api/sico/update-orders/", name="api.action.sico.updateOrders", methods={"PATCH"})
	 */
	public function updateOrders(Request $request, Context $context) {
		$aOrders = $request->get('orders');
		$this->creditPlusHelper->setContext($context);
		/** @var EntityRepository $oPaymentMethodRepo */
		$oPaymentMethodRepo = $this->container->get(PaymentMethodDefinition::ENTITY_NAME.'.repository');
		/** @var PaymentMethodEntity $oPaymentMethod */
		$oPaymentMethod = $oPaymentMethodRepo->search(
			(new Criteria())->addFilter(
				new EqualsFilter('handlerIdentifier', 'SicoCreditPlus\Service\CreditPlusPayment')),
			$context)->first();

        /** @var EntityRepository $oSalesChannelRepo */
        $oSalesChannelRepo = $this->container->get(SalesChannelDefinition::ENTITY_NAME.'.repository');

        $oSalesChannels = $oSalesChannelRepo->search(new Criteria(),$context);

        $oTransactionsIds = [];
        foreach($oSalesChannels as $oSalesChannel){
            if($aOrders == null){
                $this->getOrderFromLastWeek($context,$oSalesChannel,$oPaymentMethod);
            } else {
                $this->processGivenOrders($oSalesChannel,$aOrders,$oPaymentMethod,$context);
            }

        }

		// Ich denke ich muss was zurückgeben. Daher gebe ich ein leeren Response wert zurück
		return new Response();
	}

    private function getOrderFromLastWeek(Context $context,SalesChannelEntity $oSalesChannel,PaymentMethodEntity $oPaymentMethod){
        /** @var EntityRepository $orderRepo */
        $orderRepo = $this->container->get(OrderDefinition::ENTITY_NAME . '.repository');

        $oCriteria = new Criteria();
        $oCriteria->addFilter(new EqualsFilter('transactions.paymentMethodId',$oPaymentMethod->getId()));
        $sixMonthsAgo = new \DateTime('-1 weeks');
        $oCriteria->addFilter(new RangeFilter('orderDate',[RangeFilter::GT => $sixMonthsAgo->format('Y-m-d')]));
        $oCriteria->addFilter(new EqualsFilter('salesChannelId',$oSalesChannel->getId()));
        $oCriteria->addAssociations([
            'transactions',
            'transactions.paymentMethod',
            'price',
            'state']);
        /** @var OrderCollection $oOrderCollections */
        $oOrderCollections = $orderRepo->search($oCriteria,$context)->getEntities();
        $oTransactionsIds = [];
        /** @var OrderEntity $oOrderCollection */
        foreach($oOrderCollections as $oOrderCollection){
            if($oOrderCollection->getTransactions()->first()->getPaymentMethod()->getHandlerIdentifier() != 'SicoCreditPlus\Service\CreditPlusPayment'){
                continue;
            }
            $oTransactionsIds[] = $oOrderCollection->getOrderNumber();
            $aCustomFields = $oOrderCollection->getCustomFields();
            if($aCustomFields){
                $oTransactionsIds[] = $oOrderCollection->getCustomFields()['cpDealerOrderNumber'];
            }
        }
        //return $oTransactionsIds;
        do{
            $oTransactionsIdsPart = array_splice($oTransactionsIds,0,100);
            if(!empty($oTransactionsIdsPart)){
                $oContracts = $this->creditPlusHelper->getContractDataMultiple($oSalesChannel->getId(),$oTransactionsIdsPart);
                foreach($oContracts as $oContract){
                    $sState = $this->creditPlusHelper->getCreditPlusPaymentState($oContract->state,$oContract->finallyApproved,$oContract->deliveryDone);
                    $this->updateOrderCollection($oOrderCollections,$oContract,$sState,$context);
                }
            }
        }while(!empty($oTransactionsIdsPart));
    }
    private function processGivenOrders(SalesChannelEntity $oSalesChannel, $aOrders,PaymentMethodEntity $oPaymentMethod,Context $context){
        $oTransactionsIds = [];
        foreach($aOrders as $aOrder){
            if(!isset($aOrder['transactions'][0])){
                continue;
            }
            if($aOrder['transactions'][0]['paymentMethodId'] !== $oPaymentMethod->getId()){
                continue;
            }
            if($aOrder['salesChannelId'] != $oSalesChannel->getId()){
                continue;
            }
            $oTransactionsIds[] = $aOrder['orderNumber'];
            if($aOrder['customFields']){
                $oTransactionsIds[] = $aOrder['customFields']['cpDealerOrderNumber'];
            }
        }
        do{
            $oTransactionsIdsPart = array_splice($oTransactionsIds,0,100);
            if(!empty($oTransactionsIdsPart)){
                $oContracts = $this->creditPlusHelper->getContractDataMultiple($oSalesChannel->getId(),$oTransactionsIdsPart);
                foreach($oContracts as $oContract){
                    $sState = $this->creditPlusHelper->getCreditPlusPaymentState($oContract->state,$oContract->finallyApproved,$oContract->deliveryDone);
                    $this->updateGivenOrders($oContract,$context,$sState);
                }
            }
        }while(!empty($oTransactionsIdsPart));
    }

    private function updateOrderCollection($oOrderCollections,$oContract,$sState,$context){
        /** @var OrderEntity $oOrder */
        foreach($oOrderCollections as $oOrder){
            $aCustomFields = $oOrder->getCustomFields();
            if($aCustomFields){
                $sOrdernumber = $aCustomFields['cpDealerOrderNumber'];
            } else {
                $sOrdernumber = '';
            }
            if($oContract->dealerOrderNumber == $oOrder->getOrderNumber()
                || $oContract->dealerOrderNumber == $sOrdernumber){
                $transaction = $oOrder->getTransactions()->first();
                //if($oOrder->getOrderNumber() == '21072710062'){ /*$sState = 'creditplus_approved';*//$sState = 'review_necessary';}

                /** @var EntityRepository $oCpPaymentRepo */
                $oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
                $oCpPaymentId = $oCpPaymentRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('orderId',$oOrder->getId())),$context);
                $iCpPaymentId = $oCpPaymentId->firstId();

                if($iCpPaymentId != null){
                    $oCpPaymentRepo->update([[
                        'id' => $iCpPaymentId,
                        'orderFinish' => $oContract->completeOrderUrl
                    ]],$context);
                }


                if($transaction->getStateMachineState()->getTechnicalName() != $sState){
                    $this->creditPlusHelper->setPaymentStatus($oOrder,$sState);
                }
            }
        }
    }

    public function updateGivenOrders($oContract,$context,$sState){
        /** @var EntityRepository $orderRepo */
        $orderRepo = $this->container->get(OrderDefinition::ENTITY_NAME . '.repository');
        $oCriteria = new Criteria();
        $oCriteria->addAssociations([
            'transactions',
            'transactions.paymentMethod',
            'price',
            'state']);
        $oCriteria->addFilter(new EqualsFilter('creditPlusPayment.orderNumber',$oContract->dealerOrderNumber));
        /** @var OrderEntity $oOrderEntity */
        $oOrderEntity = $orderRepo->search($oCriteria,$context)->first();
        if($oOrderEntity){
            $transaction = $oOrderEntity->getTransactions()->first();

            /** @var EntityRepository $oCpPaymentRepo */
            $oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
            $iCpPaymentId = $oCpPaymentRepo->searchIds(
                (new Criteria())->addFilter(new EqualsFilter('orderId',$oOrderEntity->getId()))
                ,$context
            )->firstId();
            if($iCpPaymentId != null) {
                $oCpPaymentRepo->update([[
                    'id' => $iCpPaymentId,
                    'orderFinish' => $oContract->completeOrderUrl
                ]], $context);
            }

            if($transaction->getStateMachineState()->getTechnicalName() != $sState){
                $this->creditPlusHelper->setPaymentStatus($oOrderEntity,$sState);
            }
        }
    }
}
