<?php

namespace SicoCreditPlus\Lib\CreditPlusObjects;
use SicoCreditPlus\Lib\CreditPlusHelper\UrlHandling;

/**
 * Created by PhpStorm.
 * User: sven.keil
 * Date: 16.12.2015
 * Time: 12:50
 */

class ShopIntegrationData extends UrlHandling {

	/**
	 * Main Shop URL
	 * Used for transmission between CreditPlus Service and Customer Server
	 *
	 * @required true
	 * @var string
	 */
	public $sTriggerUrl;

	/**
	 * URL for "inorder" submissions. This URL is called when the data
	 * entry reaches "finished" on CreditPlus and then returns before the identity check.
	 * This URL is prepared in the shop and not modified by CreditPlus.
	 *
	 * @var string
	 */
	protected $sFinishedUrl = '';

	/**
	 * Boolean signalizing a payment is tried before order is made.
	 * Changes behaviour of the payment frontend as to move the confirmation to a later view.
	 *
	 * @var bool
	 */
	protected $bInOrder = false;

	/**
	 * URL to signalize changes to a status of an order.
	 * Usually sTriggerUrl with hardcoded status 3.
	 *
	 * @var string
	 */
	protected $sStatusChangeUrl = '';

	/**
	 * Base Shop URL
	 *
	 * @required false
	 * @var array
	 */
	protected $aData;

	/**
	 * List of the shoppingCart items in the the
	 * remote webshop
	 *
	 * ShoppingCartItem[]
	 *
	 * @required false
	 * @var ShoppingCartItem[]
	 */
	public $aShoppingCart;

	/**
	 * Konstruktor
	 *
	 * Info: baseUrl overrides data
	 *
	 * @param array $aInput
	 * @throws \Exception If input is no ARRAY
	 */
	function __construct( $aInput = array() ) {

		parent::__construct();

		if ( is_array($aInput) ) {
			// Initialize array with input-data

			if ( isset($aInput['triggerUrl']) ) {
				$this->setTriggerUrl($aInput['triggerUrl']);
			} else {
				$this->setTriggerUrl($this->getTriggerUrl());
			}

			if ( isset($aInput['finishedUrl']) ) {
				$this->setFinishedUrl($aInput['finishedUrl']);
			}

			if ( isset($aInput['inorder']) ) {
				$this->setInOrder($aInput['inorder']);
			}

			if ( isset($aInput['statusChangeUrl']) ) {
				$this->setStatusChangeUrl($aInput['statusChangeUrl']);
			}

			if ( isset($aInput['shoppingCart']) ) {
				$this->aShoppingCart = $aInput['shoppingCart'];
			}

		} else {
			throw new \Exception('Wrong input type');
		}
	}

	/**
	 * Check if This Object has valid Data
	 *
	 * @return boolean
	 */
	public function isValid() {

		//baseUrl

		//shoppingCart

		return true;

	}


	/**
	 * returns the Data as Array for API
	 * --> empty array if not valid
	 *
	 */
	public function getApiArray() {

		$aApiData = $this->getDummyArray();
		$aApiData['shoppingCart'] = $this->getShoppingCartApiArray();
		return $aApiData;
	}


	/**
	 *
	 * returns a Dummy-Data Array for API Requests
	 *
	 * @return array
	 */
	public function getDummyArray() {

		$aDummyData = array(
			'baseUrl' => $this->sTriggerUrl,
			'shoppingCart' => array(
				array(
					'amount' => 1,
					'description' => '1L Coca Cola PET Mehrwegflasche',
					'ean' => '5449000017888',
					'listPrice' => 1000.11,
					'promotion' => false,
					'sellingPrice' => 1000.00
				),
				array(
					'amount' => 1,
					'description' => 'Versandkosten',
					'ean' => '2000000000008',
					'listPrice' => 500.55,
					'promotion' => false,
					'sellingPrice' => 500.00
				)
			)
		);

		if ( $this->bInOrder ) {
			$aDummyData['beforeOrderCompletion'] = $this->bInOrder;
			$aDummyData['backToShoppingCartUrl'] = $this->sFinishedUrl;
			$aDummyData['statusChangedNotificationUrl'] = $this->sStatusChangeUrl;
		}

		return $aDummyData;
	}

	/**
	 * Set TriggerURL for continued communication between CreditPlus Service and Customer Server
	 *
	 * If the given URL does not end in ampersand (&),
	 * and does not contain a question mark (?),
	 * a question mark (?) will be added at the end.
	 *
	 * If the given URL does not end in ampersand (&),
	 * but a question mark (?) exists,
	 * an ampersand (&) will be added.
	 *
	 * @param string $sTriggerUrl Trigger URL for system. e.g. http://oxid.creditplus.de/index.php?cl=order&fnc=dispatch&don=CP201601012231& - This will be enriched by CreditPlus with parameters 'callback' and 'signature'
	 * @throws \Exception
	 */
	public function setTriggerUrl( $sTriggerUrl ) {
		if ( strpos($sTriggerUrl, '?') === false ) {
			// No ?, add one
			$sTriggerUrl .= '?';
		} elseif ( substr($sTriggerUrl, -1) === '&' ) {
			// Ends with & - no addon needed, empty string just added for warning suppression
			$sTriggerUrl .= '';
		} else {
			$sTriggerUrl .= '&';
		}
		$this->sTriggerUrl = $sTriggerUrl;
	}

	public function setFinishedUrl( $sFinishedUrl ) {
		$this->sFinishedUrl = $sFinishedUrl;
	}

	public function setStatusChangeUrl( $sStatusChangeUrl ) {
		$this->sStatusChangeUrl = $sStatusChangeUrl;
	}

	public function setInOrder( $bInOrder = false ) {
		$this->bInOrder = $bInOrder;
	}

	/**
	 * @return ShoppingCartItem[]
	 */
	public function getShoppingCart() {
		return $this->aShoppingCart;
	}

	/**
	 * @param ShoppingCartItem[] $aShoppingCart
	 */
	public function setShoppingCart( $aShoppingCart ) {
		$this->aShoppingCart = $aShoppingCart;
	}

	/**
	 * Makes an array of simple value assignments out of an ShoppingCartItem Array using "getApiArray"
	 * @return array
	 * @see ShoppingCartItem::getApiArray()
	 */
	protected function getShoppingCartApiArray() {
		$aCart = $this->getShoppingCart();
		$aReturn = array();
		foreach ( $aCart as $oItem ) {
			$aReturn[] = $oItem->getApiArray();
		}
		return $aReturn;
	}
}
