<?php declare(strict_types=1);

namespace SicoCreditPlus\Migration;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception as DBALException;
use Shopware\Core\Framework\Migration\MigrationStep;
use Shopware\Core\Framework\Uuid\Uuid;

class Migration1717228161PaymentInfoDropPrevention extends MigrationStep
{
    public function getCreationTimestamp(): int
    {
        return 1717728161;
    }

	/**
	 * Adds fields to bind the CreditPlus payment data to the correct order version.
	 * Scans CreditPlus logs to recreate deleted entries.
	 * Adds field to product table.
	 *
	 * @param Connection $connection
	 * @return void
	 * @throws DBALException
	 */
    public function update(Connection $connection): void
    {
		// Add new field to store the version id
		try {
			$connection->executeStatement("
				ALTER TABLE sico_credit_plus_payment ADD COLUMN order_version_id BINARY(16) NULL DEFAULT version_id AFTER order_id;
			");
		} catch (\Exception $e) {
			try {
				// MySQL 5.7 does not allow to add a default value of another column, create a trigger instead
				$connection->executeStatement("
					ALTER TABLE sico_credit_plus_payment ADD COLUMN order_version_id BINARY(16) NULL DEFAULT NULL AFTER order_id;
				");
				$connection->executeStatement("
					CREATE TRIGGER sico_credit_plus_payment_before_insert BEFORE INSERT ON sico_credit_plus_payment FOR EACH ROW
					BEGIN
						IF NEW.order_version_id IS NULL THEN
							SET NEW.order_version_id = NEW.version_id;
						END IF;
					END;
				");
			} catch (\Exception $oInnerException) {
				// Maybe the trigger failed to be created, this is as far as we can go.
				if ( $oInnerException instanceof DBALException ) {
					// This is a Doctrine exception, rethrow it
					throw $oInnerException;
				}
			}
			// If field already exists, ignore
		}
		$connection->executeStatement("
			UPDATE sico_credit_plus_payment LEFT JOIN `order` ON sico_credit_plus_payment.order_id = `order`.id AND sico_credit_plus_payment.version_id = `order`.version_id SET sico_credit_plus_payment.order_version_id = `order`.version_id WHERE NOT (sico_credit_plus_payment.order_id IS NULL);
		");
		// Remove orphans left over from previous copies before foreign key fails
		$connection->executeStatement("
			DELETE FROM sico_credit_plus_payment WHERE order_version_id IS NULL;
		");
		// Switch from single reference field to multi-reference field
		try {
			$connection->executeStatement("
				ALTER TABLE sico_credit_plus_payment DROP FOREIGN KEY `sico_credit_plus_payment_ibfk_1`;
			");
		} catch (\Exception $e) {
			// Ignore if foreign key does not exist
		}
		try {
			$connection->executeStatement("
				ALTER TABLE sico_credit_plus_payment DROP INDEX `sico_credit_plus_payment_ibfk_1`;
			");
		} catch (\Exception $e) {
			// Ignore if index does not exist
			// Might have left through the previous statement
		}
		// This is the new foreign key with version_id
		$connection->executeStatement("
			ALTER TABLE sico_credit_plus_payment ADD FOREIGN KEY `sico_credit_plus_payment_ibfk_1` (order_id, order_version_id) REFERENCES `order` (id, version_id) ON DELETE CASCADE;
		");
		// Funny field required by the Shopware API to address data on foreign tables
		try {
			$connection->executeStatement("
				ALTER TABLE `order` ADD COLUMN creditPlusPayment BINARY(16) NULL DEFAULT id;
			");
		} catch (\Exception $e) {
			try {
				// Same as above: MySQL 5.7 does not support default values from other fields
				$connection->executeStatement("
					ALTER TABLE `order` ADD COLUMN creditPlusPayment BINARY(16) NULL DEFAULT NULL;
				");
				$connection->executeStatement("
					CREATE TRIGGER order_before_insert_creditplus BEFORE INSERT ON `order` FOR EACH ROW
					BEGIN
						IF NEW.creditPlusPayment IS NULL THEN
							SET NEW.creditPlusPayment = NEW.id;
						END IF;
					END;
				");
			} catch (\Exception $oInnerException) {
				// Maybe the trigger failed to be created, this is as far as we can go.
				if ( $oInnerException instanceof DBALException ) {
					throw $oInnerException;
				}
			}
			// Ignore if field already exists
		}
		// Fill the new field with data
		$connection->executeStatement("
			UPDATE `order` SET creditPlusPayment = id WHERE creditPlusPayment IS NULL;
		");

		// Create new half-dummy entries for those we lost with pre-order (first select) and post-order (second select)
		$oRes = $connection->executeQuery("
SELECT o.id, o.version_id, o.order_number, o.custom_fields 
FROM `order` o 
    LEFT JOIN sico_credit_plus_payment scpp ON o.id = scpp.order_id 
WHERE custom_fields LIKE '%cpDealerOrderNumber%' AND scpp.id IS NULL
UNION
SELECT o.id, o.version_id, o.order_number, o.custom_fields
FROM `order` o
	 INNER JOIN order_transaction ot ON o.id = ot.order_id AND o.version_id = ot.order_version_id
	 INNER JOIN payment_method pm ON ot.payment_method_id = pm.id
	 LEFT JOIN sico_credit_plus_payment scpp ON o.id = scpp.order_id
WHERE
    ( NOT (o.custom_fields LIKE '%cpDealerOrderNumber%') ) AND
    pm.handler_identifier = :cpPaymentIdentifier AND
    scpp.id IS NULL
GROUP BY o.id, o.version_id, o.order_number, o.custom_fields
;", ['cpPaymentIdentifier' => 'SicoCreditPlus\\Service\\CreditPlusPayment']);
		if ( $oRes->rowCount() > 0 ) {
			$sInsertStatement = "INSERT INTO sico_credit_plus_payment (id, version_id, order_id, url, gen_timestamp, orderNumber, created_at, updated_at, token, order_finish, back_url, status_change_url) VALUES (:id, :versionId, :orderId, :url, :genTimestamp, :orderNumber, NOW(), NOW(), :token, :orderFinish, :backUrl, :statusChangeUrl);";
			while ( $aRow = $oRes->fetchAssociative() ) {
				$aCustomFields = [];
				if ( $aRow['custom_fields'] ) {
					$aCustomFields = json_decode($aRow['custom_fields'], true);
				}
				if ( !isset($aCustomFields['cpDealerOrderNumber']) ) {
					// Fake post-order as pre-order
					$aCustomFields['cpDealerOrderNumber'] = $aRow['order_number'];
				}
				if ( isset($aCustomFields['cpDealerOrderNumber']) && ($sDealerOrderNumber = $aCustomFields['cpDealerOrderNumber']) ) {
					// Search for URL data in logs, if available
					$sURL = $sOrderFinish = $sBackURL = $sStatusChangeURL = '';
					$sBaseDir = dirname(__FILE__, 6);
					// Find log entry with statusChangedNotificationUrl from creation request (sending side) as var_export
					$sLogFileContent = exec("grep -rC 0 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \"'statusChangedNotificationUrl'\"");
					$aMatches = [];
					if ( preg_match("#'statusChangedNotificationUrl' => '([^']+)'#", $sLogFileContent, $aMatches) ) {
						// The database field is named differently
						$aStatusChangeURL = explode('?', $aMatches[1]);
						$sStatusChangeURL = $aStatusChangeURL[0] ?? '';
						if ( $sStatusChangeURL ) {
							$sStatusChangeURL .= '?';
						}
						$sBackURL = $aMatches[1];
					}
					// Find log entry with customerUrl from creation response (receiving side) as XML
					$sLogFileContent = exec("grep -rC 0 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \":createCreditOfferResponse\" | grep -v \"printDebugCode\"");
					$aMatches = [];
					if ( preg_match('#<customerUrl>([^<]+)</customerUrl>#', $sLogFileContent, $aMatches) ) {
						$sURL = html_entity_decode($aMatches[1]);
						$sOrderFinish = $sURL;
					}
					// Create a new payment entry
					$aNewPaymentData = [
						'id' => Uuid::randomBytes(),
						'versionId' => $aRow['version_id'],
						'orderId' => $aRow['id'],
						'orderVersionId' => $aRow['version_id'],
						'url' => $sURL,
						'genTimestamp' => time(),
						'orderNumber' => $aCustomFields['cpDealerOrderNumber'],
						'token' => 'dummyTokenRebuilt'.substr(md5('SicoCreditPlus'.microtime(true)), 4, 6),
						'orderFinish' => $sOrderFinish,
						'backUrl' => $sBackURL,
						'statusChangeUrl' => $sStatusChangeURL
					];
					$connection->executeStatement($sInsertStatement, $aNewPaymentData);
				}
			}
		}
    }

    public function updateDestructive(Connection $connection): void
    {
        // No destructive updates exists
    }
}
