<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 29.12.2023
 * @time 16:10
 */

namespace SicoCreditPlus\ScheduledTasks;

use DateTime;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\AndFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\OrFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\Currency\CurrencyEntity;
use Shopware\Core\System\SalesChannel\Context\AbstractSalesChannelContextFactory;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use SicoCreditPlus\Core\Content\SicoCreditPlusCalculatedRate\SicoCreditPlusCalculatedRateEntity;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopRateTableMonthRow;

class RateCalculatorHandler extends \Shopware\Core\Framework\MessageQueue\ScheduledTask\ScheduledTaskHandler
{
	/**
	 * @var SalesChannelRepository|null Some kind of "not Entity"-Repository is loaded here
	 */
	protected ?SalesChannelRepository $oProductRepository = null;
	protected ?EntityRepository $oSalesChannelRepository = null;
	protected ?AbstractSalesChannelContextFactory $oSalesChannelContextFactory = null;
	protected ?SicoCreditPlusHelper $oCreditPlusHelper = null;
	protected ?SicoCreditPlusLogger $oCreditPlusLogger = null;
	protected ?EntityRepository $oCalculatedRateRepository = null;

	public function __construct(EntityRepository $scheduledTaskRepository, EntityRepository $oSalesChannelRepository, SalesChannelRepository $oProductRepository, AbstractSalesChannelContextFactory $oSalesChannelContextFactory, SicoCreditPlusLogger $oCreditPlusLogger, SicoCreditPlusHelper $oCreditPlusHelper, EntityRepository $oCalculatedRateRepository)
	{
		parent::__construct($scheduledTaskRepository);
		if ( !$this->oProductRepository ) {
			$this->oProductRepository = $oProductRepository;
		}
		if ( !$this->oSalesChannelRepository ) {
			$this->oSalesChannelRepository = $oSalesChannelRepository;
		}
		if ( !$this->oSalesChannelContextFactory ) {
			$this->oSalesChannelContextFactory = $oSalesChannelContextFactory;
		}
		if ( !$this->oCreditPlusLogger ) {
			$this->oCreditPlusLogger = $oCreditPlusLogger;
		}
		if ( !$this->oCreditPlusHelper ) {
			$oCreditPlusHelper->setCacheFiles(true);
			$this->oCreditPlusHelper = $oCreditPlusHelper;
		}
		if ( !$this->oCalculatedRateRepository ) {
			$this->oCalculatedRateRepository = $oCalculatedRateRepository;
		}
	}

	/**
	 * @inheritDoc
	 */
	public static function getHandledMessages(): iterable
	{
		return [ RateCalculatorTask::class ];
	}

	public function run(): void
	{
		$oContext = Context::createDefaultContext();

		$aSalesChannelContexts = [];
		$oSalesChannelCriteria = new Criteria();
		$oSalesChannelCriteria->addAssociation('type');
		/** @var SalesChannelEntity[] $aSalesChannels */
		$aSalesChannels = $this->oSalesChannelRepository->search($oSalesChannelCriteria, $oContext)->getElements();
		foreach ( $aSalesChannels as $oSalesChannel ) {
			$oType = $oSalesChannel->getType();
			// Safety feature: If type is null, the next line would fail
			$bFeed = $oType === null;
			// Every sales channel with a rocket is assumed to be a feed
			$bFeed = $bFeed || $oType->getIconName() == 'regular-rocket';
			if ( $bFeed ) {
				continue;
			}
			$sSalesChannelID = $oSalesChannel->getId();
			$oSalesChannelContext = $this->oSalesChannelContextFactory->create(Uuid::randomHex(), $sSalesChannelID);
			$this->oCreditPlusHelper->resetWebshopAPI();
			$oWSApi = $this->oCreditPlusHelper->getWebshopAPI($oSalesChannelContext->getSalesChannelId());
			if ( $oWSApi->isApiCallable() ) {
				// Only calculate currently callable contexts
				$aSalesChannelContexts[$sSalesChannelID] = $oSalesChannelContext;
			} else {
				$this->oCreditPlusLogger->error('API is not callable for sales channel "'.$oSalesChannelContext->getSalesChannel()->getName().'". Skipping this sales channel.', [], $sSalesChannelID);
			}
		}
		foreach ( $aSalesChannelContexts as $oSalesChannelContext ) {
			$oProductCriteria = new Criteria();
			$oProductCriteria->addAssociation('tax');
			$oProductCriteria->addAssociation('prices.tax');
			$oProductCriteria->addAssociation('sicoCreditPlusCalculatedRates');
			$oProductCriteria->addAssociation('sicoCreditPlusCalculatedRates.sicoSalesChannel');
			// Only update up to 100 missing per run per sales channel
			$oProductCriteria->addFilter(new OrFilter([
				new NotFilter('AND', [
					new EqualsFilter('sicoCreditPlusCalculatedRates.sicoSalesChannelId', $oSalesChannelContext->getSalesChannelId()),
					new EqualsFilter('sicoCreditPlusCalculatedRates.sicoCalculationFinished', 1)
				]), // Exists for this channel, not yet calculated
				new EqualsFilter('sicoCreditPlusCalculatedRates.sicoCalculationFinished', null), // Does not yet exist at all
				new NotFilter('AND', [
					new EqualsFilter('sicoCreditPlusCalculatedRates.sicoSalesChannelId', $oSalesChannelContext->getSalesChannelId())
				]) // Exists, but not for this channel
			]));
			$oProductCriteria->setLimit(100);

			$this->oCreditPlusHelper->setSalesChannelContext($oSalesChannelContext);
			$oProductsToIndex = $this->oProductRepository->search($oProductCriteria, $oSalesChannelContext);
			/** @var ProductEntity[] $aProductsToIndex */
			$aProductsToIndex = $oProductsToIndex->getElements();
			//$this->oCreditPlusLogger->info('RateCalculatorHandler: Starting rate calculation for sales channel.', ['SalesChannel.id' => $oSalesChannelContext->getSalesChannelId(), 'ProductsToIndex.count' => count($aProductsToIndex)], $oSalesChannelContext->getSalesChannelId());
			$oCurrency = $oSalesChannelContext->getCurrency();
			$aVisitedProducts = [];
			foreach ( $aProductsToIndex as $oProductToIndex ) {
				if ( in_array($oProductToIndex->getId(), $aVisitedProducts) ) {
					continue;
				}
				$aVisitedProducts[] = $oProductToIndex->getId();
				$this->calculateRatesForProduct($oProductToIndex, $oSalesChannelContext, $oCurrency);
			}
		}
	}

	/**
	 * @param ProductEntity $oProductToIndex
	 * @param SalesChannelContext $oSalesChannelContext
	 * @param CurrencyEntity $oCurrency
	 * @return void
	 */
	protected function calculateRatesForProduct(ProductEntity $oProductToIndex, SalesChannelContext $oSalesChannelContext, CurrencyEntity $oCurrency): void
	{
		$oWSApi = $this->oCreditPlusHelper->getWebshopAPI($oSalesChannelContext->getSalesChannelId());
		$oWSCurrency = $this->oCreditPlusHelper->getWebshopCurrency($oCurrency);
		$oCalculatedRate = null;
		/** @var SicoCreditPlusCalculatedRateEntity[] $aCalculatedRates */
		if ( $aCalculatedRates = $oProductToIndex->get('sicoCreditPlusCalculatedRates') ) {
			foreach ( $aCalculatedRates as $oFoundCalculatedRate ) {
				if ( $oFoundCalculatedRate->getSicoSalesChannelId() === $oSalesChannelContext->getSalesChannelId() ) {
					$oCalculatedRate = $oFoundCalculatedRate;
					$this->oCreditPlusLogger->info('Found a calculated rate and updating it.', ['SicoCalculatedRate.id' => bin2hex($oCalculatedRate->getId())], $oSalesChannelContext->getSalesChannelId());
				}
			}
		}
		if ( !$oCalculatedRate ) {
			$oCalculatedRate = new SicoCreditPlusCalculatedRateEntity();
			$oCalculatedRate->setId(Uuid::randomHex());
			$oCalculatedRate->setProductId($oProductToIndex->getId());
			$oCalculatedRate->setSicoProduct($oProductToIndex);
			$oCalculatedRate->setSicoSalesChannelId($oSalesChannelContext->getSalesChannelId());
			$oCalculatedRate->setSicoSalesChannel($oSalesChannelContext->getSalesChannel());
			$this->oCreditPlusLogger->info('Creating a new calculated rate.', ['SicoCalculatedRate.id' => null], $oSalesChannelContext->getSalesChannelId());
		}
		$aFinancingMonths = $this->oCreditPlusHelper->getFinancingMonths($oSalesChannelContext, $oCurrency, $oProductToIndex);
		/** @var WebshopRateTableMonthRow $oShortestMonths */
		$oShortestMonths = null;
		/** @var WebshopRateTableMonthRow $oLongestMonths */
		$oLongestMonths = null;
		/** @var WebshopRateTableMonthRow $oAbsoluteCheapest */
		$oAbsoluteCheapest = null;
		foreach ( $aFinancingMonths as $oMonthRow ) {
			if ( !$oShortestMonths || ($oShortestMonths->months > $oMonthRow->months) ) {
				$oShortestMonths = $oMonthRow;
			}
			if ( !$oLongestMonths || ($oLongestMonths->months < $oMonthRow->months) ) {
				$oLongestMonths = $oMonthRow;
			}
			if ( !$oAbsoluteCheapest || ($oWSApi->retrieveFloatPriceFromFormattedPrice($oAbsoluteCheapest->monthlyRate, $oWSCurrency) > $oWSApi->retrieveFloatPriceFromFormattedPrice($oMonthRow->monthlyRate, $oWSCurrency)) ) {
				$oAbsoluteCheapest = $oMonthRow;
			}
		}
		// Comes one, come all - like to a circus :)
		if ( $oShortestMonths ) {
			$oCalculatedRate->setSicoMinMonths((int)$oShortestMonths->months);
			$oCalculatedRate->setSicoMinMonthsRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oShortestMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoMinMonthsInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oShortestMonths->interestRate, $oWSCurrency));
			$oCalculatedRate->setSicoMaxMonths((int)$oLongestMonths->months);
			$oCalculatedRate->setSicoMaxMonthsRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oLongestMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoMaxMonthsInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oLongestMonths->interestRate, $oWSCurrency));
			$oCalculatedRate->setSicoAbsoluteMinRateMonths((int)$oAbsoluteCheapest->months);
			$oCalculatedRate->setSicoAbsoluteMinRateRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oAbsoluteCheapest->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoAbsoluteMinRateInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oAbsoluteCheapest->interestRate, $oWSCurrency));
		} else {
			$oCalculatedRate->setSicoMinMonths(-1);
			$oCalculatedRate->setSicoMinMonthsRate(-1);
			$oCalculatedRate->setSicoMinMonthsInterestRate(-1);
			$oCalculatedRate->setSicoMaxMonths(-1);
			$oCalculatedRate->setSicoMaxMonthsRate(-1);
			$oCalculatedRate->setSicoMaxMonthsInterestRate(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateMonths(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateRate(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateInterestRate(-1);
		}

		// With the realistic values above, some smaller items may not be listed as "can be financed" in a filter.
		// So these values are there to make them seem like they can be financed. They still have to cost at least 6 cents.
		$aFictionalFinancingMonths = $this->oCreditPlusHelper->getFinancingMonths($oSalesChannelContext, $oCurrency, $oProductToIndex, 0.01);
		/** @var WebshopRateTableMonthRow $oFictionalMonths */
		$oFictionalMonths = null;
		foreach ( $aFictionalFinancingMonths as $oMonthRow ) {
			if ( ($oFictionalMonths === null) || ($oWSApi->retrieveFloatPriceFromFormattedPrice($oFictionalMonths->monthlyRate, $oWSCurrency) > $oWSApi->retrieveFloatPriceFromFormattedPrice($oMonthRow->monthlyRate, $oWSCurrency)) ) {
				$oFictionalMonths = $oMonthRow;
			}
		}
		if ( $oFictionalMonths ) {
			$oCalculatedRate->setSicoFictionalMinRateMonths((int)$oFictionalMonths->months);
			$oCalculatedRate->setSicoFictionalMinRateRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oFictionalMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoFictionalMinRateInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oFictionalMonths->interestRate, $oWSCurrency));
		} else {
			$oCalculatedRate->setSicoFictionalMinRateMonths(-1);
			$oCalculatedRate->setSicoFictionalMinRateRate(-1);
			$oCalculatedRate->setSicoFictionalMinRateInterestRate(-1);
		}
		$oCalculatedRate->setSicoCurrency($oCurrency);
		$oCalculatedRate->setSicoCurrencyId($oCurrency->getId());
		$oCalculatedRate->setSicoCalculationFinished();
		$aUpsertData = [
			'productId' => $oCalculatedRate->getProductId(),
			'sicoSalesChannelId' => $oCalculatedRate->getSicoSalesChannelId(),
			'sicoMinMonths' => $oCalculatedRate->getSicoMinMonths(),
			'sicoMinMonthsInterestRate' => $oCalculatedRate->getSicoMinMonthsInterestRate(),
			'sicoMinMonthsRate' => $oCalculatedRate->getSicoMinMonthsRate(),
			'sicoMaxMonths' => $oCalculatedRate->getSicoMaxMonths(),
			'sicoMaxMonthsInterestRate' => $oCalculatedRate->getSicoMaxMonthsInterestRate(),
			'sicoMaxMonthsRate' => $oCalculatedRate->getSicoMaxMonthsRate(),
			'sicoAbsoluteMinRateMonths' => $oCalculatedRate->getSicoAbsoluteMinRateMonths(),
			'sicoAbsoluteMinRateInterestRate' => $oCalculatedRate->getSicoAbsoluteMinRateInterestRate(),
			'sicoAbsoluteMinRateRate' => $oCalculatedRate->getSicoAbsoluteMinRateRate(),
			'sicoFictionalMinRateMonths' => $oCalculatedRate->getSicoFictionalMinRateMonths(),
			'sicoFictionalMinRateInterestRate' => $oCalculatedRate->getSicoFictionalMinRateInterestRate(),
			'sicoFictionalMinRateRate' => $oCalculatedRate->getSicoFictionalMinRateRate(),
			'sicoCurrencyId' => $oCalculatedRate->getSicoCurrencyId(),
			'sicoCalculationFinished' => $oCalculatedRate->getSicoCalculationFinished(),
			'created_at' => new DateTime(),
			'updated_at' => new DateTime()
		];
		if ( $oCalculatedRate->getId() ) {
			$aUpsertData['id'] = $oCalculatedRate->getId();
		}
		$this->oCalculatedRateRepository->upsert([
			$aUpsertData
		], $oSalesChannelContext->getContext());
	}
}
