<?php declare(strict_types=1);

namespace SicoCreditPlus;

use DateTime;
use DateTimeImmutable;
use Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateSalesChannel\MailTemplateSalesChannelCollection;
use Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateSalesChannel\MailTemplateSalesChannelDefinition;
use Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateType\MailTemplateTypeDefinition;
use Shopware\Core\Content\MailTemplate\MailTemplateCollection;
use Shopware\Core\Content\MailTemplate\MailTemplateDefinition;
use Shopware\Core\Content\Media\Aggregate\MediaFolder\MediaFolderCollection;
use Shopware\Core\Content\Media\Aggregate\MediaFolder\MediaFolderDefinition;
use Shopware\Core\Content\Media\File\MediaFile;
use Shopware\Core\Content\Media\MediaCollection;
use Shopware\Core\Content\Media\MediaDefinition;
use Shopware\Core\Content\Media\MediaEntity;
use Shopware\Core\Content\Media\Thumbnail\ThumbnailService;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Api\Context\AdminApiSource;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin;
use Shopware\Core\Framework\Plugin\Context\ActivateContext;
use Shopware\Core\Framework\Plugin\Context\DeactivateContext;
use Shopware\Core\Framework\Plugin\Context\InstallContext;
use Shopware\Core\Framework\Plugin\Context\UninstallContext;
use Shopware\Core\Framework\Plugin\Context\UpdateContext;
use Shopware\Core\Framework\Plugin\Util\PluginIdProvider;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelDefinition;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use Shopware\Core\System\Snippet\Aggregate\SnippetSet\SnippetSetDefinition;
use Shopware\Core\System\Snippet\SnippetDefinition;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateDefinition;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateEntity;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineTransition\StateMachineTransitionDefinition;
use Shopware\Core\System\StateMachine\StateMachineCollection;
use Shopware\Core\System\StateMachine\StateMachineDefinition;
use Shopware\Core\Content\Media\Pathname\UrlGeneratorInterface;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionCollection;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionDefinition;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupCollection;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupDefinition;
use SicoCreditPlus\Service\CreditPlusPayment;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\Filesystem\Filesystem;

class SicoCreditPlus extends Plugin {

	/** @var string[] md5(index) = value, but it is used more than once, so it's precalculated for performance reasons. */
	public const ID_STATE = [
		'borrowing_request_shown' => 'c998b360c9a6ee34a63ec91ff69705af',
		'creditplus_referred' => '0fa7b9db294d262c0e055dff3a77dadc',
		'creditplus_accepted' => 'd3405bab5924d03b64f5986b0e903fe0',
		'creditplus_approved' => '3a863adfbf23dfd3df218c8f56cff820',
		'creditplus_approved_and_dispatched' => '808fb83d24d8fee2f4a233102c42a7b9',
		'creditplus_processing_payment' => '5b6a24fbf4e9969b7648ffa4b0265502',
		'creditplus_paid' => 'a4327c34479168ab8f98221848889ac5',
		'creditplus_declined_soft' => '1b138f7fb7a356363dd75d3f14b5d855',
		'creditplus_declined_hard' => 'b91d4ad54b32ffd6d2b8e14fd56fed36',
		'creditplus_cancelled' => 'd7e38be944b690a9d41ac4e7210ced68',
		'review_necessary' => 'dcd49c6ec507011aea53b3be895fd446',
		'creditplus_error' => 'ae1012dba2054815d066c839850775a2',
		'creditplus_docs_received' => '6a4543dd18c6498fc54db6c21ef8f910',
		'no_credit_approved' => '147fd35450652b35c76991e5a099bb77',
	];

	/**
	 * @var ThumbnailService
	 * @see SicoCreditPlus::setThumbnailService()
	 */
	private $oThumbnailService;

	/**
	 * @var UrlGeneratorInterface
	 * @see SicoCreditPlus::setUrlGenerator()
	 */
	private $oUrlGenerator;

	/**
	 * @param ThumbnailService $oThumbnailService
	 * @required
	 */
	public function setThumbnailService(ThumbnailService $oThumbnailService){
		$this->oThumbnailService = $oThumbnailService;
	}

	/**
	 * @param UrlGeneratorInterface $oUrlGenerator
	 * @required
	 */
	public function setUrlGenerator(UrlGeneratorInterface $oUrlGenerator){
		$this->oUrlGenerator = $oUrlGenerator;
	}

	public function install(InstallContext $installContext): void
	{
		parent::install($installContext);
		$this->addPaymentMethod($installContext->getContext());
		$this->updateBackendStaticPaymentStatusSnippets($installContext->getContext());
		//$this->deleteMailTemplates($installContext->getContext());
		$this->setMailTemplates($installContext->getContext());
	}

	public function activate(ActivateContext $activateContext): void
	{
		parent::activate($activateContext);

		$this->setPaymentMethodIsActive(true, $activateContext->getContext());
		$this->addDefaultDataToDatabase($activateContext->getContext());
		$this->updateBackendStaticPaymentStatusSnippets($activateContext->getContext());

		$sMessageSenderClass = 'Shopware\Core\Framework\DataAbstractionLayer\Indexing\MessageQueue\IndexerMessageSender';
		$sIndexerClass = 'Shopware\Core\Framework\DataAbstractionLayer\Indexing\Indexer\InheritanceIndexer';
		if ( class_exists($sMessageSenderClass) && class_exists($sIndexerClass) ) {
			$indexerMessasgeSender = $this->container->get($sMessageSenderClass);
			$indexerMessasgeSender->partial(new DateTimeImmutable(),[$sIndexerClass::getName()]);
		} else {
			try {
				/** @var \Shopware\Core\Framework\DataAbstractionLayer\Indexing\EntityIndexerRegistry $registry
				 * @noinspection PhpFullyQualifiedNameUsageInspection
				 */
				$registry = $this->container->get('Shopware\Core\Framework\DataAbstractionLayer\Indexing\EntityIndexerRegistry');
				$registry->sendIndexingMessage(['product.indexer']);
			} catch ( ServiceNotFoundException $oEx ) {
				// Do nothing, this is probably already handled by the Migration script
			}
		}

	}

	public function deactivate(DeactivateContext $deactivateContext): void
	{
		parent::deactivate($deactivateContext); // TODO: Change the autogenerated stub
		$this->setPaymentMethodIsActive(false, $deactivateContext->getContext());

	}

	public function uninstall(UninstallContext $uninstallContext): void
	{
		// Only set the payment method to inactive when uninstalling. Removing the payment method would
		// cause data consistency issues, since the payment method might have been used in several orders
		$this->setPaymentMethodIsActive(false, $uninstallContext->getContext());
		$this->deleteMailTemplates($uninstallContext->getContext());
		if(!$uninstallContext->keepUserData()){
			//ToDO: Eventuell sico_productgroup,sico_offered_option,sico_productgroup_s_articles und sico_credit_plus_payment löschen
		}

	}

	public function postUpdate(UpdateContext $updateContext): void
	{
		parent::postUpdate($updateContext); // TODO: Change the autogenerated stub
		$this->updateBackendStaticPaymentStatusSnippets($updateContext->getContext());
	}


	private function addPaymentMethod(Context $context){
		$paymentMethodExists = $this->getPaymentMethodId($context);

		/** @var PluginIdProvider $pluginIdProvider */
		$pluginIdProvider = $this->container->get(PluginIdProvider::class);
		$pluginId = $pluginIdProvider->getPluginIdByBaseClass(self::class, $context);

		// Payment method exists already, no need to continue here
		if ($paymentMethodExists) {
			/** @var EntityRepository $paymentRepository */
			$paymentRepository = $this->container->get('payment_method.repository');
			$paymentRepository->update([
				['id' => $paymentMethodExists,
					'handlerIdentifier' => CreditPlusPayment::class,
					'name' => 'Creditplus Finanzierung',
					'description' => 'Bezahlen Sie Ihren Einkauf schnell und bequem über monatlich niedrige Raten mit der Creditplus Finanzierung',
					'pluginId' => $pluginId
					]
			],$context);
			return;
		}


		$examplePaymentData = [
			'id' => md5('creditplus'),
			// payment handler will be selected by the identifier
			'handlerIdentifier' => CreditPlusPayment::class,
			'name' => 'Creditplus Finanzierung',
			'description' => 'Bezahlen Sie Ihren Einkauf schnell und bequem über monatlich niedrige Raten mit der Creditplus Finanzierung',
			'pluginId' => $pluginId
		];

		/** @var EntityRepository $paymentRepository */
		$paymentRepository = $this->container->get('payment_method.repository');
		$paymentRepository->create([$examplePaymentData], $context);
	}

	private function addIconToPaymentMethod(Context $context) {

		/** @var EntityRepository $mediaRepo */
		$mediaRepo = $this->container->get(MediaDefinition::ENTITY_NAME.'.repository');

		$oCriteria = new Criteria();
		$oCriteria->addFilter(new EqualsFilter('fileName','creditplus_logo_small'));
		$oCriteria->addAssociations([
			'mediaFolder',
			'mediaFolder.configuration',
			'mediaFolder.configuration.mediaThumbnailSizes'
			]);

		/** @var MediaEntity $mediaEntity */
		$mediaEntity = $mediaRepo->search(
			$oCriteria
			,$context)->first();

		$oMedia = new MediaFile(
			'custom/plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png',
			'image/png',
			'png',
			0
		);//custom/plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png

		if($mediaEntity == null){
			//$sMediaEntityId = $this->oMediaService->saveMediaFile($oMedia,'creditplus_logo_small',$context,'theme');
			$newMediaId = Uuid::randomHex();
			$sFolderId = $this->getMediaDefaultFolderId('theme',$context);
			$sUserId = null;
			if ( ($contextSource = $context->getSource()) && ($contextSource instanceof AdminApiSource) ) {
				$sUserId = $contextSource->getUserId();
			}
			$data =	[
				'id' => $newMediaId,
				'private' => false,
				'fileName' => 'creditplus_logo_small',
				'fileExtension' => $oMedia->getFileExtension(),
				'mimeType' => $oMedia->getMimeType(),
				'fileSize' => $oMedia->getFileSize(),
				'uploadedAt' => new DateTime(),
				'mediaFolderId' => $sFolderId,
				'userId' => $sUserId,
			];
			$context->scope(Context::SYSTEM_SCOPE, function (Context $context) use ($mediaRepo,$data): void {
				$mediaRepo->create([$data], $context);
			});
			$oCriteria = new Criteria([$newMediaId]);
			$oCriteria->addAssociations([
				'mediaFolder',
				'mediaFolder.configuration',
				'mediaFolder.configuration.mediaThumbnailSizes'
			]);
			$oMediaEntity = $mediaRepo->search($oCriteria,$context)->get($newMediaId);
			$this->copyFile($oMedia,$oMediaEntity);
		}
		else{
			$oMediaEntity = $mediaEntity;
			$this->copyFile($oMedia,$oMediaEntity);
		}
		//Hier muss ich den Eintrag nochmal anpassen damit das Bild auch im Frontend angezeigt wird. Weil private = 1 zeigt er nicht an
		$sFolderId = $this->getMediaDefaultFolderId('theme',$context);
		$mediaRepo->update([
			['id' => $oMediaEntity->getId(), 'mediaFolderId' => $sFolderId,'private' => false]
		],$context);

		if($oMediaEntity != null){
			$oMediaCollection = new MediaCollection();
			$oMediaCollection->add($oMediaEntity);
			$this->oThumbnailService->generate($oMediaCollection,$context);
			return $oMediaEntity->getId();
		}
		return null;
	}

	/**
	 * For a unkown reason the file will not be copied to the right place. This is a workaround for this problem.
	 * @param MediaFile $oMediaFile
	 * @param MediaEntity $oMediaEntity
	 */
	private function copyFile(MediaFile $oMediaFile, MediaEntity $oMediaEntity){
		//$stream = fopen($oMediaFile->getFileName(), 'rb');
		$path = $this->oUrlGenerator->getRelativeMediaUrl($oMediaEntity);
		$oSourceFile = $oMediaFile->getFileName();
		$projectDir = $this->container->getParameter('kernel.project_dir');
		$projectDir .= '/';
		//Wenn dieser Ordner Existiert dann befindet sich der aufrufende auf dem root ansonsten müsste er im public sitzen
		if(file_exists($projectDir.'public')){
			$path = $projectDir.'public/'.$path;
			$oSourceFile = $projectDir.trim($oSourceFile,'/');
		} else{
			$oSourceFile = $projectDir.trim($oSourceFile,'/');
		}

		if(!file_exists($path)){
			$pathParts = explode('/',$path);
			array_pop($pathParts);
			$dirPath = implode('/',$pathParts);
			$filesystem = new Filesystem();
			$filesystem->mkdir($dirPath,0777);
			//mkdir($dirPath,0777,true);
			file_put_contents($path,'aaa');//According to the php doc the copy function does not create a file
			copy($oSourceFile,$path);
		}
	}

	private function getMediaDefaultFolderId(string $folder, Context $context): ?string
	{
		/** @var EntityRepository $mediaFolderRepository */
		$mediaFolderRepository = $this->container->get(MediaFolderDefinition::ENTITY_NAME.'.repository');
		$criteria = new Criteria();
		$criteria->addFilter(new EqualsFilter('media_folder.defaultFolder.entity', $folder));
		$criteria->addAssociation('defaultFolder');
		$criteria->setLimit(1);
		/** @var MediaFolderCollection $defaultFolder */
		$defaultFolder = $mediaFolderRepository->search($criteria, $context);
		$defaultFolderId = null;
		if ($defaultFolder->count() === 1) {
			$defaultFolderId = $defaultFolder->first()->getId();
		}

		return $defaultFolderId;
	}

	private function getPaymentMethodId(Context $context): ?string{
		/** @var EntityRepository $paymentRepository */
		$paymentRepository = $this->container->get('payment_method.repository');

		// Fetch ID for update
		$paymentCriteria = (new Criteria())->addFilter(new EqualsFilter('handlerIdentifier', CreditPlusPayment::class));
		$paymentIds = $paymentRepository->searchIds($paymentCriteria, $context);
		if ($paymentIds->getTotal() === 0) {
			return null;
		}
		return $paymentIds->getIds()[0];
	}

	private function setPaymentMethodIsActive(bool $active, Context $context): void
	{
		/** @var EntityRepository $paymentRepository */
		$paymentRepository = $this->container->get('payment_method.repository');
		$paymentMethodId = $this->getPaymentMethodId($context);
		// Payment does not even exist, so nothing to (de-)activate here
		if (!$paymentMethodId) {
			return;
		}
		if($active){
			$sMediaId = $this->addIconToPaymentMethod($context);
		}


		$paymentMethod = [
			'id' => $paymentMethodId,
			'active' => $active
		];
		if(isset($sMediaId)){
			$paymentMethod['mediaId'] = $sMediaId;
		}
		$paymentRepository->update([$paymentMethod], $context);
	}

	private function addDefaultDataToDatabase($context){
		/** @var EntityRepository $sicoProductgroupRepository */
		$sicoProductgroupRepository = $this->container->get(SicoProductgroupDefinition::ENTITY_NAME.'.repository');

		$sProductgroupUUID = Uuid::randomHex();
		if($sicoProductgroupRepository != null){
			/** @var SicoProductgroupCollection $collection */
			$collection = $sicoProductgroupRepository->search(
				new Criteria(),
				$context
			)->getEntities();

			if($collection->count() <= 0){
				$sicoProductgroupRepository->create([
					['id' => $sProductgroupUUID, 'sicoName' => 'Finanzierbare Produkte', 'sicoProductTypeId' => 56, 'sicoProductClassId' => 11]
				],$context);
			}
			else{
				$sProductgroupUUID = $collection->first()->getId();
			}
		}

		/** @var EntityRepository $sicoOfferedOptionRepository */
		$sicoOfferedOptionRepository = $this->container->get(SicoOfferedOptionDefinition::ENTITY_NAME.'.repository');
		if($sicoOfferedOptionRepository != null){
			/** @var SicoOfferedOptionCollection $collection */
			$collection = $sicoOfferedOptionRepository->search(new Criteria(),$context)->getEntities();
			if($collection->count() <= 0){
				$aDefaultMonths = array(
					6 => 4.99,
					8 => 4.99,
					10 => 4.99,
					12 => 4.99,
					18 => 4.99,
					20 => 5.99,
					24 => 5.99,
					30 => 5.99,
					36 => 5.99,
					42 => 5.99,
					48 => 5.99,
					54 => 5.99,
					60 => 5.99,
					66 => 5.99,
					72 => 5.99
				);
				$newData = [];
				foreach($aDefaultMonths as $iMonth => $fRate){
					$newData[] = ['sicoProdgroupId' => $sProductgroupUUID,'sicoMonths' => $iMonth,'sicoInterest' => $fRate,'sicoProdcode' => 'Standard','sicoActive' => true,'sicoRatefactor' => -1];
				}

				$sicoOfferedOptionRepository->create($newData,$context);
			}
		}
	}

	public function setMailTemplates(Context $context){
		/** @var EntityRepository $oSalesChannelRepo */
		$oSalesChannelRepo = $this->container->get(SalesChannelDefinition::ENTITY_NAME.'.repository');
		/** @var SalesChannelEntity $oSalesChannelEntity */
		$oSalesChannelEntity = $oSalesChannelRepo->search(
			(new Criteria())->addFilter(
				new EqualsFilter('typeId',Defaults::SALES_CHANNEL_TYPE_STOREFRONT)
			),$context)->first();
		$statusMails = [
			'sCreditPlusState20',
			'sCreditPlusState24n',
			'sCreditPlusState24a',
			'sCreditPlusState25',
			'sCreditPlusState92',
			'sCreditPlusState95',
			'sCreditPlusState99',
		];
		$mails = [];

		$plain = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_plain.html.twig');
		$html = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_html.html.twig');
		$mails[] = [
			'id' => md5('sCreditplusOrderAdditionalMessage'),
			'name' => 'sCreditplusOrderAdditionalMessage',
			'mailTemplateType' => [
				'id' => md5('sCreditplusOrderAdditionalMessage_type'),
				'technicalName' => 'sCreditplusOrderAdditionalMessage',
				'name' => 'sCreditplusOrderAdditionalMessage',
				'availableEntities' => ['order' => 'order','sTargetURL' => 'string']
			],
			'systemDefault' => false,
			'senderName' => '{{ salesChannel.name }}',
			'description' => 'Bestellmail für CreditPlus',
			'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
			'contentHtml' => $html,
			'contentPlain' => $plain
		];

		$plain = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_20_plain.html.twig');
		$html = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_20_html.html.twig');
		$mails[] = [
			'id' => md5('sCreditplusOrderAdditionalMessage20'),
			'name' => 'sCreditplusOrderAdditionalMessage20',
			'mailTemplateType' => [
				'id' => md5('sCreditplusOrderAdditionalMessage_type'),
				'technicalName' => 'sCreditplusOrderAdditionalMessage',
				'name' => 'sCreditplusOrderAdditionalMessage',
				'availableEntities' => ['order' => 'order','sTargetURL' => 'string']
			],
			'systemDefault' => false,
			'senderName' => '{{ salesChannel.name }}',
			'description' => 'Bestellmail für CreditPlus',
			'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
			'contentHtml' => $html,
			'contentPlain' => $plain
		];

		$plain = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditPlusState_plain.html.twig');
		$html = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditPlusState_html.html.twig');
		foreach($statusMails as $statusMail){
			$mails[] = [
				'id' => md5($statusMail),
				'name' => $statusMail,
				'mailTemplateType' => [
					'id' => md5($statusMail.'_type'),
					'technicalName' => $statusMail,
					'name' => $statusMail,
					'availableEntities' => ['order' => 'order']
				],
				'systemDefault' => false,
				'senderName' => '{{ salesChannel.name }}',
				'description' => 'Statusänderung der CreditPlus Bestellung',
				'subject' => 'Status der Bestellung im {{ salesChannel.name }}',
				//'availableEntities' => ['order' => 'order'],//'{"order":"order"}',
				'contentHtml' => $html,
				'contentPlain' => $plain
			];
		}
		$plain = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditPlusRetryErrors_plain.html.twig');
		$html = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditPlusRetryErrors_html.html.twig');
		$mails[] = [
			'id' => md5('sCreditPlusRetryErrors'),
			'name' => 'sCreditPlusRetryErrors',
			'mailTemplateType' => [
				'id' => md5('sCreditPlusRetryErrors_type'),
				'technicalName' => 'sCreditPlusRetryErrors',
				'name' => 'sCreditPlusRetryErrors',
				'availableEntities' => ['order' => 'order']
			],
			'systemDefault' => false,
			'senderName' => '{{ salesChannel.name }}',
			'description' => 'Fehler bei CreditPlus',
			'subject' => 'Fehlerhafte CreditPlus Status Updates in {{ salesChannel.name }}',
			'contentHtml' => $html,
			'contentPlain' => $plain
		];

		/** @var EntityRepository $mailTemplateRepository */
		$mailTemplateRepository = $this->container->get(MailTemplateDefinition::ENTITY_NAME.'.repository');
		$mailTemplateRepository->upsert($mails,$context);

		$ids = [md5('sORDERForCreditPlus')];
		$ids[] = md5('sCreditPlusRetryErrors');
		foreach($statusMails as $statusMail){ $ids[] = md5($statusMail); }

		$criteria = new Criteria($ids);
		$criteria->addAssociation('salesChannels');
		/** @var MailTemplateCollection $oMailtemplates */
		$oMailtemplates = $mailTemplateRepository->search($criteria,$context);
		foreach($oMailtemplates as $oMailtemplate){

			if(method_exists($oMailtemplate,'getSalesChannels')){
				foreach($oMailtemplate->getSalesChannels() as $salesChannel){
					if($salesChannel->getId() == $oSalesChannelEntity->getId()){
						continue 2;
					}
				}
			}

			$updateSalesChannel = [
				'id' => $oMailtemplate->getId(),
				'salesChannels' => [[
					'mailTemplateId' => $oMailtemplate->getId(),
					'mailTemplateTypeId' => $oMailtemplate->getMailTemplateTypeId(),
					'salesChannelId' => $oSalesChannelEntity->getId()
				]]
			];

			$mailTemplateRepository->upsert([$updateSalesChannel],$context);
		}
	}

	public function deleteMailTemplates($context){
		/** @var EntityRepository $oMailTemplateTypeRepo */
		$oMailTemplateTypeRepo = $this->container->get(MailTemplateTypeDefinition::ENTITY_NAME.'.repository');
		/** @var EntityRepository $oMailTemplateRepo */
		$oMailTemplateRepo = $this->container->get(MailTemplateDefinition::ENTITY_NAME.'.repository');

		if(class_exists('Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateSalesChannel')){
			/** @var EntityRepository $oMailTemplateSalesChannelRepo */
			$oMailTemplateSalesChannelRepo = $this->container->get(MailTemplateSalesChannelDefinition::ENTITY_NAME.'.repository');
		} else {
			$oMailTemplateSalesChannelRepo = null;
		}


		$statusMails = [
			'sCreditPlusState20',
			'sCreditPlusState24n',
			'sCreditPlusState24a',
			'sCreditPlusState25',
			'sCreditPlusState92',
			'sCreditPlusState95',
			'sCreditPlusState99',
		];
		$idsMailTemplates = [];
		$idsMailTemplates[] = ['id' => md5('sORDERForCreditPlus')];
		$idsMailTemplatesTypes = [];
		$idsMailTemplatesTypes[] = ['id' => md5('sORDERForCreditPlus_type')];

		$idsMailTemplates[] = ['id' => md5('sCreditPlusRetryErrors')];
		$idsMailTemplatesTypes[] = ['id' => md5('sCreditPlusRetryErrors_type')];

		$idsMailTemplates[] = ['id' => md5('sCreditplusOrderAdditionalMessage')];
		$idsMailTemplatesTypes[] = ['id' => md5('sCreditplusOrderAdditionalMessage_type')];
		foreach($statusMails as $statusMail){
			$idsMailTemplates[] = ['id' => md5($statusMail)];
			$idsMailTemplatesTypes[] = ['id' => md5($statusMail.'_type')];
		}
		$aMailTemplatesId = [];
		foreach($idsMailTemplates as $idsMailTemplate){
			$aMailTemplatesId[] = $idsMailTemplate['id'];
		}

		if($oMailTemplateSalesChannelRepo){
			/** @var MailTemplateSalesChannelCollection $oMailTemplateSalesChannelCollection */
			$oMailTemplateSalesChannelCollection = $oMailTemplateSalesChannelRepo->search(
				(new Criteria())->addFilter(
					new EqualsAnyFilter('mailTemplateId',$aMailTemplatesId)
				),$context);
			$toDeleteMtScIds = [];

			if ($oMailTemplateSalesChannelCollection->count() > 0){
				foreach($oMailTemplateSalesChannelCollection as $value){
					$toDeleteMtScIds[] = ['id' => $value->getId()];
				}
				$oMailTemplateSalesChannelRepo->delete($toDeleteMtScIds,$context);
			}
		}


		$oMailTemplateRepo->delete($idsMailTemplates,$context);

		$oMailTemplateTypeRepo->delete($idsMailTemplatesTypes,$context);
	}

	public function updateBackendStaticPaymentStatusSnippets(Context $context){
		/** @var EntityRepository $stateMachineRepo */
		$stateMachineRepo = $this->container->get(StateMachineDefinition::ENTITY_NAME.'.repository');
		/** @var StateMachineCollection $stateMachineEntites */
		$stateMachineEntites = $stateMachineRepo->search((new Criteria())->addFilter(new EqualsFilter('technicalName','order_transaction.state')),$context)->getEntities();
		$sStateMachineEntitesId = $stateMachineEntites->first()->getId();
		// Sorted like self::ID_STATE
		$aBackendTranslations = array(
			'borrowing_request_shown' => 'Antragsformular liegt dem Kunden vor',
			'creditplus_referred' => 'Antrag ausgefüllt',
			'creditplus_accepted' => 'Antrag angenommen',
			'creditplus_approved' => 'Lieferfreigabe',
			'creditplus_approved_and_dispatched' => 'Ausgeliefert',
			'creditplus_processing_payment' => 'Auszahlung',
			'creditplus_paid' => 'Ausgezahlt',
			'creditplus_declined_soft' => 'Weich abgelehnt',
			'creditplus_declined_hard' => 'Abgelehnt',
			'creditplus_cancelled' => 'Storniert',
			'review_necessary' => 'Überprüfung notwendig',
			'creditplus_error' => 'Fehler',
			'creditplus_docs_received' => 'Warten auf Unterlagen',
			'no_credit_approved' => 'Es wurde kein Kredit genehmigt.',
		);
		$aNewStates = [];//order_transaction.state
		foreach($aBackendTranslations as $name => $sDefault){
			$id = self::ID_STATE[$name];
			$aNewStates[] = ['id' => $id, 'name' => $sDefault, 'technicalName' => $name, 'stateMachineId' => $sStateMachineEntitesId];
		}
		/** @var EntityRepository $stateRepo */
		$stateRepo = $this->container->get(StateMachineStateDefinition::ENTITY_NAME.'.repository');
		$stateRepo->upsert($aNewStates,$context);
		/** @var StateMachineStateEntity $oOpenState */
		$oOpenState = $stateRepo->search((new Criteria())->addFilter(new EqualsFilter('technicalName','open'),new EqualsFilter('stateMachineId',$sStateMachineEntitesId)),$context)->getEntities()->first();

		$aStateTransitions = [
			//Transitions für die Normale durchführung
			['id' => md5('open-borrowing_request_shown'),										'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['borrowing_request_shown'], 'actionName' => 'cp_borrowing_request_shown', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('borrowing_request_shown-creditplus_referred'),							'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_accepted'),								'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_approved'),								'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-creditplus_approved_and_dispatched'),				'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved_and_dispatched-creditplus_processing_payment'),	'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_processing_payment-creditplus_paid'),						'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
			//Transitions für die Abgelehnt States

			//Zuerst Soft
			['id' => md5('borrowing_request_shown-creditplus_declined_soft'),			'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_declined_soft'),				'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_declined_soft'),				'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-creditplus_declined_soft'),				'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved_and_dispatched-creditplus_declined_soft'),	'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_declined_soft'),					'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],

			//Dann Hart
			['id' => md5('borrowing_request_shown-creditplus_declined_hard'),			'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_declined_hard'),				'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_declined_hard'),				'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-creditplus_declined_hard'),				'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved_and_dispatched-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_declined_hard'),					'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],

			//Transitions vom Ausgangspunkt open, welcher bei "Vor Bestellabschluss" massiv auftritt
			['id' => md5('open-creditplus_referred'),					'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_accepted'),					'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_approved'),					'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_approved_and_dispatched'),		'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_processing_payment'),			'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_paid'),						'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_declined_soft'),				'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_declined_hard'),				'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-creditplus_cancelled'),					'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('open-review_necessary'),						'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],

			//Restliche Transitions vom Ausgangspunkt borrowing_request_shown
			['id' => md5('borrowing_request_shown-creditplus_accepted'),					'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('borrowing_request_shown-creditplus_approved'),					'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('borrowing_request_shown-creditplus_approved_and_dispatched'),	'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('borrowing_request_shown-creditplus_processing_payment'),		'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('borrowing_request_shown-creditplus_paid'),						'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

			//Restliche Transitions vom Ausgangspunkt creditplus_referred
			['id' => md5('creditplus_referred-creditplus_approved'),					'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_approved_and_dispatched'),	'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_processing_payment'),		'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_paid'),						'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

			//Restliche Transitions vom Ausgangspunkt creditplus_accepted
			['id' => md5('creditplus_accepted-creditplus_approved_and_dispatched'),	'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_processing_payment'),		'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_paid'),						'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

			//Restliche Transitions vom Ausgangspunkt creditplus_approved
			['id' => md5('creditplus_approved-creditplus_processing_payment'),	'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-creditplus_paid'),					'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

			//Restliche Transitions vom Ausgangspunkt creditplus_approved_and_dispatched
			['id' => md5('creditplus_approved_and_dispatched-creditplus_paid'),	'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

			//Transitions zum Storniert State
			['id' => md5('borrowing_request_shown-creditplus_cancelled'),			'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-creditplus_cancelled'),				'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-creditplus_cancelled'),				'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-creditplus_cancelled'),				'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved_and_dispatched-creditplus_cancelled'),	'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_processing_payment-creditplus_cancelled'),		'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],

			//Transitions zur review_necessary State
			['id' => md5('borrowing_request_shown-review_necessary'),			'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-review_necessary'),				'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-review_necessary'),				'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-review_necessary'),				'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved_and_dispatched-review_necessary'),	'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_processing_payment-review_necessary'),		'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_paid-review_necessary'),					'fromStateId' => self::ID_STATE['creditplus_paid'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_declined_soft-review_necessary'),			'fromStateId' => self::ID_STATE['creditplus_declined_soft'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_declined_hard-review_necessary'),			'fromStateId' => self::ID_STATE['creditplus_declined_hard'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_cancelled-review_necessary'),				'fromStateId' => self::ID_STATE['creditplus_cancelled'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],

			//Transitions von review_necessary State
			['id' => md5('review_necessary-open'),								'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-borrowing_request_shown'),			'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['borrowing_request_shown'], 'actionName' => 'cp_borrowing_request_shown', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_referred'),				'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_accepted'),				'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_approved'),				'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_approved_and_dispatched'),	'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_processing_payment'),		'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_paid'),					'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('review_necessary-creditplus_cancelled'),				'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],

			// Transitions zurück zum Anfang für "Zahlungsartwechsel"
			['id' => md5('borrowing_request_shown-open'),	'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_referred-open'),		'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_accepted-open'),		'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_approved-open'),		'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_declined_soft-open'),	'fromStateId' => self::ID_STATE['creditplus_declined_soft'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_declined_hard-open'),	'fromStateId' => self::ID_STATE['creditplus_declined_hard'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
			['id' => md5('creditplus_cancelled-open'),		'fromStateId' => self::ID_STATE['creditplus_cancelled'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
		];
		/** @var EntityRepository $oStateTransition */
		$oStateTransition = $this->container->get(StateMachineTransitionDefinition::ENTITY_NAME.'.repository');
		$oStateTransition->upsert($aStateTransitions,$context);

	}

	public function update(UpdateContext $updateContext): void {
		$additionalMessage20Id = md5('sCreditplusOrderAdditionalMessage20');
		/** @var EntityRepository $mailTemplateRepository */
		$mailTemplateRepository = $this->container->get(MailTemplateDefinition::ENTITY_NAME.'.repository');
		$oMailsCollection = $mailTemplateRepository->searchIds(new Criteria([$additionalMessage20Id]),$updateContext->getContext());
		$iTotalFound = $oMailsCollection->getTotal();
		if($iTotalFound < 1){
			$plain = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_20_plain.html.twig');
			$html = file_get_contents(dirname(__FILE__).'/DefaultMails/sCreditplusOrderAdditionalMessage_20_html.html.twig');
			$object = [
				'id' => md5('sCreditplusOrderAdditionalMessage20'),
				'name' => 'sCreditplusOrderAdditionalMessage20',
				'mailTemplateTypeId' => md5('sCreditplusOrderAdditionalMessage_type'),
				'systemDefault' => false,
				'senderName' => '{{ salesChannel.name }}',
				'description' => 'Bestellmail für CreditPlus',
				'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
				'contentHtml' => $html,
				'contentPlain' => $plain
			];
			/** @var EntityRepository $mailTemplateRepository */
			$mailTemplateRepository->create([$object],$updateContext->getContext());
		}
		/** @var EntityRepository $snippetRepo */
		$snippetRepo = $this->container->get(SnippetDefinition::ENTITY_NAME.'.repository');
		$snippetCollection = $snippetRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('translationKey','sicoCreditplus.error.creditplusNotAvailable')),$updateContext->getContext());
		if($snippetCollection->getTotal() < 0){
			/** @var EntityRepository $snippetSetRepo */
			$snippetSetRepo = $this->container->get(SnippetSetDefinition::ENTITY_NAME.'.repository');
			$snippetSetColl = $snippetSetRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('iso','de-DE')),$updateContext->getContext());
			$snippetSetId = $snippetSetColl->firstId();
			$obj = [
				'snippet_set_id' => $snippetSetId,
				'translationKey' => 'sicoCreditplus.error.creditplusNotAvailable',
				'value' => 'Leider können wir Ihrer Finanzierungsanfrage nicht entsprechen. Die Entscheidung über Ihre Anfrage basiert auf einer automatisierten Verarbeitung Ihrer personenbezogenen Daten, die der Bewertung einzelner Persönlichkeitsmerkmale dienen. Bitte wählen Sie eine andere Zahlungsart aus.',
				'author' => 'creditPlus'
			];
			$snippetRepo->create([$obj],$updateContext->getContext());
		}
	}
}
