<?php
namespace SicoCreditPlus\Subscriber;
use Shopware\Core\Content\Product\Events\ProductListingCollectFilterEvent;
use Shopware\Core\Content\Product\Events\ProductListingCriteriaEvent;
use Shopware\Core\Content\Product\ProductDefinition;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\Listing\Filter;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Shopware\Core\Content\ProductExport\Event\ProductExportProductCriteriaEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Aggregation\Metric\MaxAggregation;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\AndFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\RangeFilter;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelEntitySearchResultLoadedEvent;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Core\Content\SicoCreditPlusCalculatedRate\SicoCreditPlusCalculatedRateCollection;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use \Shopware\Core\Content\Product\Events\ProductListingResultEvent;
class ProductListingSubscriber implements EventSubscriberInterface{

    /** @var SystemConfigService  */
    private $systemConfigService;

    /** @var SicoCreditPlusHelper */
    private $creditPlusHelper;

    /**
     * @param SystemConfigService $systemConfigService
	 * @param SicoCreditPlusHelper $creditPlusHelper
     */
    public function __construct(SystemConfigService $systemConfigService, SicoCreditPlusHelper $creditPlusHelper)
    {
        $this->systemConfigService = $systemConfigService;
        $this->creditPlusHelper = $creditPlusHelper;
    }


	/**
	 * @return string[] Array of 'Event class' => 'subscriber function'
	 */
    public static function getSubscribedEvents(): array
    {
        return [
			ProductExportProductCriteriaEvent::class => 'onProductExportProductCriteria',
			ProductListingCriteriaEvent::class => 'onProductCriteria',
			ProductListingCollectFilterEvent::class => 'onProductCollectFilter',
            ProductListingResultEvent::class => 'onProductListing',
            SalesChannelEntitySearchResultLoadedEvent::class => 'onProductSearchResultLoadedEvent'
        ];
    }

	public function onProductExportProductCriteria(ProductExportProductCriteriaEvent $oEvent): void
	{
		$oCriteria = $oEvent->getCriteria();
		$oCriteria->addAssociation('sicoCreditPlusCalculatedRates'); // Just add the association
	}

	public function onProductCriteria(ProductListingCriteriaEvent $oEvent): void
	{
		$oCriteria = $oEvent->getCriteria();
		$oCriteria->addAssociation('sicoCreditPlusCalculatedRates'); // Just add the association
	}

	public function onProductCollectFilter(ProductListingCollectFilterEvent $event): void
	{
		$oSalesChannelContext = $event->getSalesChannelContext();
		$sSalesChannelID = $oSalesChannelContext->getSalesChannelId();
		$bShowListFilterFrom = $this->systemConfigService->get('SicoCreditPlus.config.bShowListFilterFrom', $sSalesChannelID);
		$bShowListFilterPossible = $this->systemConfigService->get('SicoCreditPlus.config.bShowListFilterPossible', $sSalesChannelID);
		$oFilterCollection = $event->getFilters();
		$oRequest = $event->getRequest();
		if ( $bShowListFilterFrom ) {
			// Show only if filter is wanted
			$aMaxRate = $oRequest->get('aSicoRates');
			if ( $aMaxRate ) {
				$dMaxRate = isset($aMaxRate['maxRate']) ? (float)$aMaxRate['maxRate'] : null;
				$dMinRate = isset($aMaxRate['minRate']) ? (float)$aMaxRate['minRate'] : 0.0;
			} else {
				$dMaxRate = null;
				$dMinRate = 0.0;
			}
			$bFilterActive = (bool)$dMaxRate;
			$aFilterData = [
				RangeFilter::GTE => $dMinRate
			];
			if ( $dMaxRate ) {
				$aFilterData[RangeFilter::LTE] = $dMaxRate;
			}
			$oMultiFilter = new AndFilter([
				new EqualsFilter('sicoCreditPlusCalculatedRates.sicoSalesChannelId', $sSalesChannelID),
				new EqualsFilter('sicoCreditPlusCalculatedRates.sicoCalculationFinished', 1),
				new RangeFilter('sicoCreditPlusCalculatedRates.sicoAbsoluteMinRateRate', $aFilterData),
			]);
			$oFilter = new Filter(
				'dSicoMaxRate',
				$bFilterActive,
				[
					new MaxAggregation('minRateExists', 'sicoCreditPlusCalculatedRates.sicoAbsoluteMinRateRate')
				],
				$oMultiFilter,
				[0,1000] //TODO: tbd max filter rate
			);
			$oFilterCollection->add($oFilter);
		}
		if ( $bShowListFilterPossible ) {
			$bFilterSet = (bool)$oRequest->get('bSicoFinancingPossible');
			$aFilterData = [
				RangeFilter::GTE => 0.01
			];
			$oMultiFilter = new AndFilter([
				new EqualsFilter('sicoCreditPlusCalculatedRates.sicoSalesChannelId', $sSalesChannelID),
				new EqualsFilter('sicoCreditPlusCalculatedRates.sicoCalculationFinished', 1),
				new RangeFilter('sicoCreditPlusCalculatedRates.sicoFictionalMinRateRate', $aFilterData),
			]);
			$oFilter = new Filter(
				'bSicoFinancingPossible',
				$bFilterSet,
				[
					new MaxAggregation('minFictionalRateExists', 'sicoCreditPlusCalculatedRates.sicoFictionalMinRateRate')
				],
				$oMultiFilter,
				[0,1000] //TODO: tbd max filter rate
			);
			$oFilterCollection->add($oFilter);
		}
	}

    public function onProductSearchResultLoadedEvent(SalesChannelEntitySearchResultLoadedEvent $event): void {
        if($event->getName() != 'sales_channel.'.ProductDefinition::ENTITY_NAME.'.search.result.loaded'){
            return;
        }
        $this->creditPlusHelper->setSalesChannelContext($event->getSalesChannelContext());
        $this->enrichProducts($event->getResult(),$event->getSalesChannelContext());
    }

    public function onProductListing(ProductListingResultEvent $event): void {
        $this->creditPlusHelper->setSalesChannelContext($event->getSalesChannelContext());
        $this->enrichProducts($event->getResult(),$event->getSalesChannelContext());
    }

    protected function enrichProducts(ProductListingResult|EntitySearchResult $result, SalesChannelContext $salesChannelContext): void {
        $bShowCategory = $this->systemConfigService->get('SicoCreditPlus.config.bShowCategory',$salesChannelContext->getSalesChannelId());
        if($bShowCategory){
            /** @var ProductEntity $oProduct */
            foreach($result as $oProduct){
                $aCustomFields = $oProduct->getCustomFields();
                if($aCustomFields === null){
                    $aCustomFields = [];
                }
                // This function could be processed more than once, so I check to save processing time
                if(isset($aCustomFields['SicoCreditPlus'])){
                    continue;
                }
				// Use the cache instead of recalculation for listings
				/** @var SicoCreditPlusCalculatedRateCollection $oSicoCreditPlusCalculatedRates */
				if ( $oProduct->has('sicoCreditPlusCalculatedRates') && ($oSicoCreditPlusCalculatedRates = $oProduct->get('sicoCreditPlusCalculatedRates')) ) {
					foreach ( $oSicoCreditPlusCalculatedRates as $oSicoCreditPlusCalculatedRate ) {
						if ( $oSicoCreditPlusCalculatedRate->getSicoSalesChannelId() !== $salesChannelContext->getSalesChannelId() ) {
							// Skip foreign calculated rows
							continue;
						}
						if ( $oSicoCreditPlusCalculatedRate->getSicoCalculationFinished() !== 1 ) {
							// Calculation outstanding is skipped, too
							continue;
						}
						$dMinRate = $oSicoCreditPlusCalculatedRate->getSicoAbsoluteMinRateRate();
						$dMinRateMonths = $oSicoCreditPlusCalculatedRate->getSicoAbsoluteMinRateMonths();
						$dMinInterestRate = $oSicoCreditPlusCalculatedRate->getSicoAbsoluteMinRateInterestRate();
						$bHasFinancing = $dMinRate > 0;
						$aCustomFields['SicoCreditPlus'] = [
							'hasFinancing' => $bHasFinancing,
							'cheapestRate' => $dMinRate,
							'cheapestMonths' => $dMinRateMonths,
							'cheapestInterestRate' => $dMinInterestRate
						];
					}
					$oProduct->setCustomFields($aCustomFields);
				}
				if(isset($aCustomFields['SicoCreditPlus'])){
					// If calculation has finished, take that value
					continue;
				}
				// Fallback to slow calculation, if cached calculation has not run yet
                $oPrice = $oProduct->getPrice();
                if($oPrice){
                    $dPrice = $oPrice->getCurrencyPrice($salesChannelContext->getCurrencyId())->getGross();//$this->reFloatPrice($oPrice->getCurrencyPrice());
                    //Hier dürfen keine Monate zurückgegeben werden, wenn der Kredit nur auf ein oder mehrere Produkte beschränkt ist.
                    $aMonths = $this->creditPlusHelper->getFinancingMonths($salesChannelContext,$salesChannelContext->getCurrency(), $oProduct,null,$dPrice);
                }
                else {
                    $aMonths = $this->creditPlusHelper->getFinancingMonths($salesChannelContext,$salesChannelContext->getCurrency(), $oProduct);
                }
                $dCheapestInterestRate = null;
                $oCheapestMonth = null;
                if(!empty($aMonths)){
                    $dCheapestInterestRate = $this->creditPlusHelper->getCheapestInterestRate($aMonths);
                    $oCheapestMonth = $this->creditPlusHelper->getCheapestRate($aMonths);
                }

                $aCustomFields['SicoCreditPlus'] = [
                    'hasFinancing' => !empty($aMonths),
                    'cheapestInterestRate' => $dCheapestInterestRate,
                    'cheapestMonth' => $oCheapestMonth
                ];
                $oProduct->setCustomFields($aCustomFields);
            }
        }
    }
 }
