<?php
/**
 * This file is part of the SinkaCom CreditPlus Module Package.
 *
 * @link      http://www.sinkacom.de/
 * @copyright (C) SinkaCom AG 2015-2019
 * @version   OXID eShop CE
 */
/**
 * Created by PhpStorm.
 * @author Mihovil.Bubnjar
 * @date 03.01.2019
 * @time 15:20
 */
namespace Sinkacom\CreditPlusModule\Component;
use OxidEsales\Eshop\Application\Model as ESModel;
use OxidEsales\Eshop\Core as ESCore;
use Sinkacom\CreditPlusModule\Lib\Controller\CreditPlusWebshopAPI;
use Sinkacom\CreditPlusModule\Lib\CreditPlusObjects\WebshopContract;
use Sinkacom\CreditPlusModule\Lib\CreditPlusObjects\WebshopCurrency;
use Sinkacom\CreditPlusModule\Lib\CreditPlusObjects\WebshopRateTableMonthRow;
use Sinkacom\CreditPlusModule\Model as SCModel;
use Exception;
use stdClass;

class CommonsComponent extends ESCore\Controller\BaseController {

	/**
	 * @var CreditPlusWebshopAPI|null $_oWSApi
	 */
	protected static $_oWSApi = null;
	/** @var WebshopContract[] */
	protected static $_oContractData = array();

	/**
	 * @param SCModel\Order $oOrder
	 * @return bool
	 */
	public function isStateChanged( $oOrder ) {
		$oContractData = $oOrder->getContractData();
		$sContractState = $this->translatePaymentState($oContractData);
		if ( $sContractState !== 'review_necessary' ) {
			if ( $oOrder->oxorder__oxtransstatus->value != $sContractState ) {
				// Contract state changed, send mail
				$oOrder->oxorder__oxtransstatus->setValue($sContractState);
				$oOrder->save();
				return true;
			}
		}
		return false;
	}

	/**
	 * @param WebshopContract $oContractData
	 * @return string
	 */
	protected function translatePaymentState( $oContractData ) {
		$sPaymentState = $this->getCreditPlusPaymentState($oContractData->state, $oContractData->finallyApproved, $oContractData->deliveryDone, $oContractData->informationRequest);
		return $sPaymentState;
	}

	/**
	 * @param int $iStateNumber
	 * @param bool $bFinallyApproved
	 * @param bool $bDeliveryDone
	 * @param bool $bRequestInformation
	 *
	 * @return string
	 */
	protected function getCreditPlusPaymentState( $iStateNumber, $bFinallyApproved = false, $bDeliveryDone = false, $bRequestInformation = false ) {
		$sStatus = 'review_necessary';
		if ( $bRequestInformation ) {
			// Just for removing a yellow code status
			$sStatus = 'review_necessary';
		}
		if ( $iStateNumber == 20 ) {
			$sStatus = 'creditplus_referred';
		} elseif ( $iStateNumber == 24 ) {
			$sStatus = 'creditplus_accepted';
			if ( $bFinallyApproved == false ) {
				// Already set above
			} elseif ( ($bFinallyApproved == true) && ($bDeliveryDone == true) ) {
				$sStatus = 'creditplus_approved_and_sent';
			} elseif ( $bFinallyApproved == true ) {
				$sStatus = 'creditplus_approved';
			}
		} elseif ( $iStateNumber == 21 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 22 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 23 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 25 ) {
			$sStatus = 'creditplus_docs_received';
		} elseif ( $iStateNumber == 32 ) {
			$sStatus = 'creditplus_processing_payment';
		} elseif ( $iStateNumber == 92 ) {
			$sStatus = 'creditplus_declined_soft';
		} elseif ( $iStateNumber == 93 ) {
			$sStatus = 'creditplus_declined_hard';
		} elseif ( $iStateNumber == 99 ) {
			$sStatus = 'creditplus_paid';
		} elseif ( $iStateNumber == 95 ) {
			$sStatus = 'creditplus_cancelled';
		} elseif ( $iStateNumber == 94 ) {
			$sStatus = 'creditplus_error';
		}
		return $sStatus;
	}

	/**
	 * @param SCModel\Order $oOrder
	 *
	 * @throws ESCore\Exception\SystemComponentException If a class is missing
	 * @throws \phpmailerException If the "From" value is not correctly formed
	 */
	public function sendChangedMail( $oOrder ) {
		/** @var ESCore\Email $oMail */
		$oMail = oxNew(ESCore\Email::class);
		/** @var ESCore\UtilsView $oUtilsView */
		$oUtilsView = ESCore\Registry::get(ESCore\UtilsView::class);
		/** @var \Smarty $oSmarty */
		$oSmarty = $oUtilsView->getSmarty();
		/** @var ESCore\Language $oLang */
		$oLang = ESCore\Registry::getLang();
		$sTitle = $oLang->translateString('SCCP_MAIL_TITLE_STATECHANGE_TEMPLATE');
		$sTitle = str_replace('<Bestellnummer>', $oOrder->oxorder__oxordernr->value, $sTitle);
		// OXID treats "template_dir" as array, so we add our module folder to get the templates found.
		if ( is_array($oSmarty->template_dir) ) {
			$oSmarty->template_dir[] = dirname(dirname(__FILE__)).'/views/';
		}
		$oMail->setCharSet('utf-8');
		$oMail->setSubject($sTitle);
		$oSmarty->assign('title', $sTitle);
		$oSmarty->assign('sDON', $oOrder->oxorder__oxtransid->value);
		$oSmarty->assign('sOrderNumber', $oOrder->oxorder__oxordernr->value);
		$oSmarty->assign('sNewState', $this->getCreditPlusPaymentStateForHumans($oOrder->oxorder__oxtransstatus->value));
		$oSmarty->assign('sOrderDate', $oOrder->oxorder__oxorderdate->value);
		$oSmarty->assign('oViewConf', $this->getViewConfig());
		$oSmarty->assign('oEmailView', $oMail);
		$sState = $oOrder->oxorder__oxtransstatus->value;
		$sFileHtml = 'email/html/sc_statechange_'.$sState.'.tpl';
		$sFileText = 'email/plain/sc_statechange_'.$sState.'.tpl';
		$sType = 'html';
		$sHtmlText = '';
		$sTextText = '';
		$oConfig = $this->getConfig();
		$sRecipient = $oConfig->getShopConfVar('sStateMail_Recipient_'.substr(md5($sState),0,7),null, 'module:sccp');
		$sRecipientName = $oConfig->getShopConfVar('sStateMail_RecipientName_'.substr(md5($sState),0,7),null, 'module:sccp');
		$oSmarty->assign('sRecipientName', $sRecipientName);
		$sFrom = $oConfig->getActiveShop()->oxshops__oxowneremail->value;
		$sFromName = $oConfig->getActiveShop()->oxshops__oxname->value;
		try {
			$sHtmlText = $oSmarty->fetch($sFileHtml);
			if ( trim($sHtmlText) == '' ) {
				$sType = 'text';
			}
		} catch ( Exception $oEx ) {
			$sType = 'text';
		}
		try {
			$sTextText = $oSmarty->fetch($sFileText);
		} catch ( Exception $oEx ) {
			// No Text mode available
		}
		if ( $sType == 'html' ) {
			$oMail->setBody($sHtmlText);
			$oMail->setAltBody($sTextText);
		} else {
			$oMail->setBody($sTextText);
		}
		if ( ($sTextText || $sHtmlText) && $sRecipient && $sRecipientName ) {
			$oMail->setRecipient($sRecipient, $sRecipientName);
			$oMail->setFrom($sFrom, $sFromName);
			$oMail->send();
		}
	}

	/**
	 * @param string $sPaymentState
	 *
	 * @return string
	 */
	protected function getCreditPlusPaymentStateForHumans( $sPaymentState ) {
		/** @var ESCore\Language $oLang */
		$oLang = ESCore\Registry::getLang();
		try {
			$sText = $oLang->translateString('SCCP_PAYMENT_STATE_'.$sPaymentState);
		} catch ( Exception $oEx ) {
			$sText = "Unübersetzter Status ($sPaymentState)";
		}
		return $sText;
	}

	/**
	 * @return \Sinkacom\CreditPlusModule\Lib\Controller\CreditPlusWebshopAPI|null Webshop API object
	 */
	public function getWebshopAPI() {
		if ( self::$_oWSApi === null ) {
			$oConfig = $this->getConfig();
			$aSoapParams = array(
				'soap-user' => $oConfig->getShopConfVar('sSoapUser',null,'module:sccp'),
				'soap-pass' => $oConfig->getShopConfVar('sSoapPass',null,'module:sccp'),
				'soap-type' => $oConfig->getShopConfVar('sSoapType',null,'module:sccp'),
				'wsdl' => $oConfig->getShopConfVar('sWSDL',null,'module:sccp')
			);
			self::$_oWSApi = new CreditPlusWebshopAPI($aSoapParams);
			self::$_oWSApi->setDebugState(true,false);
			try {
				/** @var LoggerComponent $oLogger */
				$oLogger = oxNew(LoggerComponent::class);
				self::$_oWSApi->setLogger($oLogger);
			} catch ( ESCore\Exception\SystemComponentException $oEx ) {
				// Doesn't happen, wouldn't matter too much if it did though.
			}
		}
		return self::$_oWSApi;
	}

	/**
	 * Requests multiple WebshopContracts by sending the transaction IDs through the API
	 * Utilizes local caching per item if same item is requested more than once.
	 *
	 * @param string[] $aTransactionIDs
	 * @return WebshopContract[] Array of WebshopContracts with TransactionID as its index
	 */
	public function getContractDataMultiple( $aTransactionIDs ) {
		$aReturn = array();
		if ( !$aTransactionIDs ) {
			return array();
		}
		$aKeys = array();
		// Don't refetch already fetched data
		// Empty TransactionID == empty WebshopContract
		foreach ( $aTransactionIDs as $iKey => $sTransactionID ) {
			if ( $sTransactionID == '' ) {
				self::$_oContractData[$sTransactionID] = new WebshopContract();
				$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				unset($aTransactionIDs[$iKey]);
			} elseif ( self::$_oContractData[$sTransactionID] ) {
				$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				unset($aTransactionIDs[$iKey]);
			} else {
				$aKeys[$sTransactionID] = $iKey;
			}
		}
		// If requests needs to be made, do it
		if ( $aTransactionIDs ) {
			$oConfig = $this->getConfig();
			$oWSApi = $this->getWebshopAPI();
			$oCreditOfferResponse = null;
			if ( $oWSApi->isApiCallable() ) {
				$oCreditOfferResponse = $oWSApi->getContractsCPWebshop( array('dealerOrderNumber' => $aTransactionIDs, 'dealerNumber' => $oConfig->getShopConfVar('sCPDealer', null, 'module:sccp') ) );
			}
			if ( !is_object($oCreditOfferResponse) ) {
				$oCreditOfferResponse = new stdClass();
				$oCreditOfferResponse->return = array( 0 => false );
			}

			if ( !is_array($oCreditOfferResponse->return) ) {
				$oCreditOfferResponse->return = array($oCreditOfferResponse->return);
			}
			// Cache contract data by TransactionID
			foreach ( $oCreditOfferResponse->return as $oCreditOfferResponseSingle ) {
				if ( is_object($oCreditOfferResponseSingle) ) {
					$sTransactionID = $oCreditOfferResponseSingle->dealerOrderNumber.'';
					self::$_oContractData[$sTransactionID] = WebshopContract::fromObject($oCreditOfferResponseSingle);
					$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
					unset($aTransactionIDs[$aKeys[$sTransactionID]]);
				} else {
					$sTransactionID = '';
					if ( count($aTransactionIDs) == 1 ) {
						$sTransactionID = $aTransactionIDs[0];
					}
					self::$_oContractData[$sTransactionID] = new WebshopContract();
					$aReturn[$sTransactionID] = self::$_oContractData[$sTransactionID];
				}
			}
		}
		// Return WebshopContract[] (using TransactionID as index)
		return $aReturn;
	}

	/**
	 * @param SCModel\Order $oOrder
	 *
	 * @throws Exception If the request queue still has an open entry (for requeuing)
	 */
	public function checkForUpdateMail($oOrder) {
		$oContractData = $oOrder->getContractData();
		$sNewState = $this->translatePaymentState($oContractData);
		if ( $sNewState !== 'review_necessary' ) {
			if ( $this->isStateChanged($oOrder) ) {
				$this->sendChangedMail($oOrder);
			}
		} else {
			/** @var ESCore\Model\ListModel|SCModel\RequestQueue[] $oList */
			$oList = oxNew(ESCore\Model\ListModel::class);
			$oList->init(SCModel\RequestQueue::class);
			$sOrderID = $oOrder->getId();
			$oList->selectString("SELECT * FROM sccp_requestqueue WHERE scalreadyexecuted = 0 AND scorderid = '$sOrderID'");
			if ( $oList && $oList->count() ) {
				// Unfinished entries exist, don't create new ones
				throw new Exception('Open request queue entry');
			} else {
				$this->createRequestQueue($sOrderID);
			}
		}
	}

	protected function createRequestQueue( $sOrderID ) {
		try {
			/** @var SCModel\RequestQueue $oNewQueue */
			$oNewQueue = oxNew( SCModel\RequestQueue::class );
			$oNewQueue->setId();
			// 40-80 minutes
			$iSeconds = rand(2400, 4800);
			$sNextDate = date('Y-m-d H:i:s',time()+$iSeconds);
			$oNewQueue->assign(array(
				'scorderid' => $sOrderID,
				'scexecutiondate' => $sNextDate,
				'scalreadyexecuted' => 0,
				'scretryround' => 1,
			));
			$oNewQueue->save();
		} catch ( Exception $oEx ) {
			// We've tried and failed...
		}
	}

	/**
	 * @param SCModel\Article $oProduct
	 *
	 * @return array
	 */
	public function getFinancingTable($oProduct) {

		$oConfig = $this->getConfig();
		$oShop = $oConfig->getActiveShop();
		$oLang = ESCore\Registry::getLang();

		$oCurrency = $oConfig->getActShopCurrencyObject();
		$sMinRate = $oConfig->getShopConfVar('sMinRate', $oShop->getShopId(), 'module:sccp');

		// Calculate Financing Table
		$dMinRate = floatval(str_replace(',', '.', $sMinRate));

		// $aMonthRows = $oCPAPI->getFinancingOptions($oDB,'713000',$fProdPrice,$fEffZins,$fMinRate,$aMonths,$oWebshopCurrency);
		$aMonthRows = $oProduct->getSccpFinancingMonths($dMinRate);
		if ( !$aMonthRows ) {
			$aFinancingTable = array('aMonthRows' => array());
			return $aFinancingTable;
		}

		$dCheapestInterestRate = null;
		foreach ( $aMonthRows as $oMonthRow ) {

			$dEffInterest = floatval(str_replace(array(
				',',
				'%'
			), array(
				'.',
				''
			), $oMonthRow->interestRate));
			if ( ($dCheapestInterestRate === null) || ($dEffInterest < $dCheapestInterestRate) ) {
				$dCheapestInterestRate = $dEffInterest;
			}
		}
		/** @var $oPayment ESModel\Payment */
		$oPayment = oxNew(ESModel\Payment::class);
		if ( !$oPayment ) {
			// oxPayment class missing, doesn't happen
			$aFinancingTable = array('aMonthRows' => array());
			return $aFinancingTable;
		}
		$oPayment->load('sccp_financing');
		$aMarker = array(
			'###MINDESTBESTELLWERT###',
			'###JAHRESZINS###'
		);
		$sMinimalOrderSum = number_format($oPayment->oxpayments__oxfromamount->value, $oCurrency->decimal, $oCurrency->dec, $oCurrency->thousand);
		$aReplacements = array(
			($oCurrency->side == 'left')?$oCurrency->sign.' '.$sMinimalOrderSum:$sMinimalOrderSum.' '.$oCurrency->sign,
			number_format($dCheapestInterestRate, $oCurrency->decimal, $oCurrency->dec, $oCurrency->thousand)
		);
		$sFinancingDescription = str_replace($aMarker, $aReplacements, $oLang->translateString('SCCP_FINANZIERUNG_DESCRIPTION'));
		$aFinancingTable = array();
		$aFinancingTable['sFinancingDescription'] = $sFinancingDescription;
		$aFinancingTable['aMonthRows'] = $aMonthRows;
		return $aFinancingTable;
	}

	/**
	 * This function gives the lowest financing rate as a string, e.g. "Finance this for $ 14.33 per month over 48 months"
	 * @param SCModel\Article $oProduct
	 * @return string The text with the lowest rate
	 */
	public function getLowestFinancingRate($oProduct) {
		$oLang = ESCore\Registry::getLang();

		$aFinancingTable = $this->getFinancingTable($oProduct);
		if ( !$aFinancingTable || !$aFinancingTable['aMonthRows'] ) {
			return '';
		}
		$aReturn = array();
		$oConfig = $this->getConfig();
		$oCurrency = $oConfig->getActShopCurrencyObject();
		$oWebshopCurrency = WebshopCurrency::fromArray(array(
			'decimals' => $oCurrency->decimal,
			'decimalSeparator' => $oCurrency->dec,
			'name' => $oCurrency->name,
			'side' => $oCurrency->side,
			'sign' => $oCurrency->sign,
			'thousandSeparator' => $oCurrency->thousand
		));

		$oWSApi = $this->getWebshopAPI();
		$sCheapestRate = '';
		$dCheapestRate = null;
		if ( $aFinancingTable['aMonthRows'] ) {
			/** @var WebshopRateTableMonthRow $oMonthRow */
			foreach ( $aFinancingTable['aMonthRows'] as $oMonthRow ) {
				$dPrice = $oWSApi->retrieveFloatPriceFromFormattedPrice($oMonthRow->monthlyRate,$oWebshopCurrency);
				if ( ($dCheapestRate == null) || ($dPrice < $dCheapestRate) ) {
					$aReturn['months'] = $oMonthRow->months;
					$aReturn['cost'] = $oMonthRow->monthlyRate;
					$dCheapestRate = $dPrice;
				}
			}
			$aMarker = array(
				'###PRICE###',
				'###MONTHS###'
			);
			$aReplacements = array(
				$aReturn['cost'],
				$aReturn['months']
			);
			$sCheapestRate = str_replace($aMarker, $aReplacements, $oLang->translateString('SCCP_FINANZIERUNG_MINIINFO'));
		}
		return $sCheapestRate;
	}

	public function addFinancingStyles() {
		$oConfig = $this->getConfig();
		// Add StyleSheet
		$aStyles = (array)$oConfig->getGlobalParameter('styles');
		$sShopRoot = $oConfig->getConfigParam('sShopDir');
		if ( substr($sShopRoot,-1) == '/' ) {
			$sShopRoot = substr($sShopRoot,0,-1);
		}
		$sCSSFile = dirname(dirname(__FILE__)).'/sccp.css';
		$aStyles[] = str_replace(realpath($sShopRoot),'',realpath($sCSSFile));
		$aStyles = array_unique($aStyles);
		$oConfig->setGlobalParameter('styles', $aStyles);
	}
}
