<?php
/**
 * This file is part of the SinkaCom CreditPlus Module Package.
 *
 * @link      http://www.sinkacom.de/
 * @copyright (C) SinkaCom AG 2015-2019
 * @version   OXID eShop CE
 */

/**
/**
 * Created by PhpStorm.
 * User: mihovil.bubnjar
 * Date: 22.01.16
 * Time: 16:10
 */

namespace Sinkacom\CreditPlusModule\Controller\Admin;

use OxidEsales\Eshop\Application\Controller\Admin\AdminDetailsController;
use OxidEsales\Eshop\Core as ESCore;
use OxidEsales\Eshop\Application\Model as ESModel;
use Sinkacom\CreditPlusModule\Component\CommonsComponent;
use Sinkacom\CreditPlusModule\Lib\CreditPlusObjects\WebshopVoucher;
use Sinkacom\CreditPlusModule\Model as SCModel;
use Exception;

/**
 * Admin order article manager.
 * Collects order articles information, updates it on user submit, etc.
 * Admin Menu: Orders -> CreditPlus Orders -> Articles.
 */

class CpOrderArticle extends AdminDetailsController
{

	/**
	 * Active order object
	 *
	 * @var ESModel\Order
	 */
	protected $_oEditObject = null;

	/**
	 * Executes parent method parent::render(), creates oxorder and oxvoucherlist
	 * objects, appends voucherlist information to order object and passes data
	 * to Smarty engine, returns name of template file "order_article.tpl".
	 *
	 * @return string
	 */
	public function render()
	{
		parent::render();

		if ($oOrder = $this->getEditObject()) {
			$this->_aViewData["edit"] = $oOrder;
			$this->_aViewData["aProductVats"] = $oOrder->getProductVats(true);
		}

		return "sccp_cporder_article.tpl";
	}

	/**
	 * Returns editable order object
	 *
	 * @return ESModel\Order
	 */
	public function getEditObject()
	{
		$soxId = $this->getEditObjectId();
		if ($this->_oEditObject === null && isset($soxId) && $soxId != "-1") {
			$this->_oEditObject = oxNew(ESModel\Order::class);
			$this->_oEditObject->load($soxId);
		}

		return $this->_oEditObject;
	}

	/**
	 * Removes article from order list.
	 */
	public function deleteThisArticle()
	{
		// get article id
		$sOrderArtId = ESCore\Registry::getRequest()->getRequestParameter('sArtID');
		$sOrderId = $this->getEditObjectId();

		/** @var SCModel\OrderArticle|ESModel\OrderArticle $oOrderArticle */
		$oOrderArticle = oxNew(ESModel\OrderArticle::class);
		$oOrder = oxNew(ESModel\Order::class);

		// order and order article exits?
		if ($oOrderArticle->load($sOrderArtId) && $oOrder->load($sOrderId)) {

			// deleting record
			$aError = $oOrderArticle->delete();
			if ( $aError ) {
				$this->_aViewData['sError'] = $aError['sError'];
				$this->_aViewData['sErrorMessage'] = $aError['sErrorMessage'];
			}
			// recalculating order
			$oOrder->recalculateOrder();
		}
	}

	/**
	 * Cancels order item
	 */
	public function storno()
	{
		$oConfig = $this->getConfig();
		$oRequest = ESCore\Registry::getRequest();
		$oLang = ESCore\Registry::getLang();
		$sOrderArtId = $oRequest->getRequestParameter('sArtID');
		/** @var SCModel\OrderArticle|ESModel\OrderArticle $oArticle */
		$oArticle = oxNew(ESModel\OrderArticle::class);
		$oArticle->load($sOrderArtId);

		if ($oArticle->oxorderarticles__oxstorno->value == 1) {
			// We cannot reactivate products, as we can only lessen the amount and not increase it
			$this->_aViewData['sError'] = 500;
			$this->_aViewData['sErrorMessage'] = $oLang->translateString('SCCP_CPORDER_ARTICLE_ERROR_ALREADY_CANCELLED');
		} else {
			$aError = $oArticle->sccpReturnProduct();
			if($aError === null){
				$oArticle->oxorderarticles__oxstorno->setValue(1);
				$sStockSign = 1;
				// stock information
				if ($oConfig->getConfigParam('blUseStock')) {
					$oArticle->updateArticleStock(floatval($oArticle->oxorderarticles__oxamount->value) * $sStockSign, $oConfig->getConfigParam('blAllowNegativeStock'));
				}
				$oArticle->setIsNewOrderItem(false);
				$oArticle->save();
				//get article id
				$sArtId = $oArticle->oxorderarticles__oxartid->value;
				if ( $sArtId ) {
					try{
						/** @var SCModel\Order|ESModel\Order $oOrder */
						$oOrder = oxNew(ESModel\Order::class);
						if ($oOrder->load($this->getEditObjectId())) {
							$oOrder->recalculateOrder();
							$oDB = ESCore\DatabaseProvider::getDb(ESCore\DatabaseProvider::FETCH_MODE_ASSOC);
							$sOrderOxid = $oOrder->oxorder__oxid->value;
							/** @var ESCore\Database\Adapter\ResultSetInterface $oRes */
							$oRes = $oDB->select("SELECT OXID oxid FROM oxorderarticles oa WHERE oa.OXORDERID = '$sOrderOxid' AND oa.OXSTORNO = 0");
							if ( !$oRes || $oRes->count() == 0 ) {
								// If this was the last article, cancel the order completely
								$oOrder->cancelOrder();
							}
						}
					} catch( ESCore\Exception\DatabaseConnectionException $oEx ) {
						// Database connection missing
						$aError = array(
							'sError' => 500,
							'sErrorMessage' => 'Database connection failed'
						);
					}  catch( ESCore\Exception\DatabaseErrorException $oEx ) {
						// Database query failed
						$aError = array(
							'sError' => 500,
							'sErrorMessage' => 'Database query failed'
						);
					} catch( Exception $oEx ) {
						// Transaction failed in recalculate order
						$aError = array(
							'sError' => 500,
							'sErrorMessage' => 'Transaction failed'
						);
					}
				}
			}

			if ( $aError ) {
				$this->_aViewData['sError'] = $aError['sError'];
				$this->_aViewData['sErrorMessage'] = $aError['sErrorMessage'];
			} else {
				$this->_aViewData['sError'] = 200;
				$this->_aViewData['sErrorMessage'] = $oLang->translateString('SCCP_CPORDER_ARTICLE_MESSAGE_REDUCED_BY_PRODUCT_PRICE');
			}
		}
	}

	/**
	 * Updates order articles stock and recalculates order
	 */
	public function updateOrder()
	{
		$aOrderArticles = ESCore\Registry::getRequest()->getRequestParameter('aOrderArticles');
		$oLang = ESCore\Registry::getLang();
		try {
			/** @var SCModel\Order|ESModel\Order $oOrder */
			$oOrder = oxNew(ESModel\Order::class);

			if (is_array($aOrderArticles) && $oOrder->load($this->getEditObjectId())) {

				$aOldAmounts = array();
				$dOldSum = floatval($oOrder->oxorder__oxtotalordersum->rawValue);
				/** @var ESModel\OrderArticleList $oOrderArticles */
				$oOrderArticles = $oOrder->getOrderArticles(true);
				$oConfig = $this->getConfig();
				$blUseStock = $oConfig->getConfigParam('blUseStock');
				/** @var ESModel\OrderArticle $oOrderArticle */
				foreach ($oOrderArticles as $oOrderArticle) {
					$sItemId = $oOrderArticle->getId();
					if (isset($aOrderArticles[$sItemId])) {
						$aOldAmounts[$sItemId] = $oOrderArticle->oxorderarticles__oxamount->value;
						// update stock
						if ($blUseStock) {
							$oOrderArticle->setNewAmount($aOrderArticles[$sItemId]['oxamount']);
						} else {
							$oOrderArticle->assign($aOrderArticles[$sItemId]);
							$oOrderArticle->save();
						}
					}
				}
				// recalculating order
				$oOrder->recalculateOrder();
				$dNewSum = floatval($oOrder->oxorder__oxtotalordersum->rawValue);
				if ($dNewSum > $dOldSum) {
					$this->returnArticlesToPreviousState($oOrder, $oOrderArticles, $aOldAmounts);
					$this->_aViewData['sError'] = 500;
					$this->_aViewData['sErrorMessage'] = $oLang->translateString('SCCP_CPORDER_ARTICLE_ERROR_NO_INCREASE_POSSIBLE');
				} else {
					$dReturnAmount = $dOldSum - $dNewSum;
					$sUserName = 'User requested';
					if (isAdmin()) {
						$sUserName = substr($this->getSession()->getUser()->oxuser__oxusername->value, 0, 27);
					}
					$sDate = date('c');
					$oContractData = $oOrder->getContractData();
					/** @var CommonsComponent $oCmpCommons */
					$oCmpCommons = oxNew(CommonsComponent::class);
					$oWSApi = $oCmpCommons->getWebshopAPI();
					$aOrderData = array(
						'dealerOrderNumber' => $oContractData->dealerOrderNumber,
						'changeDate' => $sDate,
						'changedBy' => $sUserName,
						'dealerNumber' => $oContractData->dealerNumber,
						// Storno = Restwert auf neue Summe setzen
						'loanAmount' => $dNewSum,
						'cpReferenceNumber' => $oContractData->cpReferenceNumber
					);
					$oResponse = $oWSApi->changeOrderCPWebshop($aOrderData);
					// Fehlerhandling
					if (is_object($oResponse) && (property_exists($oResponse, 'confirmation')) && (property_exists($oResponse->confirmation, 'confirmationItems')) && (property_exists($oResponse->confirmation->confirmationItems, 'errorCode'))) {
						$iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
						$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
						$aError = array(
							'sError' => (500 + $iErrorCode),
							'sErrorMessage' => $sErrorMessage
						);
						if ($iErrorCode == 1) {
							// Kein Auftrag gefunden
						} elseif ($iErrorCode == 2) {
							// Pflichtfelder fehlen, kann durch Code heraus nicht passieren
						} elseif (($iErrorCode == 6) || ($iErrorCode == 18) || ($iErrorCode == 11) || ($iErrorCode == 15)) {
							// changeOrder im Status "Bezahlt", "In Bezahlung", "Ausgeliefert" oder "Lieferfreigabe" kann nicht stattfinden
							$oVoucher = new WebshopVoucher($dReturnAmount, 'Warenkorbmodifikation', $sDate, $sUserName, false, $oContractData->id, $oContractData->dealerOrderNumber, $oContractData->dealerNumber);
							$oWSApi->addReturnProduct($oVoucher);

							$oResponse = $oWSApi->returnProductCPWebshop();
							// Fehlerhandling
							if (is_object($oResponse) && (property_exists($oResponse, 'confirmation')) && (property_exists($oResponse->confirmation, 'confirmationItems')) && (property_exists($oResponse->confirmation->confirmationItems, 'errorCode'))) {
								$iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
								$sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
								$this->returnArticlesToPreviousState($oOrder, $oOrderArticles, $aOldAmounts);
								$aError = array(
									'sError' => (500 + $iErrorCode),
									'sErrorMessage' => $sErrorMessage
								);
							} else if (($oResponse === null) || !is_object($oResponse)) {
								$this->returnArticlesToPreviousState($oOrder, $oOrderArticles, $aOldAmounts);
								$aError = array(
									'sError' => 500,
									'sErrorMessage' => $oLang->translateString('SCCP_CPORDER_ARTICLE_MESSAGE_ERROR_NOT_RETURNED')
								);
							} else {
								$aError = array(
									'sError' => 200,
									'sErrorMessage' => $oLang->translateString('SCCP_CPORDER_ARTICLE_MESSAGE_PARTIALLY_RETURNED')
								);
							}
						} elseif ($iErrorCode == 12) {
							// Summe der Retouren übersteigt Restwert des Kreditvertrags
						} elseif ($iErrorCode == 14) {
							// Irgendein Pflichtfeld sprengt sein Format/Länge
							// dealerOrderNumber > 40
							// cancelationFrom > 27
						} else {
							// Keine Ahnung was passiert ist
							$aError['debug'] = 'Keine Ahnung 1';
						}
						foreach ($aError as $sKey => $sValue) {
							$this->_aViewData[$sKey] = $sValue;
						}
					} else if (($oResponse === null) || !is_object($oResponse)) {
						$this->returnArticlesToPreviousState($oOrder, $oOrderArticles, $aOldAmounts);
						$this->_aViewData['sError'] = 500;
						$this->_aViewData['sErrorMessage'] = $oLang->translateString('SCCP_CPORDER_ARTICLE_MESSAGE_ERROR_NOT_RETURNED');
					} else {
						$this->_aViewData['sError'] = 200;
						$this->_aViewData['sErrorMessage'] = $oLang->translateString('SCCP_CPORDER_ARTICLE_MESSAGE_PARTIALLY_RETURNED');
					}

				}
			}
		} catch( Exception $oEx ){
			// Transaction error happened on recalculate
			$this->_aViewData['sError'] = 500;
			$this->_aViewData['sErrorMessage'] = 'Database error, transaction not possible';
		}
	}

	/**
	 * @param SCModel\Order|ESModel\Order $oOrder
	 * @param ESModel\OrderArticleList|ESModel\OrderArticle[] $oOrderArticles
	 * @param int[] $aOldAmounts
	 *
	 * @throws Exception If the recalculation fails
	 */
	protected function returnArticlesToPreviousState($oOrder, $oOrderArticles, $aOldAmounts) {
		$oConfig = $this->getConfig();
		$blUseStock = $oConfig->getConfigParam('blUseStock');
		/** @var ESModel\OrderArticle $oOrderArticle */
		foreach ($oOrderArticles as $oOrderArticle) {
			$sItemId = $oOrderArticle->getId();
			if (isset($aOrderArticles[$sItemId])) {
				// update stock
				if ($blUseStock) {
					$oOrderArticle->setNewAmount($aOldAmounts[$sItemId]);
				} else {
					$oOrderArticle->save();
				}
			}
		}
		$oOrder->recalculateOrder();
	}
}

class_alias(CpOrderArticle::class,'sccp_cporder_article');
