<?php
/**
 * This file is part of the SinkaCom CreditPlus Module Package.
 *
 * @link      http://www.sinkacom.de/
 * @copyright (C) SinkaCom AG 2015-2019
 * @version   OXID eShop CE
 */

namespace Sinkacom\CreditPlusModule\Controller\Admin;

use OxidEsales\Eshop\Application\Controller\Admin\AdminListController;
use OxidEsales\Eshop\Core as ESCore;

/**
 * Admin order list manager.
 * Performs collection and managing (such as filtering or deleting) function.
 * Admin Menu: Orders -> CreditPlus Orders -> List.
 */
class CpOrderList extends AdminListController {

	protected $_sThisTemplate = 'sccp_cporder_list.tpl';

	/**
	 * Name of chosen object class (default null).
	 *
	 * @var string
	 */
	protected $_sListClass = 'oxorder';

	/**
	 * Enable/disable sorting by DESC (SQL) (defaultfalse - disable).
	 *
	 * @var bool
	 */
	protected $_blDesc = true;

	/**
	 * Default SQL sorting parameter (default null).
	 *
	 * @var string
	 */
	protected $_sDefSortField = "oxorderdate";

	/**
	 * Executes parent method parent::render() and returns name of template
	 * file "order_list.tpl".
	 *
	 * @return string
	 */
	public function render() {
		$sReturn = parent::render();

		$aFolders = $this->getConfig()->getConfigParam('aOrderfolder');
		$sFolder = ESCore\Registry::getRequest()->getRequestParameter("folder");
		// first display new orders
		if ( !$sFolder && is_array($aFolders) ) {
			$aNames = array_keys($aFolders);
			$sFolder = $aNames[0];
		}

		$aSearch = array(
			'oxorderarticles' => 'ARTID',
			'oxpayments' => 'PAYMENT'
		);
		$sSearch = ESCore\Registry::getRequest()->getRequestParameter("addsearch");
		$sSearchfld = ESCore\Registry::getRequest()->getRequestParameter("addsearchfld");

		$this->_aViewData["folder"] = $sFolder?$sFolder:-1;
		$this->_aViewData["addsearchfld"] = $sSearchfld?$sSearchfld:-1;
		$this->_aViewData["asearch"] = $aSearch;
		$this->_aViewData["addsearch"] = $sSearch;
		$this->_aViewData["afolder"] = $aFolders;

		return $sReturn;
	}

	/**
	 * Adding folder check
	 *
	 * @param array $aWhere SQL condition array
	 * @param string $sqlFull SQL query string
	 *
	 * @return string $sQ
	 */
	protected function _prepareWhereQuery( $aWhere, $sqlFull ) {
		$sQ = parent::_prepareWhereQuery($aWhere, $sqlFull);
		$myConfig = $this->getConfig();
		$aFolders = $myConfig->getConfigParam('aOrderfolder');
		$sFolder = ESCore\Registry::getRequest()->getRequestParameter('folder');
		try {
			$oDb = ESCore\DatabaseProvider::getDb();
		} catch ( ESCore\Exception\DatabaseConnectionException $oEx ) {
			return $sQ;
		}
		if(is_array($aFolders)){
			//searchong for empty oxfolder fields
			if ( $sFolder && $sFolder != '-1' && in_array($sFolder,$aFolders)) {
				$sQ .= " and ( oxorder.oxfolder = ".$oDb->quote($sFolder)." )";
			} elseif ( !$sFolder ) {
				$aFolderNames = array_keys($aFolders);
				$sQ .= " and ( oxorder.oxfolder = ".$oDb->quote($aFolderNames[0])." )";
			}
		}

		return $sQ;
	}

	/**
	 * Builds and returns SQL query string. Adds additional order check.
	 *
	 * @param object $oListObject list main object
	 *
	 * @return string
	 */
	protected function _buildSelectString( $oListObject = null ) {
		$sSql = parent::_buildSelectString($oListObject);

		$sSearch = ESCore\Registry::getRequest()->getRequestParameter('addsearch');
		$sSearch = trim($sSearch);
		$sSearchField = ESCore\Registry::getRequest()->getRequestParameter('addsearchfld');

		$sSql = str_replace('oxorder where ', 'oxorder where (oxpaymenttype LIKE "sccp\\_%") AND (NOT (oxtransid = "")) AND ', $sSql);
		try {
			$oDb = ESCore\DatabaseProvider::getDb();
		} catch ( ESCore\Exception\DatabaseConnectionException $oEx ) {
			// Database connection failed
			return $sSql;
		}
		if ( $sSearch ) {
			switch ( $sSearchField ) {
				case 'oxorderarticles':
					$sQ = "oxorder left join oxorderarticles on oxorderarticles.oxorderid=oxorder.oxid where ( oxorderarticles.oxartnum like ".$oDb->quote("%{$sSearch}%")." or oxorderarticles.oxtitle like ".$oDb->quote("%{$sSearch}%")." ) and ";
					break;
				case 'oxpayments':
					$sQ = "oxorder left join oxpayments on oxpayments.oxid=oxorder.oxpaymenttype where oxpayments.oxdesc like ".$oDb->quote("%{$sSearch}%")." and ";
					break;
				default:
					$sQ = "oxorder where oxorder.oxpaid like ".$oDb->quote("%{$sSearch}%")." and ";
					break;
			}
			$sSql = str_replace('oxorder where', $sQ, $sSql);
		}

		return $sSql;
	}


	/**
	 * Returns sorting fields array
	 *
	 * @return array
	 */
	public function getListSorting() {
		$aSorting = parent::getListSorting();
		if ( isset($aSorting["oxorder"]["oxbilllname"]) ) {
			$this->_blDesc = false;
		}

		return $aSorting;
	}
}

class_alias(CpOrderList::class,'sccp_cporder_list');
