<?php

namespace SicoCreditPlus\Lib\Controller;
use SicoCreditPlus\Lib\CreditPlusHelper\AbstractShopLogger;
use SicoCreditPlus\Lib\CreditPlusHelper\CreditPlusMainData;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopCreditOffer;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopCurrency;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopFinanceArticle;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopVoucher;

/**
 * Credit-Plus Webshop API
 *
 * @author Sven Keil
 */
class CreditPlusWebshopAPI {

	/**
	 * Zählt die Requests
	 *
	 * @var integer
	 */
	private $iCallCount = 0;

	/**
	 * Debug-Ausgabe steuern
	 *
	 * @var bool
	 */
	private $bDebugInfo = false;
	private $bDebugPrintInfo = false;

	/**
	 * WSSoapClient Object
	 * @var WSSoapClient
	 */
	private $oWSSoapClient;

	/**
	 * WSSoapClient Object
	 * @var CreditPlusMainData
	 */
	private $oMainData;

	/**
	 * Logger object to log according to Shop standards.
	 * Object is provided by Shop
	 * @var AbstractShopLogger
	 */
	private $oLogger;

	private $_aVouchers = array();

	private $bCacheFiles = false;
	private static $aCacheFiles = [];

	/**
	 * @param array $aParams Set params like soap-user, soap-pass, soap-type, wsdl
	 */
	public function __construct($aParams = array()) {
		//initialize
		$this->oMainData = new CreditPlusMainData();
		if ( isset($aParams['soap-user']) ) {
			$this->oMainData->setSoapUser($aParams['soap-user']);
		}
		if ( isset($aParams['soap-pass']) ) {
			$this->oMainData->setSoapPass($aParams['soap-pass']);
		}
		if ( isset($aParams['soap-type']) ) {
			$this->oMainData->setSoapType($aParams['soap-type']);
		}
		if ( isset($aParams['wsdl']) ) {
			$this->oMainData->setWSDL($aParams['wsdl']);
		}
		if ( isset($aParams['cache']) ) {
			$this->bCacheFiles = (bool)$aParams['cache'];
		}

		$aMainData = $this->oMainData->getDefaultData();
		$this->createWSSoap($aMainData['wsdl'], $aMainData["soap-user"], $aMainData["soap-pass"], $aMainData["soap-type"]);
	}

	/**
	 * @param bool $bDebugInfo
	 * @param bool $bDebugPrintInfo
	 */
	public function setDebugState( $bDebugInfo = false, $bDebugPrintInfo = false ) {
		$this->bDebugInfo = $bDebugInfo;
		$this->bDebugPrintInfo = $bDebugPrintInfo;
	}

	/**
	 * @param AbstractShopLogger $oLogger
	 */
	public function setLogger( $oLogger = null ) {
		$this->oLogger = $oLogger;
	}

	/**
	 * Write SoapClient Request-Info
	 * to Log-File
	 *
	 * @param bool|string $sToday Timestamp for log file name, false generates current timestamp
	 * @param bool $bError Whether or not this file is prefixed with error
	 * @param string $sFunctionName Which function has been called
	 */
	private function writeLastRequestInfo( $sToday = false, $bError = false, $sFunctionName = '' ) {
		if ( $this->oWSSoapClient != null ) {
			if ( $this->bDebugInfo && $this->oLogger ) {
				if ( $sToday === false ) {
					$sToday = date('Y-m-d H:i:s');
				}
				$sLastRequest = $this->oWSSoapClient->__getLastRequest();
				if ( $sLastRequest == '' ) { $sLastRequest = '(leerer String)'; }
				$this->oLogger->writeLastRequestInfo($sToday, $bError, $sFunctionName, $sLastRequest);
			}
		}
	}

	/**
	 * Write SoapClient Request-Info
	 * to Log-File
	 *
	 * @param bool|string $sToday String for date or false to use current date
	 * @param bool $bError Whether or not to use error prefix in file name
	 * @param string $sFunctionName Which function was called
	 */
	private function writeLastResponseInfo( $sToday = false, $bError = false, $sFunctionName = '' ) {
		if ( $this->oWSSoapClient != null ) {
			$sLastResponse = $this->oWSSoapClient->__getLastResponse();
			if ( $sLastResponse == '' ) { $sLastResponse = '(leerer String)'; }
		} else {
			$sLastResponse = 'SoapClient was null.';
		}
		if ( $this->bDebugInfo && $this->oLogger ) {
			if ( $sToday === false ) {
				$sToday = date('Y-m-d H:i:s');
			}
			$this->oLogger->writeLastResponseInfo($sToday, $bError, $sFunctionName, $sLastResponse);
		}
	}

	/**
	 * Write Request-Information-Data
	 * to Log-File
	 *
	 * @param string $sFunctionName The called API function name (e.g. getContracts)
	 * @param array $aArgumentData The arguments passed to $sFunctionName
	 * @param bool|string $sToday False, if time should be taken at the write time or string with file-system friendly timestamp
	 * @param bool $bError true, if logged event is an error
	 */
	private function writeRequestInformation( $sFunctionName, $aArgumentData, $sToday = false, $bError = false ) {
		if ( $this->bDebugInfo && $this->oLogger ) {
			if ( $sToday === false ) {
				$sToday = date('Y-m-d H:i:s');
			}
			$this->oLogger->writeRequestInformation($sToday, $bError, $sFunctionName, $aArgumentData);
		}
	}

	/**
	 * Set WS-Security credentials
	 *
	 * @param string $sWSDL
	 * @param string $sUsername
	 * @param string $sPassword
	 * @param string $sPasswordType
	 */
	public function createWSSoap( $sWSDL, $sUsername, $sPassword, $sPasswordType ) {
		$oStreamContext = stream_context_create(array(
			'http' => array(
				'timeout' => 1
			),
			'https' => array(
				'timeout' => 1
			)
		));
		if ( $sWSDL && (substr($sWSDL,0, 4) === 'http') ) {
			//set_error_handler(function(){throw new \Exception('could not connect to url');});
			try{
				if ( $this->bCacheFiles && isset(self::$aCacheFiles[$sWSDL]) ) {
					$sWSDLContent = self::$aCacheFiles[$sWSDL];
				} else {
					$sWSDLContent = file_get_contents($sWSDL, false, $oStreamContext);
				}

				if ( ($sWSDLContent !== false) && (strpos($sWSDLContent,'wsdl:definitions') !== false) ) {
					if ( $this->bCacheFiles ) {
						self::$aCacheFiles[$sWSDL] = $sWSDLContent;
					}
					$sWSDLData = 'data:text/plain;base64,'.base64_encode($sWSDLContent);
					$this->oWSSoapClient = new WSSoapClient($sWSDLData, array(
						'encoding' => 'UTF-8',
						"trace" => true,
						"exceptions" => true
					));
					$this->oWSSoapClient->__setUsernameToken($sUsername, $sPassword, $sPasswordType);
				} else {
					$this->oWSSoapClient = null;
				}
			} catch (\ErrorException $e){
				if($this->oLogger){
					$this->oWSSoapClient = null;
					$this->oLogger->writeLastResponseInfo(date('Y-m-d H:i:s'), true,'createWSSoap', 'Could not connect to '.$sWSDL);
				} else {
					$this->oWSSoapClient = null;
				}

			}

		}
	}

	/**
	 * Shows whether the API WSDL was found
	 * @return bool
	 */
	public function isApiCallable() {
		if ( isset($this->oWSSoapClient) && ( $this->oWSSoapClient instanceof WSSoapClient ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Ping the Webshop API.
	 * Only checks whether the API is responding or not
	 *
	 * @return mixed|string
	 */
	public function pingCPWebshop() {
		$oResponse = "Error - Call failed";
		$sCallFunctionName = "ping";
		$aMainData = $this->oMainData->getDefaultData();
		$aData = array(
			'dealerNumber' => $aMainData['dealerNumber']
		);
		try {
			$oResponse = $this->callApi($sCallFunctionName, $aData);
		} catch ( \Exception $e ) {
			$this->printDebugCode(__FUNCTION__.' Exception', $e);
		}
		$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * call createCreditOffer from Webshop API.
	 * Dummy if no Data given
	 *
	 * @param bool|array $aInput
	 * @param CreditPlusMainData $oDefaultData
	 * @param array $aCustomerAddress
	 * @param array $aShippingAddress
	 * @param array $aShopIntegrationData
	 * @return string|object "Error" or API Response Object
	 */
	public function createCreditOfferCPWebshop( $aInput = false, $oDefaultData = null, $aCustomerAddress = array(), $aShippingAddress = array(), $aShopIntegrationData = array() ) {
		$oResponse = "Error - Call failed";
		$sCallFunctionName = "createCreditOffer";

		$bIsDummyRequest = false;
		if ( $aInput == false ) {
			$bIsDummyRequest = true;
		}

		try {
			if ( $bIsDummyRequest ) {
				$shopIData = new WebshopCreditOffer();
				$aArgumentData = $shopIData->getDummyArray();
			} else {
				$shopIData = new WebshopCreditOffer($aInput, $oDefaultData, $aCustomerAddress, $aShippingAddress, $aShopIntegrationData);
				$aArgumentData = $shopIData->getCompleteApiArray();
			}

			$this->printDebugCode("createCreditOffer with ARRAY", $aArgumentData);
			$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);

		} catch ( \Exception $e ) {
			$this->printDebugCode(__FUNCTION__.' Exception', $e);
		}
		$this->printDebugCode($sCallFunctionName,
			[
				'RequestHeaders' => $this->oWSSoapClient->__getLastRequestHeaders(),
				'Request' => $this->oWSSoapClient->__getLastRequest(),
				'ResponseHeaders' => $this->oWSSoapClient->__getLastResponseHeaders(),
				'Response' => $this->oWSSoapClient->__getLastResponse(),
				'File' => __FILE__,
				'LINE' => __LINE__
			]);

		return $oResponse;
	}


	/**
	 * Return a product via API, initiates a "refund" and diminishes payout
	 * @param WebshopVoucher[] $aVouchers
	 * @return string|Object "Error", Error Message or the API Return Object
	 */
	public function returnProductCPWebshop( $aVouchers = array() ) {

		$oResponse = "Error - Call failed";
		$sCallFunctionName = "returnProduct";

		$bWorkStack = false;
		// If we are not sending specific Vouchers, we are using Vouchers from stack,
		// which we need to clean after we sent it.
		if ( !$aVouchers ) {
			$aVouchers = $this->_aVouchers;
			$bWorkStack = true;
		}

		if ( $aVouchers ) {
			$aArgumentData = array(
				'param1' => array(
					'voucher' => $aVouchers,
					'language' => 'de'
				)
			);
			try {
				$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
				if ( $bWorkStack ) {
					// Clean up after pushing all stacked Vouchers into the API
					$this->_aVouchers = array();
				}
			} catch ( \Exception $e ) {
				$oResponse = $e->getMessage();
				$this->printDebugCode(__FUNCTION__.' Exception', $e);
			}
		}

		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * @param WebshopVoucher $oWebshopVoucher
	 */
	public function addReturnProduct( $oWebshopVoucher ) {
		$aVoucher = array(
			'value' => $oWebshopVoucher->value,
			'number' => $oWebshopVoucher->number,
			'date' => $oWebshopVoucher->date,
			'agentName' => $oWebshopVoucher->agentName,
			'done' => $oWebshopVoucher->done,
			'contractId' => $oWebshopVoucher->contractId,
			'dealerOrderNumber' => $oWebshopVoucher->dealerOrderNumber,
			'dealerNumber' => $oWebshopVoucher->dealerNumber
		);
		$this->_aVouchers[] = $aVoucher;
	}


	/**
	 * getContracts the Webshop API.
	 * @param array $aFilter Filters based on the CreditPlus API description
	 * @return string|Object String on Error, Object if response from CreditPlus API
	 */
	public function getContractsCPWebshop( $aFilter ) {
		$oResponse = "Error - Call failed";

		$sCallFunctionName = 'getContracts';

		$aMainData = $this->oMainData->getDefaultData();
		$aData = array(
			'dealerNumber' => $aMainData['dealerNumber']
		);

		if ( is_array($aFilter['dealerOrderNumber']) ) {
			foreach ( $aFilter['dealerOrderNumber'] as $sDON ) {
				$aFilter[] = new \SoapVar($sDON, XSD_STRING, null, null, 'dealerOrderNumber');
			}
			unset($aFilter['dealerOrderNumber']);
		}

		$aData = array_merge($aData, $aFilter);
		$aArgumentData = array(
			'param1' => array(
				'filter' => new \SoapVar($aData, SOAP_ENC_OBJECT),
				'language' => 'de'
			)
		);

		$this->printDebugCode("getContracts with ARRAY", $aArgumentData);

		try {
			$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
		} catch ( \Exception $e ) {
			$this->printDebugCode(__FUNCTION__.' Exception', $e);
		}

		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * Fetch the completeOrderUrl (only available on "inorder" setting).
	 * This will only give valid values on status 24 and should be called within "getContracts"
	 *
	 * @param string $sDealerOrderNumber Dealer Order Number, either CP-number or order number (depending on other settings). Use the value from getContracts.
	 *
	 * @return object|string Either SimpleXmlIterator or similar object. Or "Error - call failed" in case of an error.
	 */
	public function completeOrderUrlCPWebshop( $sDealerOrderNumber ) {
		$oResponse = "Error - Call failed";

		$sCallFunctionName = 'completeOrderUrl';

		$aArgumentData = array(
			'param1' => array(
				'dealerOrderNumber' => $sDealerOrderNumber
			)
		);
		$this->printDebugCode("$sCallFunctionName with ARRAY", $aArgumentData);
		try {
			$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
		} catch ( \Exception $e ) {
			$this->printDebugCode(__FUNCTION__.' Exception', $e);
		}

		$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * getContracts the Webshop API.
	 * @param string $sDate Date Filter based on the CreditPlus API description - Currently requires xsd:dateTime format (=date('c'))
	 * @return string|Object String on Error, Object if response from CreditPlus API
	 */
	public function getRemittancesCPWebshop( $sDate = '' ) {
		$oResponse = "Error - Call failed";

		$sCallFunctionName = 'getRemittance';

		if ( !$sDate || ($sDate === '') ) {
			$sDate = date('c', time() - 86400); // 1 Day back
		}

		$aArgumentData = array(
			'param1' => array(
				'remittanceDate' => $sDate,
				'language' => 'de'
			)
		);

		$this->printDebugCode(__FUNCTION__." with ARRAY", $aArgumentData);

		try {
			$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
		} catch ( \Exception $e ) {
			$this->printDebugCode(__FUNCTION__.' Exception', $e);
		}
		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * Signalize the Webshop API that your shop sent the items to customer. Triggers starting the payout process for CreditPlus.
	 *
	 * @param array('dealerNumber' => '', 'dealerOrderNumber' => '', 'invoiceNumber' => '', 'invoicePrice' => '', 'deliveryDate' => '') $aOrderData
	 * @return string|object Text with error or object with data
	 */
	public function commitDeliveryCPWebshop( $aOrderData ) {
		$oResponse = "Call failed with Exception";
		$sCallFunctionName = 'commitDelivery';

		$aMainData = $this->oMainData->getDefaultData();
		$aData = array(
			'dealerNumber' => $aMainData['dealerNumber'],
			'dealerOrderNumber' => '',
			'invoiceNumber' => '',
			'invoicePrice' => 0.00,
			'deliveryDate' => date('c')
		);

		$aData = array_merge($aData, $aOrderData);
		$aArgumentData = array(
			'param1' => array(
				'delivery' => $aData,
				'language' => 'de'
			)
		);

		$this->printDebugCode("commitDeliery with ARRAY", $aArgumentData);
		if ( $aData['dealerOrderNumber'] && $aData['invoiceNumber'] && $aData['invoicePrice'] && $aData['deliveryDate'] ) {
			try {
				$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
			} catch ( \Exception $e ) {
				$this->printDebugCode(__FUNCTION__.' Exception', $e);
			}
		} else {
			$oResponse = 'Missing mandatory parameters.';
			$aMissing = array();
			if ( !$aData['dealerOrderNumber'] ) {
				$aMissing[] = 'dealerOrderNumber missing.';
			}
			if ( !$aData['invoiceNumber'] ) {
				$aMissing[] = 'invoiceNumber missing.';
			}
			if ( !$aData['invoicePrice'] ) {
				$aMissing[] = 'invoicePrice missing.';
			}
			if ( !$aData['deliveryDate'] ) {
				$aMissing[] = 'deliveryDate missing.';
			}
			$this->printDebugCode(__FUNCTION__.' Exception', $aMissing);
		}

		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * Cancel an Order before it is delivered.
	 * This may not be called after the Contract has deliveryDone set to true.
	 * @param array('dealerNumber' => '', 'dealerOrderNumber' => '', 'invoiceNumber' => '', 'invoicePrice' => '', 'deliveryDate' => '') $aCancelData
	 * @return string|object Text with error or object with data
	 */
	public function cancelOrderCPWebshop( $aCancelData ) {
		$oResponse = "Call failed with Exception";
		$sCallFunctionName = 'cancelOrder';

		$aMainData = $this->oMainData->getDefaultData();
		$aData = array(
			'dealerNumber' => $aMainData['dealerNumber'],
			'dealerOrderNumber' => '',
			'cancelationDate' => date('c'),
			'cancelationFrom' => ''
		);

		$aData = array_merge($aData, $aCancelData);
		$aArgumentData = array(
			'param1' => array(
				'cancelation' => $aData,
				'language' => 'de'
			)
		);

		$this->printDebugCode("cancelOrderCPWebshop with ARRAY", $aArgumentData);
		if ( $aData['dealerOrderNumber'] && $aData['cancelationDate'] && $aData['cancelationFrom'] ) {
			try {
				$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
			} catch ( \Exception $e ) {
				$this->printDebugCode(__FUNCTION__.' Exception', $e);
			}
		} else {
			$oResponse = 'Missing mandatory parameters.';
			$aMissing = array();
			if ( !$aData['dealerOrderNumber'] ) {
				$aMissing[] = 'dealerOrderNumber missing.';
			}
			if ( !$aData['cancelationDate'] ) {
				$aMissing[] = 'cancelationDate missing.';
			}
			if ( !$aData['cancelationFrom'] ) {
				$aMissing[] = 'cancelationFrom missing.';
			}
			$this->printDebugCode(__FUNCTION__.' Exception', $aMissing);
		}
		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * Change an Order before it is delivered.
	 * This may not be called after the Contract has deliveryDone set to true.
	 * @param array('dealerNumber' => '', 'dealerOrderNumber' => '', 'invoiceNumber' => '', 'invoicePrice' => '', 'deliveryDate' => '') $aChangeData
	 * @return string|object Text with error or object with data
	 */
	public function changeOrderCPWebshop( $aChangeData ) {
		$oResponse = "Call failed with Exception";
		$sCallFunctionName = 'changeOrder';

		$aMainData = $this->oMainData->getDefaultData();
		$aData = array(
			'dealerOrderNumber' => '',
			'changeDate' => date('c'),
			'dealerNumber' => $aMainData['dealerNumber'],
			'loanAmount' => 0.00,
			'cpReferenceNumber' => 0
		);

		$aData = array_merge($aData, $aChangeData);
		$aArgumentData = array(
			'param1' => array(
				'change' => $aData,
				'language' => 'de'
			)
		);

		$this->printDebugCode("changeOrderCPWebshop with ARRAY", $aArgumentData);
		if ( $aData['dealerOrderNumber'] && $aData['changeDate'] && isset($aData['loanAmount']) ) {
			try {
				$oResponse = $this->callApi($sCallFunctionName, $aArgumentData);
			} catch ( \Exception $e ) {
				$this->printDebugCode(__FUNCTION__.' Exception', $e);
			}
		} else {
			$oResponse = 'Missing mandatory parameters.';
			$aMissing = array();
			if ( !$aData['dealerOrderNumber'] ) {
				$aMissing[] = '<br />dealerOrderNumber missing.';
			}
			if ( !$aData['changeDate'] ) {
				$aMissing[] = '<br />changeDate missing.';
			}
			if ( !$aData['loanAmount'] ) {
				$aMissing[] = '<br />loanAmount missing.';
			}
			$this->printDebugCode(__FUNCTION__.' Exception', $aMissing);
		}

		//$this->printDebugCode($sCallFunctionName, $this->oWSSoapClient);

		return $oResponse;
	}

	/**
	 * Print out debug information or write them into the log - depending on settings
	 * @param string $sMessage Message to print
	 * @param object|array $oObject Related object
	 * @return int
	 */
	private function printDebugCode( $sMessage = '', $oObject = null ) {

		//SKeil 2015-12-09
		//zum debuggen
		if ( $this->bDebugPrintInfo == true ) {
			echo "<div style='background-color: yellow; display:none;' class='debug'>";
			echo "<p>DEBUG :: <b>START -- START -- START -- START</b> :: DEBUG</p>";

			$aBacktrace = debug_backtrace();
			$aLast = next($aBacktrace);
			$sLastClass = $aLast['class'];
			$sLastFunction = $aLast['function'];

			echo "<h3>$sLastClass | $sLastFunction</h3>";
			echo "<p>Ausgabe <b>MESSAGE</b></p>";
			var_dump($sMessage);

			echo "<p>Ausgabe <b>OBJECT</b></p>";
			var_dump($oObject);

			echo "<p>DEBUG :: <b>END -- END -- END -- END</b> :: DEBUG</p>";
			echo "</div>";

			return 1;
		}
		$bError = $oObject instanceof \Exception;
		$this->writeRequestInformation( 'printDebugCode', array($sMessage, $oObject), date('Y-m-d H:i:s'), $bError );

		return 0;
	}


	/**
	 * Overwrites the original method adding the security header.
	 * As you can see, if you want to add more headers, the method needs to be modified.
	 *
	 * @param string $sFunctionName Which API function this will call
	 * @param array $aArgumentData Which parameteres are being transmitted?
	 * @return object|string
	 */
	private function callApi( $sFunctionName, $aArgumentData ) {
		$this->iCallCount++;
		$sToday = date("Y-m-d H:i:s");

		try {
			$oResponse = $this->oWSSoapClient->__call($sFunctionName, $aArgumentData);
			$this->writeLastRequestInfo($sToday, false, $sFunctionName);
			$this->writeLastResponseInfo($sToday, false, $sFunctionName);
		} catch ( \Exception $e ) {
			$oResponse = $e->getMessage();
			$this->writeLastRequestInfo($sToday, true, $sFunctionName);
			$this->writeLastResponseInfo($sToday, true, $sFunctionName);
			$this->writeRequestInformation($sFunctionName, $aArgumentData, $sToday, true);
		}

		return $oResponse;
	}

	/**
	 * @param float $dPrice Financed Price
	 * @param int $iMonths Financed Months
	 * @param float $dInterest Yearly nominal interest
	 * @param bool|float $dRateFactor If not false the factor used to calculate the monthly payment
	 * @return float The amount that will be paid each month
	 */
	public function getMonthRateByPriceMonthsAndInterest( $dPrice = 0.00, $iMonths = 0, $dInterest = 0.00, $dRateFactor = false ) {
		$dInterest = $dInterest/100.0;

		if ( $dRateFactor === false ) {
			// Calculate monthly payments by financial mathematics rate = price * (q/12) / (1 - (1/(1+(q/12))^n) | n = Months, 12 = months per year
			if ( $dInterest != 0 ) {
				//$dRate = $dPrice * ( ($dInterest/12) / ( 1 - pow(1+($dInterest/12),0-$iMonths) ) );

				// Summe mit Optimierung
				$dQ = (1/pow(1+$dInterest,1/12));
				$dS = (((1-pow($dQ,$iMonths+1))/(1-$dQ))-1);

				/*
				// Summe nach Formel - ohne mathematische Optimierung auf geometrische Reihe
				$dQ = pow(1+$dInterest,1/12);
				$dS = 0;
				for ( $iMonth = 1 ; $iMonth <= $iMonths ; $iMonth++ ) {
					$dS += (1/pow($dQ,$iMonth));
				}
				*/
				$dQny = pow((1+$dInterest),15/365);

				$dRate = $dPrice * $dQny * (1 / $dS);
			} else {
				// The formula above crashes because of divisions by zero...
				// As it doesn't need higher mathematics, this will suffice:
				$dRate = $dPrice/$iMonths;
			}
		} else {
			$dRate = $dPrice * $dRateFactor;
		}
		$dRate = round(round(round(round($dRate, 5), 4), 3), 2);
		return $dRate;
	}

	/**
	 * @param float $dInterest Nominal Interest as percentage (9.00 for 9% p.a.)
	 * @return float Effective Interest as Factor
	 */
	public function getEffectiveInterestFromNominalInterest($dInterest = 0.00) {
		$dInterest = $dInterest/100.0;
		$dEffective = pow(( 1 + $dInterest/12),12) -1;
		return round($dEffective*100.0,2);
	}

	/**
	 * @param float $dInterest Effective Interest as percentage (9.00 for 9% p.a.)
	 * @return float Nominal Interest as Factor
	 */
	public function getNominalInterestFromEffectiveInterest( $dInterest = 0.00 ) {
		$dInterest = $dInterest / 100.0;
		$dNominal = 12.0 * ( pow(($dInterest+1),1/12) - 1 );
		return round($dNominal*100.0,2);
	}

	/**
	 * Takes away all the formatting things from a formatted price and returns a float value of it
	 * This is always currency specific, as it will use the currency object from the shop
	 * @param string $sPrice Formatted Price (e.g. 12.421,41 EUR)
	 * @param WebshopCurrency $oCurrency Currency Object (properties sign, thousand, dec, decimal are required)
	 * @return float Float value of price (e.g. 12421.41)
	 */
	public function retrieveFloatPriceFromFormattedPrice($sPrice, &$oCurrency) {
		// Remove Currency sign, spaces, number formatting options and recreate float from formatted price...
		$sPrice = str_replace($oCurrency->sign,'',$sPrice);
		$sPrice = str_replace(' ','',$sPrice);
		$sPrice = str_replace($oCurrency->thousandSeparator,'',$sPrice);
		$sPrice = str_replace($oCurrency->decimalSeparator,'',$sPrice);
		$sPrice = substr($sPrice,0,-$oCurrency->decimals).'.'.substr($sPrice,-$oCurrency->decimals);
		$dPrice = floatval($sPrice);
		return $dPrice;
	}


	/**
	 * Takes away the formatting from a formatted value and returns it as a float.
	 * @param string $sInterestRate Formatted interest rate (e.g. 6,20 %)
	 * @param WebshopCurrency $oCurrency
	 * @return float Float value of interest rate (e.g. 6.2)
	 */
	public function retrieveFloatFromFormattedInterestRate($sInterestRate, &$oCurrency ) {
		return floatval(str_replace(array(
			' ',
			$oCurrency->thousandSeparator,
			$oCurrency->decimalSeparator,
			'%'
		), array(
			'',
			'',
			'.',
			''
		), $sInterestRate));
	}

	/**
	 * Returns the article, which should be used to display the table
	 * @param WebshopFinanceArticle[] $aArticles Array of Basket Articles with corresponding prices and amounts
	 * @param string $sMethod One strategy of "most-expensive", "weighted-majority", "number-majority", "cheapest"
	 * @return WebshopFinanceArticle Article which should be used for reference when creating the table
	 */
	public function getFinancingArticleReference( $aArticles = array(), $sMethod = 'weighted-majority') {
		$this->sortArticlesByMostSpecificFirst($aArticles);
		$oArticle = $aArticles[0];

		if ( $sMethod == 'most-expensive' ) {
			$oArticle = $this->getFinancingArticleReferenceMostExpensive($aArticles);
		} elseif ( $sMethod == 'weighted-majority' ) {
			$oArticle = $this->getFinancingArticleReferenceWeightedMajority($aArticles);
		} elseif ( $sMethod == 'expensive-product' ) {
			$oArticle = $this->getFinancingArticleReferenceMostExpensiveProduct($aArticles);
		} elseif ( $sMethod == 'number-majority' ) {
			$oArticle = $this->getFinancingArticleReferenceNumberMajority($aArticles);
		} elseif ( $sMethod == 'cheapest' ) {
			$oArticle = $this->getFinancingArticleReferenceCheapest($aArticles);
		}

		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles The array of possible articles
	 * @return WebshopFinanceArticle The article to be used
	 */
	protected function getFinancingArticleReferenceMostExpensive($aArticles) {
		/** @var WebshopFinanceArticle $oArticle */
		$oArticle = reset($aArticles);
		foreach ( $aArticles as $oPossibleArticle ) {
			if ( $oArticle->mostExpensiveInterestRate < $oPossibleArticle->mostExpensiveInterestRate ) {
				$oArticle = $oPossibleArticle;
			} elseif ( $oArticle->mostExpensiveInterestRate == $oPossibleArticle->mostExpensiveInterestRate ) {
				if ( $oArticle->cheapestInterestRate < $oPossibleArticle->cheapestInterestRate ) {
					// If most expensive interest rate is equal, use cheapest for comparison
					$oArticle = $oPossibleArticle;
				}
			}
		}
		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles The array of possible articles
	 * @return WebshopFinanceArticle The article to be used
	 */
	protected function getFinancingArticleReferenceWeightedMajority($aArticles) {
		/** @var WebshopFinanceArticle[] $aSortable */
		$aSortable = array();
		foreach ( $aArticles as $oArticle ) {
			$dIndex = ($oArticle->unitprice);
			// If not set, or cheapest rate is smaller or cheapest rate is equal and most expensive rate is smaller
			// == best possible outcome for customer, as it doesn't matter in which way he put the items in his basket
			if (
				!isset($aSortable[$dIndex])
				|| ($aSortable[$dIndex]->cheapestInterestRate > $oArticle->cheapestInterestRate)
				|| (
					($aSortable[$dIndex]->cheapestInterestRate === $oArticle->cheapestInterestRate) &&
					($aSortable[$dIndex]->mostExpensiveInterestRate > $oArticle->mostExpensiveInterestRate)
				)
			) {
				$aSortable[$dIndex] = $oArticle;
			}
		}
		ksort($aSortable);
		$oArticle = array_pop($aSortable);
		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles The array of possible articles
	 * @return WebshopFinanceArticle The article to be used
	 */
	protected function getFinancingArticleReferenceMostExpensiveProduct($aArticles) {
		/** @var WebshopFinanceArticle[] $aSortable */
		$aSortable = array();
		foreach ( $aArticles as $oArticle ) {
			$dIndex = ($oArticle->unitprice/$oArticle->amount);
			// If not set, or cheapest rate is smaller or cheapest rate is equal and most expensive rate is smaller
			// == best possible outcome for customer, as it doesn't matter in which way he put the items in his basket
			if (
				!isset($aSortable[$dIndex])
				|| ($aSortable[$dIndex]->cheapestInterestRate > $oArticle->cheapestInterestRate)
				|| (
					($aSortable[$dIndex]->cheapestInterestRate === $oArticle->cheapestInterestRate) &&
					($aSortable[$dIndex]->mostExpensiveInterestRate > $oArticle->mostExpensiveInterestRate)
				)
			) {
				$aSortable[$dIndex] = $oArticle;
			}
		}
		ksort($aSortable);
		$oArticle = array_pop($aSortable);
		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles The array of possible articles
	 * @return WebshopFinanceArticle The article to be used
	 */
	protected function getFinancingArticleReferenceNumberMajority($aArticles) {
		/** @var WebshopFinanceArticle[] $aSortable */
		$aSortable = array();
		foreach ( $aArticles as $oArticle ) {
			$dIndex = ($oArticle->amount);
			// If not set, or cheapest rate is smaller or cheapest rate is equal and most expensive rate is smaller
			// == best possible outcome for customer, as it doesn't matter in which way he put the items in his basket
			if (
				!isset($aSortable[$dIndex])
				|| ($aSortable[$dIndex]->cheapestInterestRate > $oArticle->cheapestInterestRate)
				|| (
					($aSortable[$dIndex]->cheapestInterestRate === $oArticle->cheapestInterestRate) &&
					($aSortable[$dIndex]->mostExpensiveInterestRate > $oArticle->mostExpensiveInterestRate)
				)
			) {
				$aSortable[$dIndex] = $oArticle;
			}
		}
		ksort($aSortable);
		/** @var WebshopFinanceArticle $oArticle */
		$oArticle = array_pop($aSortable);
		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles The array of possible articles
	 * @return WebshopFinanceArticle The article to be used
	 */
	protected function getFinancingArticleReferenceCheapest($aArticles) {
		/** @var WebshopFinanceArticle $oArticle */
		$oArticle = reset($aArticles);
		foreach ( $aArticles as $oPossibleArticle ) {
			if ( $oArticle->cheapestInterestRate > $oPossibleArticle->cheapestInterestRate ) {
				$oArticle = $oPossibleArticle;
			} elseif ( $oArticle->cheapestInterestRate == $oPossibleArticle->cheapestInterestRate ) {
				if ( $oArticle->mostExpensiveInterestRate > $oPossibleArticle->mostExpensiveInterestRate ) {
					// If cheapest rate is equal, compare by most expensive rate
					$oArticle = $oPossibleArticle;
				} elseif ( $oArticle->mostExpensiveInterestRate === $oPossibleArticle->mostExpensiveInterestRate ) {
					foreach ( $oPossibleArticle->aMonthRows as $iMonth => $dInterest ) {
						if ( isset($oArticle->aMonthRows[$iMonth]) && ( $oArticle->aMonthRows[$iMonth] > $dInterest ) ) {
							$oArticle = $oPossibleArticle;
							break;
						}
					}
				}
			}
		}
		return $oArticle;
	}

	/**
	 * @param WebshopFinanceArticle[] $aArticles
	 *
	 * @return WebshopFinanceArticle[]
	 */
	protected function sortArticlesByMostSpecificFirst(&$aArticles) {
		usort($aArticles, array($this, 'isFinancingArticleMoreSpecific'));
		return $aArticles;
	}
	/**
	 * @param WebshopFinanceArticle $oBaseArticle
	 * @param WebshopFinanceArticle $oMaybeMoreSpecifcArticle
	 *
	 * @return bool
	 */
	public function isFinancingArticleMoreSpecific( $oBaseArticle, $oMaybeMoreSpecifcArticle ) {
		if ( $oBaseArticle->isProductSpecific == false ) {
			return 1;
		}
		if ( $oMaybeMoreSpecifcArticle->isProductSpecific ) {
			return 1;
		}
		return -1;
	}
}
