<?php declare(strict_types=1);

namespace SicoCreditPlus\Migration;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Result;
use Shopware\Core\Framework\Migration\MigrationStep;

/**
 * Migration very similar to Migration1655194919StatusMails
 * This one does the same stuff but updates the 1.2.20 Version mails to 1.2.21 unless 1.2.20 already did that.
 */
class Migration1655311679StatusMailsTrueFalse extends MigrationStep
{
    public function getCreationTimestamp(): int
    {
        return 1655311679;
    }

	/**
	 * @param Connection $connection Database connection, default Doctrine stuff
	 * @return void
	 * @throws Exception Regular Doctrine stuff
	 */
	public function update(Connection $connection): void
	{
		$this->updateAfterOrderStatusMail($connection);
		$this->updateBeforeOrderStatusMail($connection);
	}

	/**
	 * @param Connection $oConnection Database connection, default Doctrine stuff
	 * @return void
	 * @throws Exception Regular Doctrine stuff
	 */
	private function updateAfterOrderStatusMail(Connection $oConnection)
	{
		$oRes = $oConnection->executeQuery("SELECT HEX(mtt.mail_template_id) AS mail_template_id, HEX(mtt.language_id) AS language_id, mtt.content_html, mtt.content_plain 
		FROM mail_template_translation mtt 
		    INNER JOIN mail_template mt ON mtt.mail_template_id = mt.id 
		    INNER JOIN mail_template_type mttype ON mt.mail_template_type_id = mttype.id
		WHERE mttype.technical_name LIKE 'sCreditPlusState%' AND (NOT mttype.technical_name LIKE 'sCreditPlusStateBefore%')");
		$sOldContentPlain = $this->getOldContent('sCreditPlusState_plain');
		$sOldContentHtml = $this->getOldContent('sCreditPlusState_html');
		$sNewContentPlain = $this->getNewContent('sCreditPlusState_plain');
		$sNewContentHtml = $this->getNewContent('sCreditPlusState_html');
		$this->updateTemplatesWithParams($oConnection, $sOldContentPlain, $sOldContentHtml, $sNewContentPlain, $sNewContentHtml, $oRes);
	}

	/**
	 * @param Connection $oConnection Database connection, default Doctrine stuff
	 * @return void
	 * @throws Exception Regular Doctrine stuff
	 */
	private function updateBeforeOrderStatusMail(Connection $oConnection)
	{
		$oRes = $oConnection->executeQuery("SELECT HEX(mtt.mail_template_id) AS mail_template_id, HEX(mtt.language_id) AS language_id, mtt.content_html, mtt.content_plain 
		FROM mail_template_translation mtt 
		    INNER JOIN mail_template mt ON mtt.mail_template_id = mt.id 
		    INNER JOIN mail_template_type mttype ON mt.mail_template_type_id = mttype.id
		WHERE mttype.technical_name LIKE 'sCreditPlusStateBefore%'");
		$sOldContentPlain = $this->getOldContent('sCreditPlusState_beforeOrder_plain');
		$sOldContentHtml = $this->getOldContent('sCreditPlusState_beforeOrder_html');
		$sNewContentPlain = $this->getNewContent('sCreditPlusState_beforeOrder_plain');
		$sNewContentHtml = $this->getNewContent('sCreditPlusState_beforeOrder_html');
		$this->updateTemplatesWithParams($oConnection, $sOldContentPlain, $sOldContentHtml, $sNewContentPlain, $sNewContentHtml, $oRes);
		// Update buggy wrong values created by pre 1.2.21 updates
		$oRes = $oConnection->executeQuery("SELECT HEX(mtt.mail_template_id) AS mail_template_id, HEX(mtt.language_id) AS language_id, mtt.content_html, mtt.content_plain 
		FROM mail_template_translation mtt 
		    INNER JOIN mail_template mt ON mtt.mail_template_id = mt.id 
		    INNER JOIN mail_template_type mttype ON mt.mail_template_type_id = mttype.id
		WHERE mttype.technical_name LIKE 'sCreditPlusStateBefore%'");
		$this->updateTemplatesWithParams($oConnection, $sOldContentHtml, $sOldContentHtml, $sNewContentPlain, $sNewContentHtml, $oRes);
	}

	/**
	 * @param Connection $oConnection Database connection, default Doctrine stuff
	 * @param string|boolean $sOldContentPlain Old Plain content for mail templates (must match database value)
	 * @param string|boolean $sOldContentHtml Old HTML content for mail templates (must match database value)
	 * @param string|boolean $sNewContentPlain New Plain content for mail templates
	 * @param string|boolean $sNewContentHtml New HTML content for mail templates
	 * @param Result $oRes The mail templates to be updated
	 * @return void
	 * @throws Exception Regular Doctrine stuff
	 */
	private function updateTemplatesWithParams(Connection $oConnection, $sOldContentPlain, $sOldContentHtml, $sNewContentPlain, $sNewContentHtml, $oRes)
	{
		// Old and new texts may not be empty
		if ( !$sOldContentPlain ) {
			return;
		}
		if ( !$sOldContentHtml ) {
			return;
		}
		if ( !$sNewContentPlain ) {
			return;
		}
		if ( !$sNewContentHtml ) {
			return;
		}
		while ( $aRow = $oRes->fetchAssociative() ) {
			if ( trim(str_replace(["\r","\n"], ['',''], $aRow['content_html'])) !== trim(str_replace(["\r","\n"], ['',''], $sOldContentHtml)) ) {
				// Old content must be identical in the database
				continue;
			}
			if ( trim(str_replace(["\r","\n"], ['',''], $aRow['content_plain'])) !== trim(str_replace(["\r","\n"], ['',''], $sOldContentPlain)) ) {
				// Old content must be identical in the database
				continue;
			}
			$aUpdateParams = [
				'contentHtml' => $sNewContentHtml,
				'contentPlain' => $sNewContentPlain,
				'mailTemplateId' => $aRow['mail_template_id'],
				'languageId' => $aRow['language_id']
			];
			$oConnection->executeStatement("UPDATE mail_template_translation SET content_html = :contentHtml, content_plain = :contentPlain WHERE mail_template_id = UNHEX(:mailTemplateId) AND language_id = UNHEX(:languageId)", $aUpdateParams);
		}
	}

	public function updateDestructive(Connection $connection): void
	{
		// implement update destructive
	}

	private function getOldContent($sFileName) {
		$sDir = dirname(__DIR__).'/DefaultMails/Update_1_2_21/';
		if ( file_exists($sDir.$sFileName.'.html.twig') ) {
			return file_get_contents($sDir.$sFileName.'.html.twig');
		}
		return '';
	}

	private function getNewContent($sFileName) {
		$sDir = dirname(__DIR__).'/DefaultMails/';
		if ( file_exists($sDir.$sFileName.'.html.twig') ) {
			return file_get_contents($sDir.$sFileName.'.html.twig');
		}
		return '';
	}
}
