<?php
/**
 * Created by PhpStorm.
 * User: Philipp.Holzmann
 * Date: 30.01.2020
 * Time: 11:51
 */

namespace SicoCreditPlus\Subscriber;
use Shopware\Core\Checkout\Cart\AbstractCartPersister;
use Shopware\Core\Checkout\Payment\PaymentMethodEntity;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Storefront\Page\Checkout\Confirm\CheckoutConfirmPageLoadedEvent;
use SicoCreditPlus\Service\CreditPlusPayment;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use Symfony\Contracts\Translation\TranslatorInterface;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Struct\ArrayEntity;

class CheckoutConfirmSubscriber implements EventSubscriberInterface{

	/** @var SicoCreditPlusHelper */
	private $creditPlusHelper;

	/** @var TranslatorInterface  */
	private $translator;

	/** @var SystemConfigService  */
	private $systemConfigService;

	/** @var EntityRepository  */
	private $currencyRepo;

	/** @var EntityRepository */
	private $productRepo;

	/** @var EntityRepository  */
	private $paymentRepo;

	private $cartPersister;

	public function __construct(
		SicoCreditPlusHelper $creditPlusHelper,
		TranslatorInterface $translator,
		SystemConfigService $systemConfigService,
        EntityRepository $currencyRepo,
        EntityRepository $productRepo,
        EntityRepository $paymentRepo,
        AbstractCartPersister $cartPersister
	)
	{
		$this->creditPlusHelper = $creditPlusHelper;
		$this->translator = $translator;
		$this->systemConfigService = $systemConfigService;
		$this->currencyRepo = $currencyRepo;
		$this->productRepo = $productRepo;
		$this->paymentRepo = $paymentRepo;
		$this->cartPersister = $cartPersister;
	}

	/**
	 * Returns an array of event names this subscriber wants to listen to.
	 *
	 * The array keys are event names and the value can be:
	 *
	 *  * The method name to call (priority defaults to 0)
	 *  * An array composed of the method name to call and the priority
	 *  * An array of arrays composed of the method names to call and respective
	 *    priorities, or 0 if unset
	 *
	 * For instance:
	 *
	 *  * array('eventName' => 'methodName')
	 *  * array('eventName' => array('methodName', $priority))
	 *  * array('eventName' => array(array('methodName1', $priority), array('methodName2')))
	 *
	 * @return array The event names to listen to
	 */
	public static function getSubscribedEvents()
	{
		return [CheckoutConfirmPageLoadedEvent::class => 'checkoutConfirmPageLoaded'];
	}

	public function checkoutConfirmPageLoaded(CheckoutConfirmPageLoadedEvent $event){
		//Delete the Session Variable if the User is thrown back to the CheckoutConfirm Subscriber
		$event->getRequest()->getSession()->remove('cpDealerOrderNumber');

		$cart = $event->getPage()->getCart();
		$cart->getData()->remove('cpDealerOrderNumber');
		$this->cartPersister->save($cart,$event->getSalesChannelContext());

		$this->creditPlusHelper->setSalesChannelContext($event->getSalesChannelContext());
		/** @var PaymentMethodEntity $cpPayment */
		$cpPayment = $this->paymentRepo->search(
			(new Criteria())->addFilter(
				new EqualsFilter('handlerIdentifier', CreditPlusPayment::class))
			,$event->getContext())->first();
		$oCart = $event->getPage()->getCart();

		$aArticleNumbers = array();
		foreach($oCart->getLineItems() as $oLineItems){
			if($oLineItems->isGood()){
				$referencedId = $oLineItems->getReferencedId();
				if(!empty($referencedId)){
					$aArticleNumbers[] = $referencedId;
				}
			}
		}

		$aArticles = [];
		if ( $aArticleNumbers ) {
			/** @var ProductCollection $aArticles */
			$aArticles =  $this->productRepo->search(new Criteria($aArticleNumbers),$event->getContext())->getEntities();
		}

		//$oBasket = $event->getPage()->
		$oCurrency = $event->getSalesChannelContext()->getCurrency();
		$sCurrency = $event->getContext()->getCurrencyId();
		if($oCurrency == null){
			$oCurrency = $this->currencyRepo->search(new Criteria([$sCurrency]),$event->getContext())->get($sCurrency);
		}
		$oWSCurrency = $this->creditPlusHelper->getWebshopCurrency($oCurrency);

		$oReference = $this->creditPlusHelper->getFinancingArticleReference($event->getSalesChannelContext(),null,$oCart->getLineItems(),$oCart->getPrice()->getTotalPrice());
		if ( !$aArticles ) {
			$oReference = null;
		}
		$bSicoFinancingExcluded = $event->getRequest()->getSession()->get('bSicoFinancingExcluded',false);
		if($oReference == null || $bSicoFinancingExcluded){
			$transactionMode = $this->systemConfigService->get('SicoCreditPlus.config.sTransactionMode',$event->getSalesChannelContext()->getSalesChannelId());
			if(!($transactionMode == 'inorder') && $bSicoFinancingExcluded){
				$error = $this->translator->trans('sicoCreditplus.error.creditplusNotAvailable');
				$event->getRequest()->getSession()->getFlashBag()->add('danger',$error);
			}

			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
				'bShowPayment' => false,
				'cpPaymentId' => $cpPayment->getId(),
				'bEnablePaymentButton' => false
			]);

			$event->getPage()->addExtension('SicoCreditPlus',$entity);
			return;
		}
		if ( $oReference->financingMonths ) {
			$aMonths = $oReference->financingMonths;
		} else {
			$aMonths = $this->creditPlusHelper->getFinancingMonths($event->getSalesChannelContext(),$oCurrency, $aArticles->get($oReference->id),null,$oCart->getPrice()->getTotalPrice());
		}

		$dCheapestInterestRate = $this->creditPlusHelper->getCheapestInterestRate($aMonths);
		$oCheapestMonth = $this->creditPlusHelper->getCheapestRate($aMonths);
		$sMinimalOrderSum = number_format(floatval($this->systemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',$event->getSalesChannelContext()->getSalesChannelId())), $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator);
		$aReplacements = array(
			($oWSCurrency->side == 'left')?$oWSCurrency->sign.' '.$sMinimalOrderSum:$sMinimalOrderSum.' '.$oWSCurrency->sign,
			number_format($dCheapestInterestRate, $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator)
		);
		$sFinancingDescription = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingDescription');

		$aMarker = array(
			'###MINDESTBESTELLWERT###',
			'###JAHRESZINS###'
		);
		$sFinancingDescription = str_replace($aMarker, $aReplacements, $sFinancingDescription);

		$sCheapestRateText = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingCheapestRate');
		$aCheapestMarker = array(
			'###PRICE###',
			'###MONTHS###',
			'###LOGOLINK###'
		);
		$sBasePath = $event->getRequest()->getBaseUrl();
		$aCheapestReplacements = array(
			$oCheapestMonth->monthlyRate,
			$oCheapestMonth->months,
			$sBasePath.'bundles/sicocreditplus/img/creditplus_logo_small.png'
		);
		$sCheapestRateText = str_replace($aCheapestMarker, $aCheapestReplacements, $sCheapestRateText);

		$bShowPAngVText = ($this->translator->trans('sicoCreditplus.tabs.sccppangvtext_text') != '');
		$bShowPaymentSetting = intval($this->systemConfigService->get('SicoCreditPlus.config.bShowPayment',$event->getSalesChannelContext()->getSalesChannelId()));
		$bPaymentExcluded = (!$this->creditPlusHelper->isPaymentExcluded($event->getSalesChannelContext(), $oCart, $aMonths));
		$bShowPayment = $bShowPaymentSetting && $bPaymentExcluded;
		if($bShowPayment){
			$i = 0;
			$aSicoLowestRates = [];
			foreach($aMonths as $oMonth){
				if($i >= (count($aMonths) - 3)){
					$aSicoLowestRates = array_merge($aSicoLowestRates,[$oMonth]);
				}
				$i++;
			}
			$entity = new ArrayEntity([
				'aSicoRateTable' => $aMonths,
				'sSicoFinancingDescription'=> $sFinancingDescription,
				'sSicoCheapestRateText' => $aMonths?$sCheapestRateText:'',
				'bShowPAngVText' => $bShowPAngVText,
				'bShowPayment' => $bShowPayment,
				'bEnablePaymentButton' => $bPaymentExcluded,
				'aSicoLowestRates' => $aSicoLowestRates
			]);
		}
		else{
			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
				'bShowPayment' => $bShowPayment,
				'bEnablePaymentButton' => $bPaymentExcluded
			]);
		}
		$transactionMode = $this->systemConfigService->get('SicoCreditPlus.config.sTransactionMode',$event->getSalesChannelContext()->getSalesChannelId());
		$entity->assign(['cpPaymentId' => $cpPayment->getId()]);
		$entity->assign(['isInOrderMode' => ($transactionMode == 'inorder')]);
		$event->getPage()->addExtension('SicoCreditPlus',$entity);
	}
}
