<?php
/**
 * Created by PhpStorm.
 * User: philipp.holzmann
 * Date: 28.07.2021
 * Time: 12:28
 */
namespace SicoCreditPlus\Subscriber;
use Shopware\Core\Checkout\Cart\Order\CartConvertedEvent;
use Shopware\Core\Checkout\Cart\Order\OrderConvertedEvent;
use Shopware\Core\Checkout\Cart\Price\Struct\CartPrice;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\DataAbstractionLayer\Version\Aggregate\VersionCommitData\VersionCommitDataEntity;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopVoucher;
use SicoCreditPlus\Struct\CpUpdateOrder;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ControllerArgumentsEvent;
use Symfony\Component\HttpKernel\KernelEvents;

class OrderSubscriber implements EventSubscriberInterface{
	/** @var EntityRepository */
	protected $oOrderRepo;

	/** @var EntityRepository */
	protected $oSicoPaymentRepo;

	/** @var SicoCreditPlusLogger */
	protected $oLogger;

	/** @var SicoCreditPlusHelper */
	protected $oSicoCreditPlusHelper;

    protected $oVersionCommitRepo;
    protected $oVersionCommitDataRepo;

	/**
	 * OrderSubscriber constructor.
	 * @param EntityRepository $oOrderRepo
	 * @param EntityRepository $oSicoPaymentRepo
	 * @param SicoCreditPlusLogger $oLogger
	 * @param SicoCreditPlusHelper $oSicoCreditPlusHelper
	 */
	public function __construct(
        EntityRepository $oOrderRepo,
        EntityRepository $oSicoPaymentRepo,
        SicoCreditPlusLogger $oLogger,
        SicoCreditPlusHelper $oSicoCreditPlusHelper,
        EntityRepository $oVersionCommitRepo,
        EntityRepository $oVersionCommitDataRepo
    )
	{
		$this->oOrderRepo = $oOrderRepo;
		$this->oSicoPaymentRepo = $oSicoPaymentRepo;
		$this->oLogger = $oLogger;
		$this->oSicoCreditPlusHelper = $oSicoCreditPlusHelper;
        $this->oVersionCommitRepo = $oVersionCommitRepo;
        $this->oVersionCommitDataRepo = $oVersionCommitDataRepo;
	}


	/**
	 * @return string[] Array of 'Event Class' => 'subscriber function'
	 */
	public static function getSubscribedEvents()
	{
		return [
            OrderConvertedEvent::class => 'orderConvertEvent',
            /* In the Recalculation Service Shopware convert the order to cart recalulate it and convert it then back to order.*/
            CartConvertedEvent::class => 'cartConvertEvent',
            KernelEvents::CONTROLLER_ARGUMENTS => 'beforeMerge'//I want to take a bath afterwards
		];
	}

    public function orderConvertEvent(OrderConvertedEvent $event){
        $oOrder = $event->getOrder();
        $oCart = $event->getConvertedCart();
        $oCart->addExtension('cpUpdateOrder',new CpUpdateOrder($oOrder->getPrice()->getTotalPrice()));
    }

    public function cartConvertEvent(CartConvertedEvent $event){
        $oCart = $event->getCart();
        $this->oSicoCreditPlusHelper->setContext($event->getContext());
        /*If this ist true then the event is fired in the checkout process and not in the recaluclation process*/
        if($event->getConversionContext()->shouldIncludeOrderDate() == true){
            $oCart->removeExtension('cpUpdateOrder');//Remove Extension if it called on checkout
            return;
        }

        if(!$oCart->hasExtension('cpUpdateOrder')){
            return;
        }
        $aConvertedOrder = $event->getOriginalConvertedCart();
        $sOrderId = $aConvertedOrder['id'];

        /** @var OrderEntity $oOrder */
        $oOrder = $this->fetchOrder($sOrderId, $event->getContext());

        /** @var CartPrice $oPrice */
        $oPrice = $aConvertedOrder['price'];
        $oOrder->setPrice($oPrice);


        $aPaymentMethodsIds = $oOrder->getTransactions()->getPaymentMethodIds();
        $sCreditPlusPaymentId = $this->oSicoCreditPlusHelper->getCreditPlusPaymentMethodId($event->getContext());
        //Order is not a CreditPlus Order
        if(!in_array($sCreditPlusPaymentId,$aPaymentMethodsIds)){
            return;
        }

        $oContractData = $this->oSicoCreditPlusHelper->getContractData($oOrder->getSalesChannelId(),$oOrder);

		if ( !$oContractData || !$oContractData->cpReferenceNumber ) {
			throw new \Exception('Die Bestellung '.$oOrder->getOrderNumber().' wurde noch nicht bei CreditPlus abgeschlossen.');
		}
        $dInvoiceAmount = (float)$oContractData->price;
        if($oContractData->voucher){
            foreach($oContractData->voucher as $oVoucher){
                $dInvoiceAmount -= $oVoucher->value;
            }
        }

        $dNewSum = $oOrder->getPrice()->getTotalPrice();
        $dReturnAmount = $dInvoiceAmount-$dNewSum;

        if($dReturnAmount == 0){
            return;
        }
        if($dReturnAmount < 0){
            throw new \Exception('Bei CreditPlus Bestellungen darf der Preis nicht höher werden');
        }
    }

    private function fetchOrder(string $orderId, Context $context)
    {
        $criteria = (new Criteria([$orderId]))
            ->addAssociation('lineItems')
            ->addAssociation('transactions')
            ->addAssociation('deliveries.shippingMethod')
            ->addAssociation('deliveries.positions.orderLineItem')
            ->addAssociation('deliveries.shippingOrderAddress.country')
            ->addAssociation('deliveries.shippingOrderAddress.countryState');

        return $this->oOrderRepo
            ->search($criteria, $context)
            ->get($orderId);
    }





    // LaLeLu Ich bastle mir ein preDispatch event zu sammen. Davor weil danach die version id es danach nicht mehr gibt
    public function beforeMerge(ControllerArgumentsEvent $event){
        $requestUri = $event->getRequest()->getRequestUri();
        if(preg_match('/\/api\/_action\/version\/merge\/order\/[0-9a-f]{32}/',$requestUri) == 0){
            return;
        }
        $arguments = $event->getArguments();//Array with [Context,Entity,VersionId]
        $context = $arguments[0];
        $versionId = $arguments[2];

        $this->oSicoCreditPlusHelper->setContext($context);

        $versionCommitCriteria = new Criteria();
        $versionCommitCriteria->addFilter(new EqualsFilter('versionId', $versionId));
        $versionCommitCriteria->addSorting(new FieldSorting('autoIncrement'));
        $versionCommitCriteria->addAssociation('version_commit_data');
        $oVersionCommitIds = $this->oVersionCommitRepo->search($versionCommitCriteria,$context)->getIds();
        $versionCommitDataCriteria = new Criteria();
        $versionCommitDataCriteria->addFilter(new EqualsAnyFilter('versionCommitId',$oVersionCommitIds));
        $oVersionCommitDataCollection = $this->oVersionCommitDataRepo->search($versionCommitDataCriteria,$context)->getEntities();

        /** @var VersionCommitDataEntity $oVersionCommitDataEntity */
        foreach($oVersionCommitDataCollection as $oVersionCommitDataEntity){
            if($oVersionCommitDataEntity->getEntityName() != 'order'){
                continue;
            }
            $aPayload = $oVersionCommitDataEntity->getPayload();
            if($aPayload != null && isset($aPayload['price'])){
                /** @var OrderEntity $oOrder */
                $oOrder = $this->fetchOrder($aPayload['id'],$context);
                if(!$oOrder){
                    continue;
                }
                $fOldPrice = $oOrder->getPrice()->getTotalPrice();
                $fNewPrice = $aPayload['price']['totalPrice'];

                if($fNewPrice == null){
                    continue;
                }
                $aPaymentMethodsIds = $oOrder->getTransactions()->getPaymentMethodIds();
                $sCreditPlusPaymentId = $this->oSicoCreditPlusHelper->getCreditPlusPaymentMethodId($context);
                //Order is not a CreditPlus Order
                if(!in_array($sCreditPlusPaymentId,$aPaymentMethodsIds)){
                    return;
                }
                $oContractData = $this->oSicoCreditPlusHelper->getContractData($oOrder->getSalesChannelId(),$oOrder);
                $dInvoiceAmount = (float)$oContractData->price;
                if($oContractData->voucher){
                    foreach($oContractData->voucher as $oVoucher){
                        $dInvoiceAmount -= $oVoucher->value;
                    }
                }
                $dReturnAmount = $dInvoiceAmount-$fNewPrice;

                if($dReturnAmount == 0){
                    continue;
                }
                if($dReturnAmount < 0){
                    throw new \Exception('Bei CreditPlus Bestellungen darf der Preis nicht höher werden');
                }

                $this->updateOrder(
                    $fNewPrice,
                    $oOrder->getSalesChannelId(),
                    $dReturnAmount,
                    $oOrder);
                return;
            }

        }

    }

    /**
     * @param $newPrice
     * @param $sSalesChannelId
     * @param $dReturnAmount
     * @param OrderEntity $oOrder
     * @return void
     */
    protected function updateOrder($newPrice,$sSalesChannelId,$dReturnAmount,$oOrder){
        $oContractData = $this->oSicoCreditPlusHelper->getContractData($sSalesChannelId,$oOrder);
        $sDate = date('c');
        $oWSApi = $this->oSicoCreditPlusHelper->getWebshopAPI($sSalesChannelId);
        $sUser = 'user1';//$event->getCo

        $aOrderData = array(
            'dealerOrderNumber' => $oContractData->dealerOrderNumber,
            'changeDate' => $sDate,
            'changedBy' => $sUser,
            'dealerNumber' => $oContractData->dealerNumber,
            'loanAmount' => $newPrice,
            'cpReferenceNumber' => $oContractData->cpReferenceNumber
        );

        $oResponse = $oWSApi->changeOrderCPWebshop($aOrderData);
        if ( is_object($oResponse) && ( property_exists($oResponse,'confirmation') ) && ( property_exists($oResponse->confirmation,'confirmationItems') )  && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
            $iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
            $sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
            $aError = array(
                'sError' => (500+$iErrorCode),
                'sErrorMessage' => $sErrorMessage
            );
            if ( $iErrorCode == 1 ) {
                // Kein Auftrag gefunden
            } elseif ( $iErrorCode == 2 ) {
                // Pflichtfelder fehlen, kann durch Code heraus nicht passieren
            } elseif ( ($iErrorCode == 6) || ($iErrorCode == 18) || ($iErrorCode == 11)  || ($iErrorCode == 15) ) {
                // changeOrder im Status "Bezahlt", "In Bezahlung", "Ausgeliefert" oder "Lieferfreigabe" kann nicht stattfinden
                $oVoucher = new WebshopVoucher($dReturnAmount,'Warenkorbmodifikation',$sDate,$sUser,false,$oContractData->id,$oContractData->dealerOrderNumber,$oContractData->dealerNumber);
                $oWSApi->addReturnProduct($oVoucher);

                $oResponse = $oWSApi->returnProductCPWebshop();
                // Fehlerhandling
                if ( is_object($oResponse) && ( property_exists($oResponse,'confirmation') ) && ( property_exists($oResponse->confirmation,'confirmationItems') )  && ( property_exists($oResponse->confirmation->confirmationItems,'errorCode') ) ) {
                    $iErrorCode = $oResponse->confirmation->confirmationItems->errorCode;
                    $sErrorMessage = $oResponse->confirmation->confirmationItems->errorMessage;
                    $aError = array(
                        'sError' => (500+$iErrorCode),
                        'sErrorMessage' => $sErrorMessage
                    );
                } else {
                    $aError = array();
                }
            } elseif ( $iErrorCode == 12 ) {
                // Summe der Retouren übersteigt Restwert des Kreditvertrags
            } elseif ( $iErrorCode == 14 ) {
                // Irgendein Pflichtfeld sprengt sein Format/Länge
                // dealerOrderNumber > 40
                // cancelationFrom > 27
            } else {
                // Keine Ahnung was passiert ist
            }
        } else {
            $aError = array();
        }
		//return $aError;
		if(!empty($aError)){
			$this->oLogger->log(true, 'Error while updating Order', $aError);
		}
	}
}
