<?php
/**
 * Created by PhpStorm.
 * User: Philipp.Holzmann
 * Date: 16.01.2020
 * Time: 09:26
 */

namespace SicoCreditPlus\Subscriber;

use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Storefront\Page\Product\ProductPageCriteriaEvent;
use SicoCreditPlus\Core\Content\SicoCreditPlusCalculatedRate\SicoCreditPlusCalculatedRateCollection;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Struct\ArrayEntity;
use Shopware\Core\System\Snippet\SnippetService;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use Symfony\Contracts\Translation\TranslatorInterface;

class ProductDetailSubscriber implements EventSubscriberInterface{

	/** @var SicoCreditPlusHelper */
	private $creditPlusHelper;

	/** @var SnippetService  */
	private $translator;

	/** @var SystemConfigService  */
	private $systemConfigService;

	/** @var EntityRepository  */
	private $currencyRepo;

	public function __construct(
		SicoCreditPlusHelper $creditPlusHelper,
		TranslatorInterface $translator,
		SystemConfigService $systemConfigService,
        EntityRepository $currencyRepo
		)
	{
		$this->creditPlusHelper = $creditPlusHelper;
		$this->translator = $translator;
		$this->systemConfigService = $systemConfigService;
		$this->currencyRepo = $currencyRepo;
	}

	/**
	 * Returns an array of event names this subscriber wants to listen to.
	 *
	 * The array keys are event names and the value can be:
	 *
	 * * The method name to call (priority defaults to 0)
	 * * An array composed of the method name to call and the priority
	 * * An array of arrays composed of the method names to call and respective
	 *   priorities, or 0 if unset
	 *
	 * For instance:
	 *
	 * * array('eventName' => 'methodName')
	 * * array('eventName' => array('methodName', $priority))
	 * * array('eventName' => array(array('methodName1', $priority), array('methodName2'))
	 *
	 * @return array The event names to listen to
	 */
	public static function getSubscribedEvents()
	{
		return [
			ProductPageCriteriaEvent::class => 'productPageCriteria',
			ProductPageLoadedEvent::class => 'productPageLoaded',
		];
	}

	public function productPageCriteria(ProductPageCriteriaEvent $oEvent): void
	{

		$oCriteria = $oEvent->getCriteria();
		$oCriteria->addAssociation('sicoCreditPlusCalculatedRates'); // Just add the association
	}

	public function productPageLoaded(ProductPageLoadedEvent $event){
        $sRateTableShowMode = $this->systemConfigService->get('SicoCreditPlus.config.sRateTableMode',$event->getSalesChannelContext()->getSalesChannelId());
        $iRateTableShowMode = (int) $sRateTableShowMode;
		$this->creditPlusHelper->setSalesChannelContext($event->getSalesChannelContext());
		$oProduct = $event->getPage()->getProduct();
		$oCurrency = $event->getSalesChannelContext()->getCurrency();
		$sCurrency = $event->getContext()->getCurrencyId();
		$oPrice = $oProduct->getPrice();
		if ( $aPrices = $oProduct->getPrices() ) {
			foreach ( $aPrices as $oPricePossibility ) {
				if ( $oPricePossibility->getQuantityStart() <= 1 ) {
					$oPrice = $oPricePossibility->getPrice();
					break;
				}
			}
		}
		if($oCurrency == null){
			$oCurrency = $this->currencyRepo->search(new Criteria([$sCurrency]),$event->getContext())->get($sCurrency);
		}
		$oWSCurrency = $this->creditPlusHelper->getWebshopCurrency($oCurrency);
		if($oPrice){
			$dPrice = $oPrice->getCurrencyPrice($sCurrency)->getGross();//$this->reFloatPrice($oPrice->getCurrencyPrice());
			//Hier dürfen keine Monate zurückgegeben werden, wenn der Kredit nur auf ein oder mehrere Produkte beschränkt ist.
			$aMonths = $this->creditPlusHelper->getFinancingMonths($event->getSalesChannelContext(),$oCurrency, $oProduct,null,$dPrice);
		}
		else {
			$aMonths = $this->creditPlusHelper->getFinancingMonths($event->getSalesChannelContext(),$oCurrency, $oProduct);
		}
		$dCheapestInterestRate = 0.0;
		$oCheapestMonth = null;
		// Use the cache instead of recalculation for detail page as well
		/** @var SicoCreditPlusCalculatedRateCollection $oCalculatedRates */
		if ( $oProduct->has('sicoCreditPlusCalculatedRates') && $oCalculatedRates = $oProduct->get('sicoCreditPlusCalculatedRates') ) {
			foreach ( $oCalculatedRates as $oCalculatedRate ) {
				if ( $oCalculatedRate->getSicoCalculationFinished() !== 1 ) {
					// Cache is only ok, if it is current
					continue;
				}
				if ( $oCalculatedRate->getSicoSalesChannelId() !== $event->getSalesChannelContext()->getSalesChannelId() ) {
					// Data for other sales channels don't matter
					continue;
				}
				if (
					$oCalculatedRate->getSicoMinMonths() === -1 &&
					$oCalculatedRate->getSicoMaxMonths() === -1 &&
					$oCalculatedRate->getSicoAbsoluteMinRateMonths() === -1
				) {
					continue;
				}
				$dCheapestInterestRate = $oCalculatedRate->getSicoAbsoluteMinRateInterestRate();
				$oCheapestMonth = new \stdClass();
				$oCheapestMonth->monthlyRate = $oCalculatedRate->getSicoAbsoluteMinRateRate();
				$oCheapestMonth->months = $oCalculatedRate->getSicoAbsoluteMinRateMonths();
			}
		}
		// If the cache is empty, calculate new values on the fly
		if ( $oCheapestMonth === null ) {
			$dCheapestInterestRate = $this->creditPlusHelper->getCheapestInterestRate($aMonths);
			$oCheapestMonth = $this->creditPlusHelper->getCheapestRate($aMonths);
		}
		if($oCheapestMonth == null){ // Product is too cheap for creditplus
			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
                'iRateTableMode' => $iRateTableShowMode
			]);
			$event->getPage()->addExtension('SicoCreditPlus',$entity);
			return;
		}
		$sMinimalOrderSum = number_format(floatval(
            $this->systemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',
            $event->getSalesChannelContext()->getSalesChannelId())
        ), $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator);
		$aReplacements = array(
			($oWSCurrency->side == 'left')?$oWSCurrency->sign.' '.$sMinimalOrderSum:$sMinimalOrderSum.' '.$oWSCurrency->sign,
			number_format($dCheapestInterestRate, $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator)
		);
		$sFinancingDescription = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingDescription');

		$aMarker = array(
			'###MINDESTBESTELLWERT###',
			'###JAHRESZINS###'
		);
		$sFinancingDescription = str_replace($aMarker, $aReplacements, $sFinancingDescription);

		$sCheapestRateText = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingCheapestRate');
		$aCheapestMarker = array(
			'###PRICE###',
			'###MONTHS###',
			'###LOGOLINK###'
		);

		$sBasePath = $event->getSalesChannelContext()->getSalesChannel()->getDomains()->first()->getUrl();
		if ( substr($sBasePath, -1) !== '/' ) {
			$sBasePath .= '/';
		}
		$aCheapestReplacements = array(
			$oCheapestMonth->monthlyRate,
			$oCheapestMonth->months,
			$sBasePath.'bundles/sicocreditplus/img/creditplus_logo_small.png'
		);
		$sCheapestRateText = str_replace($aCheapestMarker, $aCheapestReplacements, $sCheapestRateText);

		$bShowPAngVText = ($this->translator->trans('sicoCreditplus.tabs.sccppangvtext_text') != '');
		$bShowDetails = intval($this->systemConfigService->get('SicoCreditPlus.config.bShowDetails',$event->getSalesChannelContext()->getSalesChannel()->getId()));
		if($bShowDetails){
			$i = 0;
			$aSicoLowestRates = [];
            if(count($aMonths) >= 3) {
                foreach ($aMonths as $oMonth) {
                    if ($i >= (count($aMonths) - 3)) {
                        $aSicoLowestRates = array_merge($aSicoLowestRates, [$oMonth]);
                    }
                    $i++;
                }
            } else {
                $aSicoLowestRates = $aMonths;
            }
			$entity = new ArrayEntity([
				'aSicoRateTable' => $aMonths,
				'sSicoFinancingDescription'=> $sFinancingDescription,
				'sSicoCheapestRateText' => $aMonths?$sCheapestRateText:'',
				'bShowPAngVText' => $bShowPAngVText,
				'aSicoLowestRates' => $aSicoLowestRates,
                'iRateTableMode' => $iRateTableShowMode
			]);
		}
		else{
			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
                'iRateTableMode' => $iRateTableShowMode
			]);
		}

		$event->getPage()->addExtension('SicoCreditPlus',$entity);
	}
}
