<?php declare(strict_types=1);

namespace SicoCreditPlus;

use Shopware\Core\Content\MailTemplate\Aggregate\MailTemplateType\MailTemplateTypeDefinition;
use Shopware\Core\Content\MailTemplate\MailTemplateCollection;
use Shopware\Core\Content\MailTemplate\MailTemplateDefinition;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin;
use Shopware\Core\Framework\Plugin\Context\ActivateContext;
use Shopware\Core\Framework\Plugin\Context\DeactivateContext;
use Shopware\Core\Framework\Plugin\Context\InstallContext;
use Shopware\Core\Framework\Plugin\Context\UninstallContext;
use Shopware\Core\Framework\Plugin\Context\UpdateContext;
use Shopware\Core\Framework\Plugin\Util\PluginIdProvider;
use Shopware\Core\System\SalesChannel\SalesChannelDefinition;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use Shopware\Core\System\Snippet\Aggregate\SnippetSet\SnippetSetDefinition;
use Shopware\Core\System\Snippet\SnippetDefinition;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateDefinition;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateEntity;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineTransition\StateMachineTransitionDefinition;
use Shopware\Core\System\StateMachine\StateMachineCollection;
use Shopware\Core\System\StateMachine\StateMachineDefinition;
use SicoCreditPlus\Service\CreditPlusPayment;
use SicoCreditPlus\Service\PluginActivateService;
use Symfony\Contracts\Service\Attribute\Required;


class SicoCreditPlus extends Plugin
{
	/** @var string[] md5(index) = value, but it is used more than once, so it's precalculated for performance reasons. */
	public const ID_STATE = [
		'borrowing_request_shown' => 'c998b360c9a6ee34a63ec91ff69705af',
		'creditplus_referred' => '0fa7b9db294d262c0e055dff3a77dadc',
		'creditplus_accepted' => 'd3405bab5924d03b64f5986b0e903fe0',
		'creditplus_approved' => '3a863adfbf23dfd3df218c8f56cff820',
		'creditplus_approved_and_dispatched' => '808fb83d24d8fee2f4a233102c42a7b9',
		'creditplus_processing_payment' => '5b6a24fbf4e9969b7648ffa4b0265502',
		'creditplus_paid' => 'a4327c34479168ab8f98221848889ac5',
		'creditplus_declined_soft' => '1b138f7fb7a356363dd75d3f14b5d855',
		'creditplus_declined_hard' => 'b91d4ad54b32ffd6d2b8e14fd56fed36',
		'creditplus_cancelled' => 'd7e38be944b690a9d41ac4e7210ced68',
		'review_necessary' => 'dcd49c6ec507011aea53b3be895fd446',
		'creditplus_error' => 'ae1012dba2054815d066c839850775a2',
		'creditplus_docs_received' => '6a4543dd18c6498fc54db6c21ef8f910',
		'no_credit_approved' => '147fd35450652b35c76991e5a099bb77',
	];

    private PluginActivateService $pluginActivateService;

    #[Required]
    public function setActivateDeactivate(PluginActivateService $pluginActivateService): void
    {
        $this->pluginActivateService = $pluginActivateService;
    }

    public function install(InstallContext $installContext): void
    {
        parent::install($installContext);
        $this->addPaymentMethod($installContext->getContext());
        $this->updateBackendStaticPaymentStatusSnippets($installContext->getContext());
        //$this->deleteMailTemplates($installContext->getContext());
        $this->setMailTemplates($installContext->getContext());
    }

    public function activate(ActivateContext $activateContext): void
    {
        parent::activate($activateContext);
        $this->pluginActivateService->pluginActivate($activateContext->getContext());
    }

    public function deactivate(DeactivateContext $deactivateContext): void
    {
        parent::deactivate($deactivateContext); // TODO: Change the autogenerated stub
        $this->setPaymentMethodIsActive($deactivateContext->getContext());
    }

	public function uninstall(UninstallContext $uninstallContext): void
	{
		// Only set the payment method to inactive when uninstalling. Removing the payment method would
		// cause data consistency issues, since the payment method might have been used in several orders
		$this->setPaymentMethodIsActive($uninstallContext->getContext());
		$this->deleteMailTemplates($uninstallContext->getContext());
		if(!$uninstallContext->keepUserData()){
			//ToDO: Eventuell sico_productgroup,sico_offered_option,sico_productgroup_s_articles und sico_credit_plus_payment löschen
		}

    }

    public function postUpdate(UpdateContext $updateContext): void
    {
        parent::postUpdate($updateContext); // TODO: Change the autogenerated stub
        $this->updateBackendStaticPaymentStatusSnippets($updateContext->getContext());
    }


    private function addPaymentMethod(Context $context): void
    {
        $paymentMethodExists = $this->getPaymentMethodId($context);

        /** @var PluginIdProvider $pluginIdProvider */
        $pluginIdProvider = $this->container->get(PluginIdProvider::class);
        $pluginId = $pluginIdProvider->getPluginIdByBaseClass(self::class, $context);

        // Payment method exists already, no need to continue here
        if ($paymentMethodExists) {
            /** @var EntityRepository $paymentRepository */
            $paymentRepository = $this->container->get('payment_method.repository');
            $paymentRepository->update([
                ['id' => $paymentMethodExists,
                    'handlerIdentifier' => CreditPlusPayment::class,
                    'technicalName' => 'sico_credit_plus_financing',
                    'name' => 'Creditplus Finanzierung',
                    'description' => 'Bezahlen Sie Ihren Einkauf schnell und bequem über monatlich niedrige Raten mit der Creditplus Finanzierung',
                    'pluginId' => $pluginId
                ]
            ], $context);
            return;
        }


        $examplePaymentData = [
            'id' => md5('creditplus'),
            // payment handler will be selected by the identifier
            'handlerIdentifier' => CreditPlusPayment::class,
            'technicalName' => 'sico_credit_plus_financing',
            'name' => 'Creditplus Finanzierung',
            'description' => 'Bezahlen Sie Ihren Einkauf schnell und bequem über monatlich niedrige Raten mit der Creditplus Finanzierung',
            'pluginId' => $pluginId
        ];

        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');
        $paymentRepository->create([$examplePaymentData], $context);
    }

	/**
	 * Sets the payment methods technical name if payment method exists.
	 * Called from the update function.
	 *
	 * @param Context $context Standard context used in a lot of repository searches
	 * @return void
	 */
	private function setPaymentMethodTechnicalName(Context $context): void
	{
		$paymentMethodId = $this->getPaymentMethodId($context);
		// Payment method exists already, no need to continue here
		if ($paymentMethodId) {
			/** @var PluginIdProvider $pluginIdProvider */
			$pluginIdProvider = $this->container->get(PluginIdProvider::class);
			$pluginId = $pluginIdProvider->getPluginIdByBaseClass(self::class, $context);
			/** @var EntityRepository $paymentRepository */
			$paymentRepository = $this->container->get('payment_method.repository');
			$paymentRepository->update([
				[
					'id' => $paymentMethodId,
					'technicalName' => 'sico_credit_plus_financing',
					'pluginId' => $pluginId
				]
			], $context);
		}
	}


    /**
     * For a unkown reason the file will not be copied to the right place. This is a workaround for this problem.
     * @param Context $context
     * @return string|null
     */


    private function getPaymentMethodId(Context $context): ?string
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');

        // Fetch ID for update
        $paymentCriteria = (new Criteria())->addFilter(new EqualsFilter('handlerIdentifier', CreditPlusPayment::class));
        $paymentIds = $paymentRepository->searchIds($paymentCriteria, $context);
        if ($paymentIds->getTotal() === 0) {
            return null;
        }
        return $paymentIds->getIds()[0];
    }

    private function setPaymentMethodIsActive(Context $context): void
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');
        $paymentMethodId = $this->getPaymentMethodId($context);
        // Payment does not even exist, so nothing to (de-)activate here
        if (!$paymentMethodId) {
            return;
        }

        $paymentMethod = [
            'id' => $paymentMethodId,
            'active' => false
        ];

        $paymentRepository->update([$paymentMethod], $context);
    }


    public function setMailTemplates(Context $context): void
    {
        /** @var EntityRepository $oSalesChannelRepo */
        $oSalesChannelRepo = $this->container->get(SalesChannelDefinition::ENTITY_NAME . '.repository');
        /** @var SalesChannelEntity $oSalesChannelEntity */
        $oSalesChannelEntity = $oSalesChannelRepo->search(
            (new Criteria())->addFilter(
                new EqualsFilter('typeId', Defaults::SALES_CHANNEL_TYPE_STOREFRONT)
            ), $context)->first();
        $statusMails = [
            'sCreditPlusState20',
            'sCreditPlusState24n',
            'sCreditPlusState24a',
            'sCreditPlusState25',
            'sCreditPlusState92',
            'sCreditPlusState95',
            'sCreditPlusState99',
        ];
        $mails = [];

        $plain = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_plain.html.twig');
        $html = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_html.html.twig');
        $mails[] = [
            'id' => md5('sCreditplusOrderAdditionalMessage'),
            'name' => 'sCreditplusOrderAdditionalMessage',
            'mailTemplateType' => [
                'id' => md5('sCreditplusOrderAdditionalMessage_type'),
                'technicalName' => 'sCreditplusOrderAdditionalMessage',
                'name' => 'sCreditplusOrderAdditionalMessage',
                'availableEntities' => ['order' => 'order', 'sTargetURL' => 'string']
            ],
            'systemDefault' => false,
            'senderName' => '{{ salesChannel.name }}',
            'description' => 'Bestellmail für CreditPlus',
            'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
            'contentHtml' => $html,
            'contentPlain' => $plain
        ];

        $plain = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_20_plain.html.twig');
        $html = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_20_html.html.twig');
        $mails[] = [
            'id' => md5('sCreditplusOrderAdditionalMessage20'),
            'name' => 'sCreditplusOrderAdditionalMessage20',
            'mailTemplateType' => [
                'id' => md5('sCreditplusOrderAdditionalMessage_type'),
                'technicalName' => 'sCreditplusOrderAdditionalMessage',
                'name' => 'sCreditplusOrderAdditionalMessage',
                'availableEntities' => ['order' => 'order', 'sTargetURL' => 'string']
            ],
            'systemDefault' => false,
            'senderName' => '{{ salesChannel.name }}',
            'description' => 'Bestellmail für CreditPlus',
            'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
            'contentHtml' => $html,
            'contentPlain' => $plain
        ];

        $plain = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditPlusState_plain.html.twig');
        $html = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditPlusState_html.html.twig');
        foreach ($statusMails as $statusMail) {
            $mails[] = [
                'id' => md5($statusMail),
                'name' => $statusMail,
                'mailTemplateType' => [
                    'id' => md5($statusMail . '_type'),
                    'technicalName' => $statusMail,
                    'name' => $statusMail,
                    'availableEntities' => ['order' => 'order']
                ],
                'systemDefault' => false,
                'senderName' => '{{ salesChannel.name }}',
                'description' => 'Statusänderung der CreditPlus Bestellung',
                'subject' => 'Status der Bestellung im {{ salesChannel.name }}',
                //'availableEntities' => ['order' => 'order'],//'{"order":"order"}',
                'contentHtml' => $html,
                'contentPlain' => $plain
            ];
        }
        $plain = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditPlusRetryErrors_plain.html.twig');
        $html = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditPlusRetryErrors_html.html.twig');
        $mails[] = [
            'id' => md5('sCreditPlusRetryErrors'),
            'name' => 'sCreditPlusRetryErrors',
            'mailTemplateType' => [
                'id' => md5('sCreditPlusRetryErrors_type'),
                'technicalName' => 'sCreditPlusRetryErrors',
                'name' => 'sCreditPlusRetryErrors',
                'availableEntities' => ['order' => 'order']
            ],
            'systemDefault' => false,
            'senderName' => '{{ salesChannel.name }}',
            'description' => 'Fehler bei CreditPlus',
            'subject' => 'Fehlerhafte CreditPlus Status Updates in {{ salesChannel.name }}',
            'contentHtml' => $html,
            'contentPlain' => $plain
        ];

        /** @var EntityRepository $mailTemplateRepository */
        $mailTemplateRepository = $this->container->get(MailTemplateDefinition::ENTITY_NAME . '.repository');
        $mailTemplateRepository->upsert($mails, $context);

        $ids = [md5('sORDERForCreditPlus')];
        $ids[] = md5('sCreditPlusRetryErrors');
        foreach ($statusMails as $statusMail) {
            $ids[] = md5($statusMail);
        }

        $criteria = new Criteria($ids);
        $criteria->addAssociation('salesChannels');
        /** @var MailTemplateCollection $oMailtemplates */
        $oMailtemplates = $mailTemplateRepository->search($criteria, $context);
        foreach ($oMailtemplates as $oMailtemplate) {

            if (method_exists($oMailtemplate, 'getSalesChannels')) {
                foreach ($oMailtemplate->getSalesChannels() as $salesChannel) {
                    if ($salesChannel->getId() == $oSalesChannelEntity->getId()) {
                        continue 2;
                    }
                }
            }

            $updateSalesChannel = [
                'id' => $oMailtemplate->getId(),
                'salesChannels' => [[
                    'mailTemplateId' => $oMailtemplate->getId(),
                    'mailTemplateTypeId' => $oMailtemplate->getMailTemplateTypeId(),
                    'salesChannelId' => $oSalesChannelEntity->getId()
                ]]
            ];

            $mailTemplateRepository->upsert([$updateSalesChannel], $context);
        }
    }

    public function deleteMailTemplates($context): void
    {
        /** @var EntityRepository $oMailTemplateTypeRepo */
        $oMailTemplateTypeRepo = $this->container->get(MailTemplateTypeDefinition::ENTITY_NAME . '.repository');
        /** @var EntityRepository $oMailTemplateRepo */
        $oMailTemplateRepo = $this->container->get(MailTemplateDefinition::ENTITY_NAME . '.repository');

        $statusMails = [
            'sCreditPlusState20',
            'sCreditPlusState24n',
            'sCreditPlusState24a',
            'sCreditPlusState25',
            'sCreditPlusState92',
            'sCreditPlusState95',
            'sCreditPlusState99',
        ];
        $idsMailTemplates = [];
        $idsMailTemplates[] = ['id' => md5('sORDERForCreditPlus')];
        $idsMailTemplatesTypes = [];
        $idsMailTemplatesTypes[] = ['id' => md5('sORDERForCreditPlus_type')];

        $idsMailTemplates[] = ['id' => md5('sCreditPlusRetryErrors')];
        $idsMailTemplatesTypes[] = ['id' => md5('sCreditPlusRetryErrors_type')];

        $idsMailTemplates[] = ['id' => md5('sCreditplusOrderAdditionalMessage')];
        $idsMailTemplatesTypes[] = ['id' => md5('sCreditplusOrderAdditionalMessage_type')];
        foreach ($statusMails as $statusMail) {
            $idsMailTemplates[] = ['id' => md5($statusMail)];
            $idsMailTemplatesTypes[] = ['id' => md5($statusMail . '_type')];
        }

        $oMailTemplateRepo->delete($idsMailTemplates, $context);

        $oMailTemplateTypeRepo->delete($idsMailTemplatesTypes, $context);
    }

    public function updateBackendStaticPaymentStatusSnippets(Context $context)
    {
        /** @var EntityRepository $stateMachineRepo */
        $stateMachineRepo = $this->container->get(StateMachineDefinition::ENTITY_NAME . '.repository');
        /** @var StateMachineCollection $stateMachineEntites */
        $stateMachineEntites = $stateMachineRepo->search((new Criteria())->addFilter(new EqualsFilter('technicalName', 'order_transaction.state')), $context)->getEntities();
        $sStateMachineEntitesId = $stateMachineEntites->first()->getId();
        // Sorted like self::ID_STATE
        $aBackendTranslations = array(
            'borrowing_request_shown' => 'Antragsformular liegt dem Kunden vor',
            'creditplus_referred' => 'Antrag ausgefüllt',
            'creditplus_accepted' => 'Antrag angenommen',
            'creditplus_approved' => 'Lieferfreigabe',
            'creditplus_approved_and_dispatched' => 'Ausgeliefert',
            'creditplus_processing_payment' => 'Auszahlung',
            'creditplus_paid' => 'Ausgezahlt',
            'creditplus_declined_soft' => 'Weich abgelehnt',
            'creditplus_declined_hard' => 'Abgelehnt',
            'creditplus_cancelled' => 'Storniert',
            'review_necessary' => 'Überprüfung notwendig',
            'creditplus_error' => 'Fehler',
            'creditplus_docs_received' => 'Warten auf Unterlagen',
            'no_credit_approved' => 'Es wurde kein Kredit genehmigt.',
        );
        $aNewStates = [];//order_transaction.state
        foreach ($aBackendTranslations as $name => $sDefault) {
            $id = self::ID_STATE[$name];
            $aNewStates[] = ['id' => $id, 'name' => $sDefault, 'technicalName' => $name, 'stateMachineId' => $sStateMachineEntitesId];
        }
        /** @var EntityRepository $stateRepo */
        $stateRepo = $this->container->get(StateMachineStateDefinition::ENTITY_NAME . '.repository');
        $stateRepo->upsert($aNewStates, $context);
        /** @var StateMachineStateEntity $oOpenState */
        $oOpenState = $stateRepo->search((new Criteria())->addFilter(new EqualsFilter('technicalName', 'open'), new EqualsFilter('stateMachineId', $sStateMachineEntitesId)), $context)->getEntities()->first();

        $aStateTransitions = [
            //Transitions für die Normale durchführung
            ['id' => md5('open-borrowing_request_shown'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['borrowing_request_shown'], 'actionName' => 'cp_borrowing_request_shown', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('borrowing_request_shown-creditplus_referred'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_accepted'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_approved'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-creditplus_approved_and_dispatched'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved_and_dispatched-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_processing_payment-creditplus_paid'), 'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
            //Transitions für die Abgelehnt States

            //Zuerst Soft
            ['id' => md5('borrowing_request_shown-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved_and_dispatched-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_declined_soft'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],

            //Dann Hart
            ['id' => md5('borrowing_request_shown-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved_and_dispatched-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_declined_hard'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],

            //Transitions vom Ausgangspunkt open, welcher bei "Vor Bestellabschluss" massiv auftritt
            ['id' => md5('open-creditplus_referred'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_accepted'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_approved'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_approved_and_dispatched'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_processing_payment'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_paid'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_declined_soft'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_declined_soft'], 'actionName' => 'cp_contract_declined_soft', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_declined_hard'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_declined_hard'], 'actionName' => 'cp_contract_declined_hard', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-creditplus_cancelled'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('open-review_necessary'), 'fromStateId' => $oOpenState->getId(), 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],

            //Restliche Transitions vom Ausgangspunkt borrowing_request_shown
            ['id' => md5('borrowing_request_shown-creditplus_accepted'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('borrowing_request_shown-creditplus_approved'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('borrowing_request_shown-creditplus_approved_and_dispatched'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('borrowing_request_shown-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('borrowing_request_shown-creditplus_paid'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

            //Restliche Transitions vom Ausgangspunkt creditplus_referred
            ['id' => md5('creditplus_referred-creditplus_approved'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_approved_and_dispatched'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_paid'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

            //Restliche Transitions vom Ausgangspunkt creditplus_accepted
            ['id' => md5('creditplus_accepted-creditplus_approved_and_dispatched'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_paid'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

            //Restliche Transitions vom Ausgangspunkt creditplus_approved
            ['id' => md5('creditplus_approved-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-creditplus_paid'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

            //Restliche Transitions vom Ausgangspunkt creditplus_approved_and_dispatched
            ['id' => md5('creditplus_approved_and_dispatched-creditplus_paid'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],

            //Transitions zum Storniert State
            ['id' => md5('borrowing_request_shown-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved_and_dispatched-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_processing_payment-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],

            //Transitions zur review_necessary State
            ['id' => md5('borrowing_request_shown-review_necessary'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved_and_dispatched-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_processing_payment-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_processing_payment'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_paid-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_paid'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_declined_soft-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_declined_soft'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_declined_hard-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_declined_hard'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_cancelled-review_necessary'), 'fromStateId' => self::ID_STATE['creditplus_cancelled'], 'toStateId' => self::ID_STATE['review_necessary'], 'actionName' => 'cp_reviewNecessary', 'stateMachineId' => $sStateMachineEntitesId],

            //Transitions von review_necessary State
            ['id' => md5('review_necessary-open'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-borrowing_request_shown'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['borrowing_request_shown'], 'actionName' => 'cp_borrowing_request_shown', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_referred'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_referred'], 'actionName' => 'cp_duly_completed', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_accepted'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_accepted'], 'actionName' => 'cp_contract_accepted', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_approved'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_approved'], 'actionName' => 'cp_contract_approved', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_approved_and_dispatched'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_approved_and_dispatched'], 'actionName' => 'cp_delivered', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_processing_payment'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_processing_payment'], 'actionName' => 'cp_pay', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_paid'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_paid'], 'actionName' => 'cp_paid', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('review_necessary-creditplus_cancelled'), 'fromStateId' => self::ID_STATE['review_necessary'], 'toStateId' => self::ID_STATE['creditplus_cancelled'], 'actionName' => 'cp_cancelled', 'stateMachineId' => $sStateMachineEntitesId],

            // Transitions zurück zum Anfang für "Zahlungsartwechsel"
            ['id' => md5('borrowing_request_shown-open'), 'fromStateId' => self::ID_STATE['borrowing_request_shown'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_referred-open'), 'fromStateId' => self::ID_STATE['creditplus_referred'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_accepted-open'), 'fromStateId' => self::ID_STATE['creditplus_accepted'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_approved-open'), 'fromStateId' => self::ID_STATE['creditplus_approved'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_declined_soft-open'), 'fromStateId' => self::ID_STATE['creditplus_declined_soft'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_declined_hard-open'), 'fromStateId' => self::ID_STATE['creditplus_declined_hard'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
            ['id' => md5('creditplus_cancelled-open'), 'fromStateId' => self::ID_STATE['creditplus_cancelled'], 'toStateId' => $oOpenState->getId(), 'actionName' => 'cp_changePaymentMethod', 'stateMachineId' => $sStateMachineEntitesId],
        ];
        /** @var EntityRepository $oStateTransition */
        $oStateTransition = $this->container->get(StateMachineTransitionDefinition::ENTITY_NAME . '.repository');
        $oStateTransition->upsert($aStateTransitions, $context);

    }

    public function update(UpdateContext $updateContext): void
    {
        $additionalMessage20Id = md5('sCreditplusOrderAdditionalMessage20');
        /** @var EntityRepository $mailTemplateRepository */
        $mailTemplateRepository = $this->container->get(MailTemplateDefinition::ENTITY_NAME . '.repository');
        $oMailsCollection = $mailTemplateRepository->searchIds(new Criteria([$additionalMessage20Id]), $updateContext->getContext());
        $iTotalFound = $oMailsCollection->getTotal();
        if ($iTotalFound < 1) {
            $plain = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_20_plain.html.twig');
            $html = file_get_contents(dirname(__FILE__) . '/DefaultMails/sCreditplusOrderAdditionalMessage_20_html.html.twig');
            $object = [
                'id' => md5('sCreditplusOrderAdditionalMessage20'),
                'name' => 'sCreditplusOrderAdditionalMessage20',
                'mailTemplateTypeId' => md5('sCreditplusOrderAdditionalMessage_type'),
                'systemDefault' => false,
                'senderName' => '{{ salesChannel.name }}',
                'description' => 'Bestellmail für CreditPlus',
                'subject' => 'Hinweis zu der CreditPlus Finanzierung Ihrer Bestellung im {{ salesChannel.name }}',
                'contentHtml' => $html,
                'contentPlain' => $plain
            ];
            $mailTemplateRepository->create([$object], $updateContext->getContext());
        }
        /** @var EntityRepository $snippetRepo */
        $snippetRepo = $this->container->get(SnippetDefinition::ENTITY_NAME . '.repository');
        $snippetCollection = $snippetRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('translationKey', 'sicoCreditplus.error.creditplusNotAvailable')), $updateContext->getContext());
        if ($snippetCollection->getTotal() < 0) {
            /** @var EntityRepository $snippetSetRepo */
            $snippetSetRepo = $this->container->get(SnippetSetDefinition::ENTITY_NAME . '.repository');
            $snippetSetColl = $snippetSetRepo->searchIds((new Criteria())->addFilter(new EqualsFilter('iso', 'de-DE')), $updateContext->getContext());
            $snippetSetId = $snippetSetColl->firstId();
            $obj = [
                'snippet_set_id' => $snippetSetId,
                'translationKey' => 'sicoCreditplus.error.creditplusNotAvailable',
                'value' => 'Leider können wir Ihrer Finanzierungsanfrage nicht entsprechen. Die Entscheidung über Ihre Anfrage basiert auf einer automatisierten Verarbeitung Ihrer personenbezogenen Daten, die der Bewertung einzelner Persönlichkeitsmerkmale dienen. Bitte wählen Sie eine andere Zahlungsart aus.',
                'author' => 'creditPlus'
            ];
            $snippetRepo->create([$obj], $updateContext->getContext());
        }
		$this->setPaymentMethodTechnicalName($updateContext->getContext());
    }
}
