<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 21.09.2023
 * @time 15:11
 */

namespace SicoCreditPlus\Components;

use DateTimeImmutable;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Level;
use Monolog\LogRecord;
use Monolog\Processor\ProcessorInterface;
use Monolog\Processor\PsrLogMessageProcessor;
use Psr\Log\LoggerInterface;
use Throwable;

class SicoCreditPlusPsrLogger implements LoggerInterface
{

	/**
	 * @var string $sFileName
	 */
	private string $sFileName = '';

	/**
	 * @var Level|int $iMinLogLevel
	 */
	private Level|int $iMinLogLevel = Level::Emergency;

	/**
	 * @var ProcessorInterface[] $aProcessors
	 */
	private array $aProcessors = [];

	/**
	 * @var HandlerInterface[] $aHandlers
	 */
	private array $aHandlers = [];

	public function __construct(string $sFileName = 'sico_credit_plus', Level|int $iLevel = Level::Info, int $iMaxFiles = 14)
	{
		if ( $sFileName ) {
			$this->sFileName = $sFileName;
		}
		if ( $iLevel ) {
			$this->iMinLogLevel = $iLevel;
		}
		$this->aProcessors[] = new PsrLogMessageProcessor();
		$sFilePath = dirname(__FILE__, 6).'/var/log/'.$sFileName.'.log';
		$this->aHandlers[] = new RotatingFileHandler($sFilePath, $iMaxFiles, $iLevel);
	}

	private function addRecord(Level|int $iLevel = Level::Debug, \Stringable|string $sMessage = '', array $aContext = [], ?DateTimeImmutable $oDateTime = null): bool
	{
		if ( is_int($iLevel) ) {
			$iLevel = Level::fromValue($iLevel);
		}
		if ( $iLevel->value < $this->iMinLogLevel ) {
			// Skip ignored levels
			return false;
		}
		try {
			$recordInitialized = count($this->aProcessors) === 0;

			$oRecord = new LogRecord(
				$oDateTime ?? new DateTimeImmutable(),
				$this->sFileName,
				$iLevel,
				$sMessage,
				$aContext
			);
			$handled = false;

			foreach ($this->aHandlers as $oHandler) {
				if (false === $recordInitialized) {
					// skip initializing the record as long as no handler is going to handle it
					if (!$oHandler->isHandling($oRecord)) {
						continue;
					}

					try {
						foreach ($this->aProcessors as $oProcessor) {
							$oRecord = $oProcessor($oRecord);
						}
						$recordInitialized = true;
					} catch (Throwable $e) {
						$this->handleException($e, $oRecord);

						return true;
					}
				}

				// once the record is initialized, send it to all handlers as long as the bubbling chain is not interrupted
				try {
					$handled = true;
					if (true === $oHandler->handle($oRecord)) {
						break;
					}
				} catch (Throwable $e) {
					$this->handleException($e, $oRecord);

					return true;
				}
			}

			return $handled;
		} catch (Throwable $oThrow) {
			// Do nothing
			return false;
		}
	}

	public function emergency(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Emergency, $message, $context);
	}

	public function alert(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Alert, $message, $context);
	}

	public function critical(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Critical, $message, $context);
	}

	public function error(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Error, $message, $context);
	}

	public function warning(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Warning, $message, $context);
	}

	public function notice(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Notice, $message, $context);
	}

	public function info(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Info, $message, $context);
	}

	public function debug(\Stringable|string $message, array $context = []): void
	{
		$this->addRecord(Level::Debug, $message, $context);
	}

	public function log($level, \Stringable|string $message, array $context = []): void
	{
		$aAcceptedLevels = [
			Level::Debug, Level::Info, Level::Notice,
			Level::Warning, Level::Error, Level::Critical,
			Level::Emergency
		];
		if ( !in_array($level, $aAcceptedLevels) ) {
			throw new \InvalidArgumentException('Log level must be valid value.');
		} else {
			$iLevel = $level;
		}
		$this->addRecord($iLevel, $message, $context);
	}

	private function handleException($oEx, $oRecord): void
	{
		// :'(
	}
}
