<?php
/**
 * Created by PhpStorm.
 * User: Philipp.Holzmann
 * Date: 06.01.2020
 * Time: 09:51
 */

namespace SicoCreditPlus\Controller;


use Exception;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\AbstractCartPersister;
use Shopware\Core\Checkout\Cart\Exception\CartTokenNotFoundException;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Checkout\Customer\Aggregate\CustomerAddress\CustomerAddressEntity;
use Shopware\Core\Checkout\Order\Aggregate\OrderAddress\OrderAddressEntity;
use Shopware\Core\Checkout\Order\Aggregate\OrderLineItem\OrderLineItemEntity;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionCollection;
use Shopware\Core\Checkout\Order\OrderDefinition;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Payment\Cart\Token\TokenFactoryInterfaceV2;
use Shopware\Core\Content\Product\ProductDefinition;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Exception\InvalidUuidException;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\PlatformRequest;
use Shopware\Core\SalesChannelRequest;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextServiceInterface;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextServiceParameters;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\Salutation\SalutationDefinition;
use Shopware\Core\System\Salutation\SalutationEntity;
use Shopware\Core\System\StateMachine\Aggregation\StateMachineState\StateMachineStateDefinition;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Storefront\Controller\StorefrontController;
use Shopware\Storefront\Framework\Routing\Router;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use SicoCreditPlus\Core\Content\CreditPlusPayment\CreditPlusPaymentDefinition;
use SicoCreditPlus\Core\Content\CreditPlusPayment\CreditPlusPaymentEntity;
use SicoCreditPlus\Core\Content\SicoCreditPlusSavedCart\SicoCreditPlusSavedCartCollection;
use SicoCreditPlus\Core\Content\SicoCreditPlusSavedCart\SicoCreditPlusSavedCartDefinition;
use SicoCreditPlus\Core\Content\SicoCreditPlusSavedCart\SicoCreditPlusSavedCartEntity;
use SicoCreditPlus\Lib\Controller\CreditPlusWebshopAPI;
use SicoCreditPlus\Lib\CreditPlusHelper\CreditPlusMainData;
use SicoCreditPlus\Lib\CreditPlusObjects\ShoppingCartItem;
use SicoCreditPlus\Service\Interfaces\SavedCartSerializerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionDefinition;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Checkout\Order\Aggregate\OrderTransaction\OrderTransactionEntity;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;
use Shopware\Core\Checkout\Cart\Order\OrderConverter;


#[Route(defaults: ['_routeScope' => ['storefront']])]
class SicotriggerController extends StorefrontController{

	/** @var SicoCreditPlusHelper|null $_oCreditPlusHelper */
	protected ?SicoCreditPlusHelper $_oCreditPlusHelper = null;

	/** @var Router|UrlGeneratorInterface */
	protected UrlGeneratorInterface|Router $router;

	protected $oRequest = null;

	/**
	 * @var SystemConfigService
	 */
	protected SystemConfigService $systemConfigService;

	/** @var SalesChannelContext */
	protected SalesChannelContext $salesChannelContext;

	/** @var string */
	protected string $sRouteName;

	/** @var SicoCreditPlusLogger */
	protected SicoCreditPlusLogger $oLogger;

	/**
	 * @var CartService
	 */
	protected CartService $oCartService;

	/** @var OrderConverter */
	protected OrderConverter $oOrderConverter;

	protected AbstractCartPersister $oCartPersister;

	/**
	 * @var TokenFactoryInterfaceV2
	 */
	protected TokenFactoryInterfaceV2 $tokenFactory;
	/**
	 * @var SessionInterface
	 */
	protected SessionInterface $oSession;

	/** @var SalesChannelContextServiceInterface */
	protected SalesChannelContextServiceInterface $oSalesChannelContextService;

	/** @var RequestStack */
	private RequestStack $oRequestStack;

	/**
	 * SicotriggerController constructor.
	 * @param SystemConfigService $systemConfigService
	 * @param UrlGeneratorInterface $router
	 * @param SicoCreditPlusHelper $creditPlusHelper
	 * @param SicoCreditPlusLogger $oLogger
	 * @param CartService $oCartService
	 * @param OrderConverter $oOrderConverter
	 * @param AbstractCartPersister $cartPersister
	 * @param TokenFactoryInterfaceV2 $tokenFactory
	 * @param SalesChannelContextServiceInterface $oSalesChannelContextService
	 * @param RequestStack $oRequestStack
	 */
	public function __construct(
		SystemConfigService $systemConfigService,
		UrlGeneratorInterface $router,
		SicoCreditPlusHelper $creditPlusHelper,
		SicoCreditPlusLogger $oLogger,
		CartService $oCartService,
		OrderConverter $oOrderConverter,
        AbstractCartPersister $cartPersister,
		TokenFactoryInterfaceV2 $tokenFactory,
		SalesChannelContextServiceInterface $oSalesChannelContextService,
		RequestStack $oRequestStack)
	{
		$this->systemConfigService = $systemConfigService;
		$this->router = $router;
		$this->_oCreditPlusHelper = $creditPlusHelper;
		$this->oLogger = $oLogger;
		$this->oCartService = $oCartService;
		$this->oOrderConverter = $oOrderConverter;
		$this->oCartPersister = $cartPersister;
		$this->tokenFactory = $tokenFactory;
		$this->oSalesChannelContextService = $oSalesChannelContextService;
		$this->oRequestStack = $oRequestStack;
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @return Response
	 * @throws Exception
	 */
    #[Route(path: '/payment/creditplus', name: 'frontend.creditplus.trigger', options: ['seo' => false], methods: ['GET'])]
    public function indexAction(Request $request,SalesChannelContext $context): Response
    {
		//$this->container->get('')
        $this->oSession = $request->getSession();
		$oCreditPlusHelper = $this->getCreditPlusHelper();
        $oCreditPlusHelper->setSession($this->oSession);
		$oCreditPlusHelper->setSalesChannelContext($context);
		$oCreditPlusHelper->setContext($context->getContext());
		$iStatusText = (int)$request->get('statusText');
		$this->sRouteName = 'frontend.creditplus.trigger';
		$sOrderTransactionId = $request->get('don');
		$oOrder = null;
		if ( $sOrderTransactionId ) {
			if ( $this->isInorderMode($context->getSalesChannelId()) ) {
				/** @var EntityRepository $oCpPaymentRepo */
				$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
				$oCreditPlusCriteria = new Criteria();
				$oCreditPlusCriteria->addFilter(new EqualsFilter('orderNumber',$sOrderTransactionId));
				$oCreditPlusCriteria->addAssociations(['order','order.transactions', 'order.transactions.stateMachineState', 'order.stateMachineState']);//'order.transactions.statemachinestate'
				/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
				$oCpPaymentEntity = $oCpPaymentRepo->search($oCreditPlusCriteria,$context->getContext())->first();
				if($oCpPaymentEntity){
					$oSalesChannelContextParameters = new SalesChannelContextServiceParameters($context->getSalesChannelId(),
						$oCpPaymentEntity->getToken(),
						$context->getContext()->getLanguageId(),$context->getCurrencyId(),$context->getDomainId());
					$context = $this->oSalesChannelContextService->get($oSalesChannelContextParameters);
					$oCreditPlusHelper->setSalesChannelContext($context);
					$oCreditPlusHelper->setContext($context->getContext());
					$this->updateSession($oCpPaymentEntity->getToken());
					$request->getSession()->set('cpDealerOrderNumber',$oCpPaymentEntity->getOrderNumber());
					$oOrder = $this->getCart($oCpPaymentEntity->getOrderNumber(), $context);
				}
				if ( !$oOrder ) {
					$oOrder = $this->getOrder($sOrderTransactionId, $context);
				}
				if ( !$oOrder ) {
					$oOrder = $this->getCart($sOrderTransactionId, $context);
				}
			} else {
				$oOrder = $this->getOrder($sOrderTransactionId, $context);
			}
		}

		$sSccpAbortInfoText = '';
		if($oOrder){
			if($oOrder instanceof OrderEntity){
				$sOrderTransactionId = $oOrder->getTransactions()->first()->getId();
				$sSalesChannelId = $oOrder->getSalesChannelId();
				if ( $context->getSalesChannelId() === $sSalesChannelId ) {
					$oContractData = $oCreditPlusHelper->getContractData($sSalesChannelId, $oOrder);
					if ( $oContractData && $oContractData->state ) {
						switch ( $oContractData->state ) {
							case '92':
							case '93':
								// Declined will be redirected to the cart page
								$sRedirectURL = $this->getForwardURL($sSalesChannelId, array('cancelAndReorder', null, null, array('cporder' => $oOrder->getOrderNumber(), 'sOrderTransactionId' => $sOrderTransactionId, 'disablePayment' => 1)));
								return new RedirectResponse($sRedirectURL);
								break;
							case '94':
								if ( $iStatusText === 30303 ) {
									$iStatusText = 30308; // Technical error
								}
								break;
							case '95':
								if ( $iStatusText === 30303 ) {
									$iStatusText = 30310; // Already cancelled
								}
								break;
							case '99':
								// Don't touch final states which will not happen on regular walkthroughs
								if ( $iStatusText === 30303 ) {
									$iStatusText = 30309; // Already finished
								}
								break;
							default:
								if ( $iStatusText === 30303 ) {
									$iStatusText = 30312; // Closed in progress
								}
								break;
						}
					}
				}
			}
			else {
				$sOrderTransactionId = $oOrder->getData()->get('cpDealerOrderNumber');
			}

			$sReplaceMeURL = $this->router->generate('frontend.creditplus.trigger.cancelandreorder',[
				'sOrderTransactionId' => $sOrderTransactionId
			]);
			$selectedView = $this->systemConfigService->get('SicoCreditPlus.config.sCPShownAs',$context->getSalesChannelId());
			$routeName = 'frontend.creditplus.trigger.popup';
			switch($selectedView){
				case 'popup':
					$routeName = 'frontend.creditplus.trigger.popup';
					break;
				case 'iframe':
					$routeName = 'frontend.creditplus.trigger.iframe';
					break;
				case 'new_window':
					$routeName = 'frontend.creditplus.trigger.new-window';
					break;
			}
			$sRetryMeURL = $this->router->generate($routeName,['orderTransactionId' => $sOrderTransactionId]);
			/** @var TranslatorInterface $oTranslator */
			$oTranslator = $this->container->get('translator');
			$sSccpAbortInfoText = $oTranslator->trans('sicoCreditplus.checkout.sccpabortinfo_text');
			$sSccpAbortInfoText = str_replace(
				array(
					'http://retry.me/',
					'http://replace.me/'
				),
				array(
					$sRetryMeURL,
					$sReplaceMeURL
				),
				$sSccpAbortInfoText
			);
		}
		return $this->renderStorefront('@SicoCreditPlus/trigger/index.html.twig',[
			'iSuccess' => $iStatusText,
			'sSccpAbortInfoText' => $sSccpAbortInfoText
		]);
	}

	private function updateSession($token): void
    {
		$master = $this->oRequestStack->getMainRequest();
		if (!$master) {
			return;
		}
		if (!$master->attributes->get(SalesChannelRequest::ATTRIBUTE_IS_SALES_CHANNEL_REQUEST)) {
			return;
		}

		if (!$master->hasSession()) {
			return;
		}
		$session = $master->getSession();
		$session->set(PlatformRequest::HEADER_CONTEXT_TOKEN, $token);
		$master->headers->set(PlatformRequest::HEADER_CONTEXT_TOKEN, $token);
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $sData
	 * @param string $sDon
	 *
	 * @return RedirectResponse|Response
	 * @throws Exception
	 */
    #[Route(path: '/payment/creditplus/dispatch/data/{sData}/don/{sDon}', name: 'frontend.creditplus.trigger.dispatch', options: ['seo' => false], methods: ['GET'])]
    public function dispatchAction(Request $request, SalesChannelContext $context, string $sData, string $sDon): RedirectResponse|Response
    {
		// This is how an URL looks like:
		// http://cpshopware.intern.dev.sinka.de/sicotrigger/dispatch/
		// data/HcuxDYQwDEDRXTKBbZyQmArpWhrEApAYiYLizlyF2J1A%252B%252FT%252FLCSnCZI4G%252FX7VzumbVfXVWNxBBggYkAi9uFVrGVfyk%252FNho9%252FqKkz%252BxQQuGGKc1pWLUkZ2qyQC645Lq67bg%253D%253D/
		// don/CP20160816121957
		// ?callback=24
		// &signature=dd454cff93abae877b7d458241085d00
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		$bDoubleEncoded = (strpos($sData, '%25') !== false);
		if ( $bDoubleEncoded ) {
			$sData = urldecode($sData);
		}
		$this->getCreditPlusHelper()->setSalesChannelContext($context);
		$this->salesChannelContext = $context;
		$iStatus = (int)$request->get('callback');
		$sSignature = $request->get('signature');
		$this->oLogger->log(false,'params',[$iStatus,$sSignature]);

		$this->oLogger->log(false,'dispatchAction',['status' => $iStatus,'data' => $sData,'don' => $sDon]);
		$this->sRouteName = 'frontend.creditplus.trigger.dispatch';


		/** @var EntityRepository $oOrderRepo */
		$oOrderRepo = $this->container->get(OrderDefinition::ENTITY_NAME.'.repository');
		$oOrderCriteria = new Criteria();
		$oOrderCriteria->addFilter(new EqualsFilter('orderNumber', $sDon));
		$oOrderCriteria->addAssociations(['transactions', 'transactions.stateMachineState']);
		$oOrder = $oOrderRepo->search($oOrderCriteria, $context->getContext())->first();
		if ( !$oOrder ) {
			/** @var EntityRepository $oCpPaymentRepo */
			$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
			$oCreditPlusCriteria = new Criteria();
			$oCreditPlusCriteria->addFilter(new EqualsFilter('orderNumber',$sDon));
			$oCreditPlusCriteria->addAssociations(['order','order.transactions', 'order.transactions.stateMachineState', 'stateMachineState']);//'order.transactions.statemachinestate'
			/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
			$oCpPaymentEntity = $oCpPaymentRepo->search($oCreditPlusCriteria,$context->getContext())->first();
			if ( $oCpPaymentEntity && $oCpPaymentEntity->getOrderId() ) {
				$oOrder = $oCpPaymentEntity->getOrder();
			}
		}

		if ( !$oOrder && $this->isInorderMode($context->getSalesChannelId()) ) {
			if($oCpPaymentEntity){
				$oSalesChannelContextParameters = new SalesChannelContextServiceParameters($context->getSalesChannelId(),
					$oCpPaymentEntity->getToken(),
					$context->getContext()->getLanguageId(),$context->getCurrencyId(),$context->getDomainId());
				$context = $this->oSalesChannelContextService->get($oSalesChannelContextParameters);
				$this->getCreditPlusHelper()->setSalesChannelContext($context);
				$this->updateSession($oCpPaymentEntity->getToken());
				$request->getSession()->set('cpDealerOrderNumber',$oCpPaymentEntity->getOrderNumber());

				$this->oLogger->info('CpPaymentEntity hat noch keine Bestellung', ['cpPaymentEntity' => $oCpPaymentEntity], $context->getSalesChannelId());
				$oOrder = $this->getCart($oCpPaymentEntity->getOrderNumber(), $context, true);
				if ( !$oOrder ) {
					$oOrder = $this->getCart($oCpPaymentEntity->getToken(), $context, true);
				}
				$oOrder->getData()->set('cpDealerOrderNumber', $oCpPaymentEntity->getOrderNumber());
			} else {
				$token = $context->getToken();
				$oOrder = $this->getCart($token, $context, true);
				if ( $oOrder != null ) {
					$sDealerOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber');
					if ( $sDealerOrderNumber == null ) {
						$sDealerOrderNumber = $request->getSession()->get('cpDealerOrderNumber');
						$oOrder->getData()->set('cpDealerOrderNumber',$sDealerOrderNumber);
					}
				} else {
					$oOrder = $this->getOrder($sDon,$context);
				}
			}
		} elseif ( !$oOrder ) {
			$oOrder = $this->getOrder($sDon,$context);
		}

		try{
			if ( !$this->checkSignature($context->getSalesChannelId(),$sSignature, $oOrder,$sDon,(string)$iStatus,$sData) ) {
				$this->oLogger->log(true,'Signaturen passen nicht',[]);
				// Error in Signature
				throw new Exception('Signatur der Bestellung passt nicht');
			}
			$sSicoTargetURL = $this->updateContract($oOrder,$iStatus,$context);

			//Workaround um das Darstellungsproblem innerhalb des IFrames
			if($this->isInorderMode($context->getSalesChannelId())){
				return new RedirectResponse($sSicoTargetURL);
			}else {
				return $this->renderStorefront('@SicoCreditPlus/trigger/afterCreditForms.html.twig',['url' => $sSicoTargetURL]);
			}
		} catch( Exception $e){
			$sTrace = '';
			foreach($e->getTrace() as $traceKey => $trace){
				if(is_array($trace)){
					if(isset($trace['file']) && isset($trace['line'])){
						$sTrace.= $trace['file'].':'.$trace['line'].'||';
					}
				}
			}
			$this->oLogger->log(true,$e->getMessage(),[__FILE__,__LINE__,$e->getFile(),$e->getLine(),$sTrace]);
			$url = $this->router->generate('frontend.creditplus.trigger',['statusText' => 30305,'don' => $sDon]);
			return new RedirectResponse($url);
		}

	}

	/**
	 * Checks if the signature is correct. If bTestMode is active, any signature is accepted.
	 * If bTestMode is not set, only the two possible calculations are accepted.
	 *
	 * @param string $sSignature Signature given by request - Signing Parameter callback and the Base URL before that
	 * @param Cart|OrderEntity|null $oOrder Order object, if given for check
	 * @param string $sDon Dealer Order Number
	 * @param string $sCallback Callback Value
	 * @param string $sData Data Attribute
	 * @param bool $bFalseSevenCheck If true, the signature is checked with callback 3 instead of 7 and that value is directly returned
	 *
	 * @return bool True if $sSignature is OK, false if not
	 */
	protected function checkSignature(string $sSalesChannelId, string $sSignature = '', OrderEntity|Cart $oOrder = null, string $sDon = '', string $sCallback = '', string $sData = '', bool $bFalseSevenCheck = false ): bool
    {
		$bTestMode = intval($this->systemConfigService->get('SicoCreditPlus.config.bTestMode',$sSalesChannelId));
		$sSaltValue = $this->systemConfigService->get('SicoCreditPlus.config.sSignatureSalt',$sSalesChannelId);

		$sSignedURL = $this->getTriggerUrlWithDataAndDon($sData, $sDon).'&';
		if ( $bFalseSevenCheck ) {
			if ( $sCallback !== '7' ) {
				return false;
			}
			return ($sSignature == md5($sSignedURL.$sSaltValue.'3'));
		}

		if($bTestMode){
			return true;
		}
		// URL+Salt+Callback
		$bValid = ($sSignature == md5($sSignedURL.$sSaltValue.$sCallback));
		if ( $sCallback === '7' ) {
			// URL+Salt+3 if provided with 7 (old error on sending side)
			$bValid = $bValid || ($sSignature == md5($sSignedURL.$sSaltValue.'3'));
		}
		// or DON+Salt+Callback
		$bValid = $bValid || ($sSignature == md5($sDon.$sSaltValue.$sCallback));
		// or OrderNumber+Salt+Callback
		if($oOrder instanceof OrderEntity){
			if ( $sOrderNumber = $oOrder->getOrderNumber() ) {
				$bValid = $bValid || ($sSignature == md5($sOrderNumber.$sSaltValue.$sCallback));
			}
			$sOrderNumber = $oOrder->getOrderNumber();
		} else {
			if(($oOrder !== null) && ($sOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber'))){
				$bValid = $bValid || ($sOrderNumber == md5($sOrderNumber.$sSaltValue.$sCallback));
			}
			$sOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber');
		}

		if ( !$bValid ) {
			$this->oLogger->log(false,'signature strings',[
				$sSignedURL.$sSaltValue.$sCallback,
				$sSignedURL.$sSaltValue.'3',
				$sDon.$sSaltValue.$sCallback,
				$sOrderNumber.$sSaltValue.$sCallback
			]);
			$this->oLogger->log(false,'valid signatures',[
				md5($sSignedURL.$sSaltValue.$sCallback),
				md5($sSignedURL.$sSaltValue.'3'),
				md5($sDon.$sSaltValue.$sCallback),
				md5($sOrderNumber.$sSaltValue.$sCallback)
			]);
		}

		return $bValid;
	}


	/**
	 * @param Cart|OrderEntity $oOrder
	 * @param int $iStatus
	 * @param SalesChannelContext $context
	 *
	 * @return null|string String if redirect url exists
	 * @throws ContainerExceptionInterface
	 * @throws NotFoundExceptionInterface
	 */
	protected function updateContract(OrderEntity|Cart $oOrder, int $iStatus, SalesChannelContext $context ): ?string
    {
		$oCPHelper = $this->getCreditPlusHelper();
		$oCPHelper->setContext($context->getContext());
		$oContractData = $oCPHelper->getContractData($context->getSalesChannelId(),$oOrder);
		$sStateOpen = 'open';
		if ( $oContractData ) {
			if ( $oContractData->completeOrderUrl !== '' ) {
				/** @var EntityRepository $oCpPaymentRepo */
				$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
				if ( $oOrder instanceof OrderEntity ) {
					$oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('orderId',$oOrder->getId())),$context->getContext())->getEntities();
				} else {
					// $oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('token',$oOrder->getToken())),$context->getContext())->getEntities();
					$oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('orderNumber',$oOrder->getData()->get('cpDealerOrderNumber'))),$context->getContext())->getEntities();
				}

				if($oCpPaymentColl->count() > 0){
					/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
					$oCpPaymentEntity = $oCpPaymentColl->first();
					$oCpPaymentRepo->update([[
						'id' => $oCpPaymentEntity->getId(),
						'orderFinish' => $oContractData->completeOrderUrl
					]], $context->getContext());
				}
			}

			if($this->isInorderMode($context->getSalesChannelId())){
				if ( $iStatus === 7 ) {
					// Direct update with user interaction
					$iStatus = (int) $oContractData->state;
				} elseif ( $iStatus === 3 ) {
					// Regular update from CreditPlus
					$iStatus = (int) $oContractData->state;
				}
			}
		}

		// Todo: || PaymentStatus == $oHelper->getStatus($iStatus)
		//$iPaymentStateID = $oOrder->getPaymentStatus()->getId();
		if($oOrder instanceof Cart){
			//order is cart. This means we're going the first time through the program
			$sCPPaymentState = $oCPHelper->getCreditPlusPaymentState($iStatus);

			//This two lines are for the rule because the rule has no dependency injection which we can use to get
			//some information.
			$oCartData = $oOrder->getData();
			$oCartData->set('CreditPlusContractStatus',$iStatus);
			$oOrder->setData($oCartData);
			$this->oCartPersister->save($oOrder, $context);

			if ( $iStatus > 20 && $oContractData && $oContractData->finallyApproved ) {
				// Create order from saved cart when contract is accepted and not yet an order
				// Fake request data bag with blank values for order conversion
				$oDataBag = new RequestDataBag(['customerComment' => '', 'affiliateCode' => '', 'campaignCode' => '']);
				$iCallback = (int)$this->oRequestStack->getMainRequest()->get('callback');
				$sSignature = (string)$this->oRequestStack->getMainRequest()->get('signature');
				$sData = (string)$this->oRequestStack->getMainRequest()->get('sData');
				if ( strpos($sData, '%25') !== false ) {
					$sData = urldecode($sData);
				}
				// If callback is 7 but signed as 3, it's a broken link from a previous extension version
				$bFalseSevenCheck = $this->checkSignature($context->getSalesChannelId(), $sSignature, $oOrder, $oOrder->getData()->get('cpDealerOrderNumber'), (string)$iCallback, $sData, true);
				if ( ($iCallback === 3) || $bFalseSevenCheck ) {
					// If triggered by CreditPlus
					try {
						$this->oLogger->log(false, 'Creating order from cart because Status is over 20 and order does not yet exist', ['cartToken' => $oOrder->getToken(), 'callback' => $iCallback], $context->getSalesChannelId());
						$sDealerOrderNumber = $oCartData->get('cpDealerOrderNumber');
						$sOrderID = $this->oCartService->order($oOrder, $context, $oDataBag);
						// Delete saved cart after conversion
						$this->deleteSavedCart($sDealerOrderNumber, $context);
						$oOrder = $this->getOrderByOrderId($sOrderID, $context);
						$aCustomFields = $oOrder->getCustomFields();
						$aCustomFields['cpDealerOrderNumber'] = $sDealerOrderNumber;
						$oOrder->setCustomFields($aCustomFields);
						/** @var EntityRepository $oOrderRepository */
						$oOrderRepository = $this->container->get(OrderDefinition::ENTITY_NAME.'.repository');
						$oOrderRepository->upsert([['id' => $sOrderID, 'customFields' => $aCustomFields]], $context->getContext());
						/** @var EntityRepository $oCpPaymentRepo */
						$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
						$oCpPaymentCriteria = new Criteria();
						$oCpPaymentCriteria->addFilter(new EqualsFilter('orderNumber', $sDealerOrderNumber));
						$oCpPaymentCriteria->addAssociations(['order', 'order.transactions', 'order.transactions.stateMachineState', 'order.stateMachineState']);
						$oRes = $oCpPaymentRepo->search($oCpPaymentCriteria, $context->getContext());
						/** @var CreditPlusPaymentEntity $oCpPaymentEntity */
						if ( $oCpPaymentEntity = $oRes->first() ) {
							$oCpPaymentEntity->setOrder($oOrder);
							$oCpPaymentEntity->setOrderId($sOrderID);
							$oCpPaymentRepo->upsert([[
								'id' => $oCpPaymentEntity->getId(),
								'orderId' => $sOrderID
							]], $context->getContext());
							$this->oLogger->info('Linking CreditPlus Payment Object to Order', ['creditPlusPaymentId' => $oCpPaymentEntity->getId(), 'orderId' => $sOrderID], $context->getSalesChannelId());
						}
						/** @var OrderTransactionEntity $oTransaction */
						$oTransaction = $oOrder->getTransactions()->first();
						$sTransactionId = $oTransaction->getId();
						$sTargetUrl = $oCPHelper->getTargetURLForFinishingPayment($sTransactionId, $oOrder->getSalesChannelId());
						$oCPHelper->sendAdditionalMessageMail($context, $oOrder, $sTargetUrl);
					} catch ( Exception $oEx ) {
						$this->oLogger->log(true, 'Error while creating order: '.$oEx->getMessage(), [], $context->getSalesChannelId());
					}
				}
			}
			$oCPHelper->checkForUpdateMail($context->getSalesChannelId(), $oOrder);
			return $this->finishContract($context->getSalesChannelId(), $oOrder, $iStatus, $sCPPaymentState);
		}
		$iPaymentStateID = $oOrder->getTransactions()->first()->getStateMachineState()->getTechnicalName();
		$sCPPaymentState = $oCPHelper->getCreditPlusPaymentState($iStatus);
		$oCPPaymentState = $oCPHelper->getPaymentStateByName($sCPPaymentState);
        $oCPHelper->checkForUpdateMail($context->getSalesChannelId(), $oOrder);
        if ( ($iPaymentStateID == $sStateOpen) || (($oCPPaymentState !== null) ) ) {
			// First time finisher
			// or First time after backend already updated finisher
			$sURL = $this->finishContract($context->getSalesChannelId(), $oOrder, $iStatus, $sCPPaymentState);
		} else {
			// Already finished
			$sURL = $this->getForwardURL($context->getSalesChannelId(), array('index', null, null, array('statusText' => 30309)));
		}
		return $sURL;
	}

	/**
	 * @param string $sSalesChannelId
	 * @param OrderEntity|Cart $oOrder
	 * @param int $iStatus
	 * @param string $sStatus
	 * @return null|string String if redirect URL exists
	 * @throws ContainerExceptionInterface Regular PSR exception
	 * @throws NotFoundExceptionInterface Regular PSR exception
	 */
	protected function finishContract(string $sSalesChannelId, OrderEntity|Cart $oOrder, int $iStatus, string $sStatus ): ?string
    {
		$aGoodStates = array(
			// Tentative OK or Hard OK, customer needs to sign or has signed contract accepted by the bank
			24,
			// Payment in progress (Bank to Shop)
			32,
			// Contract is already paid (Bank to Shop)
			99
		);
		$aGreyStates = array(
			// Open - Manual decision
			20,
			// Open - Additional info required from customer
			25, 21, 22, 23,
			// Cancelled doesn't happen on returns usually, so we treat it like open in this case
			95,
		);
		$aBadStates = array(
			// Unknown Error
			2,
			// Declined - soft
			92,
			// Declined - hard
			93
		);
		if($oOrder instanceof OrderEntity){
			if($oOrder->getTransactions()->first()->getStateMachineState()->getTechnicalName() != $sStatus){
				$this->getCreditPlusHelper()->setPaymentStatus($oOrder, $sStatus);
			}
			$sTransactionId =  $oOrder->getTransactions()->first()->getId();
		} else {
			$this->getCreditPlusHelper()->setPaymentStatus($oOrder,$sStatus);
			$sTransactionId = $oOrder->getData()->get('cpDealerOrderNumber');
		}

		if ( in_array($iStatus, $aGoodStates) ) {
			//$sStateOfPayment = 'the_credit_has_been_preliminarily_accepted';
			return $this->getForwardURL($sSalesChannelId, array('index', null, null, array('statusText' => 30307, 'don' => $sTransactionId)), $oOrder);
		} elseif ( in_array($iStatus, $aGreyStates) ) {
			return $this->getForwardURL($sSalesChannelId, array('index', null, null, array('statusText' => 30307, 'don' => $sTransactionId)), $oOrder);
		} elseif ( in_array($iStatus, $aBadStates) ) {

			$this->oSession->set('bSicoFinancingExcluded',true);
			if($oOrder instanceof OrderEntity){
				return $this->getForwardURL($sSalesChannelId, array('cancelAndReorder', null, null, array('cporder' => $oOrder->getOrderNumber(), 'sOrderTransactionId' => $sTransactionId,'disablePayment' => 1)));
			} else {
				$this->oSession->set('bSicoFinancingExcluded', true);
				/** @var TranslatorInterface $oSnippetService */
				$oSnippetService = $this->container->get('translator');//creditplus-not-available
				$error = $oSnippetService->trans('sicoCreditplus.error.creditplusNotAvailable');
				$this->oSession->getFlashBag()->add('danger',$error);
                return $this->router->generate('frontend.checkout.confirm.page',[], UrlGeneratorInterface::ABSOLUTE_URL);
				// we are in-order and don't have an order, so we go back to the confirm page
				/*
				$aSicoTargetURLBase64 = base64_encode($sSicoTargetURL);
				$sBaseUrl = $this->router->getContext()->getBaseUrl();
				$url = $sBaseUrl.'/bundles/sicocreditplus/iframeBreakout.html?url='.$aSicoTargetURLBase64;
				return $url;
				*/
			}

		}
		$this->oLogger->log(true,'unknown Status Code',['status' => $iStatus]);
		// Unknown yet signed Status Code from CreditPlus!
		return $this->getForwardURL($sSalesChannelId,array('index', null, null, array('statusText' => 30308,'don' => $sTransactionId)));
	}


    /**
     * Takes params as given to $this->forward() and converts them to an URL
     * @param string $sSalesChannelId
     * @param array $aParams
     * @param Cart|OrderEntity|null $oOrder
     *
     * @return string URL Target
     */
	protected function getForwardURL(string $sSalesChannelId, array $aParams, OrderEntity|Cart $oOrder = null ): string
    {
		$aRouterParams = array();

		switch($aParams[0]){
			case 'index':
				$routeName = 'frontend.creditplus.trigger'; break;
			case 'cancelAndReorder':
				$routeName = 'frontend.creditplus.trigger.cancelandreorder'; break;
			default:
				$routeName = $this->sRouteName;
		}

		if ( $aParams[3] ) {
			foreach ( $aParams[3] as $sKey => $sValue ) {
				$aRouterParams[$sKey] = $sValue;
			}
		}
		if(isset($aRouterParams['statusText'])){
			if(!isset($aParams[1]) && $aRouterParams['statusText'] == 30307){
				//to get back on the order street at the inorder
				if($this->isInorderMode($sSalesChannelId) && isset($oOrder)) {
					return $this->router->generate('frontend.creditplus.trigger.before-order',[], UrlGeneratorInterface::ABSOLUTE_URL);
				}else if(isset($oOrder) && $oOrder instanceof OrderEntity){
					return $this->router->generate('frontend.checkout.finish.page',['orderId' => $oOrder->getId()], UrlGeneratorInterface::ABSOLUTE_URL);
				}

			}
		}
		return $this->router->generate($routeName,$aRouterParams, UrlGeneratorInterface::ABSOLUTE_URL);
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $orderTransactionId
	 *
	 * @return RedirectResponse|Response
	 */
    #[Route(path: '/payment/creditplus/{orderTransactionId}/popup', name: 'frontend.creditplus.trigger.popup', options: ['seo' => false], methods: ['GET'])]
    public function showPopupAction(Request $request, SalesChannelContext $context, string $orderTransactionId): RedirectResponse|Response
    {
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		$this->sRouteName = 'frontend.creditplus.trigger.popup';
		$aParam = $this->getCommonVariables($orderTransactionId, $context, $request);

		if(isset($aParam['bError'])){
			if(isset($aParam['id']) && $aParam['id'] == 4){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30309,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 3){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30310,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 2){
				return $this->redirectToRoute('frontend.checkout.register.page');
			}
			if(isset($aParam['id']) && $aParam['id'] == 1){
				throw new NotFoundHttpException('Order not found');
			}
			else{
				$this->oLogger->log(true,$aParam['sMessage'],['bt' => debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,5)]);
				return $this->redirectToRoute('frontend.creditplus.trigger.error');
			}
		}
		$aParam['aGetParams'] = $this->splitGetParamsForTemplate($aParam['sTargetURL']);

		return  $this->renderStorefront('@SicoCreditPlus/trigger/show_popup.html.twig',$aParam);
		//$response->headers->set('x-frame-options','')
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $orderTransactionId
	 *
	 * @return RedirectResponse|Response
	 */
    #[Route(path: '/payment/creditplus/{orderTransactionId}/iframe', name: 'frontend.creditplus.trigger.iframe', options: ['seo' => false], methods: ['GET'])]
    public function showIframeAction(Request $request, SalesChannelContext $context, string $orderTransactionId): RedirectResponse|Response
    {
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		$this->sRouteName = 'frontend.creditplus.trigger.iframe';
		$aParam = $this->getCommonVariables($orderTransactionId, $context, $request);

		if(isset($aParam['bError'])){
			if(isset($aParam['id']) && $aParam['id'] == 4){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30309,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 3){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30310,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 2){
				return $this->redirectToRoute('frontend.checkout.register.page');
			}
		}
		return $this->renderStorefront('@SicoCreditPlus/trigger/show_iframe.html.twig',$aParam);
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $orderTransactionId
	 *
	 * @return RedirectResponse|Response
	 */
    #[Route(path: '/payment/creditplus/{orderTransactionId}/new-window', name: 'frontend.creditplus.trigger.new-window', options: ['seo' => false], methods: ['GET'])]
    public function showNewWindowAction(Request $request, SalesChannelContext $context, string $orderTransactionId): RedirectResponse|Response
    {
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		$this->sRouteName = 'frontend.creditplus.trigger.new-window';
		$aParam = $this->getCommonVariables($orderTransactionId, $context, $request);

		if(isset($aParam['bError'])){
			if(isset($aParam['id']) && $aParam['id'] == 4){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30309,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 3){
				return $this->redirectToRoute('frontend.creditplus.trigger',['statusText' => 30310,'don' => $aParam['don']]);
			}
			if(isset($aParam['id']) && $aParam['id'] == 2){
				return $this->redirectToRoute('frontend.checkout.register.page');
			}
		}
		$aParam['aGetParams'] = $this->splitGetParamsForTemplate($aParam['sTargetURL']);
		return $this->renderStorefront('@SicoCreditPlus/trigger/show_new_window.html.twig',$aParam);
	}


	/**
	 * @param RequestDataBag $data
	 * @param SalesChannelContext $context
	 * @param Request $request
	 * @return RedirectResponse
	 * @throws Exception
	 */
    #[Route(path: '/payment/creditplus/beforePayment', name: 'frontend.creditplus.trigger.before-payment', options: ['seo' => false], methods: ['POST'])]
    public function beforePayment(RequestDataBag $data,SalesChannelContext $context,Request $request): RedirectResponse
    {
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		//Delete the Session Variable if the User is thrown back to the CheckoutConfirm Subscriber
		$request->getSession()->remove('cpDealerOrderNumber');

		$token = $context->getToken();

		try {
			$selectedView = $this->systemConfigService->get('SicoCreditPlus.config.sCPShownAs',$context->getSalesChannelId());
			$routeName = 'frontend.creditplus.trigger.popup';
			switch($selectedView){
				case 'popup':
					$routeName = 'frontend.creditplus.trigger.popup';
					break;
				case 'iframe':
					$routeName = 'frontend.creditplus.trigger.iframe';
					break;
				case 'new_window':
					$routeName = 'frontend.creditplus.trigger.new-window';
					break;
			}
			return $this->redirectToRoute($routeName,['orderTransactionId' => $token]);
		} catch(Exception $e){
			$this->oLogger->log(true,$e->getMessage(),[__FILE__,__LINE__,]);
			throw $e;
		}

		//return $this->forwardToRoute($routeName,['orderTransactionId' => $token]);
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 *
	 * @return Response
	 */
    #[Route(path: '/payment/creditplus/error', name: 'frontend.creditplus.trigger.error', options: ['seo' => false], methods: ['GET'])]
    public function showError(Request $request, SalesChannelContext $context): Response
    {
		return $this->renderStorefront('@SicoCreditPlus/trigger/error.html.twig');
	}

	/**
	 * @param string $orderTransactionId
	 * @param SalesChannelContext $salesChannelContext
	 * @param Request $oRequest
	 *
	 * @return array
	 */
	protected function getCommonVariables(string $orderTransactionId, SalesChannelContext $salesChannelContext, Request $oRequest):array{
		$this->oRequest = $oRequest;
		$this->_oCreditPlusHelper->setSalesChannelContext($salesChannelContext);
		$this->salesChannelContext = $salesChannelContext;
		try{
			if ($this->isInorderMode($salesChannelContext->getSalesChannelId())) {
				$oOrder = null;
				if (!preg_match('/^([0-9a-f]{32})$/', $orderTransactionId)) {
					$oOrder = $this->getCart($orderTransactionId, $salesChannelContext);
				}
				if ($oOrder == null) {
					$oOrder = $this->getOrder($orderTransactionId, $salesChannelContext);
					if ($oOrder == null) {
						$oOrder = $this->getOrderByOrderId($orderTransactionId, $salesChannelContext);
						if ($oOrder == null) {
							return [
								'id' => 1,
								'bError' => true,
								'sMessage' => 'Order Not Found'
							];
						}
					}
				}
			} else {
				$oOrder = $this->getOrder($orderTransactionId,$salesChannelContext);
				if($oOrder === null){
					//maybe the transaction id is not the transaction id. It could be the order id
					$oOrder = $this->getOrderByOrderId($orderTransactionId,$salesChannelContext);

					if($oOrder === null){
						return [
							'id' => 1,
							'bError' => true,
							'sMessage' => 'Order Not Found'
						];
					}
					$orderTransactionId = $oOrder->getTransactions()->first()->getId();
				}
			}

			if(!$this->needsToBePaid($oOrder)){
				return ['bError' => true,'id' => 4,'don' => $orderTransactionId];
			}
			if($this->isCancelledOrder($oOrder)){
				return ['bError' => true,'id' => 3,'don' => $orderTransactionId];
			}


			$sTargetURL = $this->getTargetURL($oOrder, $salesChannelContext);
			// If oOrder is an instance of Cart, then we don't need a checkup because the order ist not saved
			if($oOrder instanceof OrderEntity){
				$oCurrentOrderState = $oOrder->getTransactions()->first();
				/** @var EntityRepository $oStateMachineStateRepo */
				$oStateMachineStateRepo = $this->container->get(StateMachineStateDefinition::ENTITY_NAME.'.repository');
				if($oCurrentOrderState->getStateMachineState() == null){
					$oState = $oStateMachineStateRepo->search(new Criteria([$oCurrentOrderState->getStateId()]),$salesChannelContext->getContext())->first();
				} else {
					$oState = $oCurrentOrderState->getStateMachineState();
				}

				if($oCurrentOrderState->getStateId() != md5('borrowing_request_shown') && $oState->getTechnicalName() == 'open'){
					$this->getCreditPlusHelper()->setPaymentStatus($oOrder,'borrowing_request_shown');
				}
			}

			$sBackURL = $this->getBackUrl($salesChannelContext->getSalesChannelId(), $oOrder);
			//save the cpDealerOrderNumber
			if($oOrder instanceof Cart){
				$oOrder->getData()->remove('currency');
				$oOrder->getData()->remove('billingAddress');
				$oOrder->getData()->remove('shippingAddress');
				$oOrder->getData()->set('cpDealerOrderNumber',$oRequest->getSession()->get('cpDealerOrderNumber'));

				$this->oCartPersister->save($oOrder,$salesChannelContext);
			}
			return [
				'sTargetURL' => $sTargetURL,
				'sBackURL' => $sBackURL,
				'aGetParams' => [
					''
				]
			];
		}catch( Exception $e){
			$oRequest->getSession()->remove('cpDealerOrderNumber');
            $this->oLogger->log(true,'Error:'.$e->getMessage().' '.$e->getTraceAsString(),[]);
			return [
				'bError' => true,
				'sMessage' => $e->getMessage()
			];
		}

	}

	/**
	 * @param string $token
	 * @param SalesChannelContext $salesChannelContext
	 * @param bool $bForceSavedCart
	 * @return Cart|null
	 */
	protected function getCart(string $token, SalesChannelContext $salesChannelContext, bool $bForceSavedCart = false): ?Cart {
		if($token == null){
			return null;
		}
		try{
			/** @var EntityRepository $oSavedCartRepository */
			$oSavedCartRepository = $this->container->get(SicoCreditPlusSavedCartDefinition::ENTITY_NAME.'.repository');
			$this->oLogger->log(false, 'Searching for saved cart with params', ['token' => $token], $salesChannelContext->getSalesChannelId());
			// This is ugly, because the same token may be used more than once, so we check for order number first

			$oSearchCriteria = new Criteria();
			$oSearchCriteria->addFilter(new EqualsFilter('dealerOrderNumber', $token));
			$oSearchCriteria->addAssociations([
				'customer', 'customer.defaultBillingAddress', 'customer.defaultShippingAddress',
				'currency', 'shippingMethod',
				'paymentMethod', 'country', 'salesChannel'
			]);
			// Always find the newest entry with the same token
			$oSearchCriteria->addSorting(new FieldSorting('autoIncrement', FieldSorting::DESCENDING));
			/** @var SicoCreditPlusSavedCartEntity $oSavedCart */
			$oSavedCart = $oSavedCartRepository->search($oSearchCriteria, $salesChannelContext->getContext())->first();
			if ( !$oSavedCart ) {
				$oSearchCriteria = new Criteria();
				$oSearchCriteria->addFilter(new EqualsFilter('token', $token));
				$oSearchCriteria->addAssociations([
					'customer', 'customer.defaultBillingAddress', 'customer.defaultShippingAddress',
					'currency', 'shippingMethod',
					'paymentMethod', 'country', 'salesChannel'
				]);
				// Always find the newest entry with the same token
				$oSearchCriteria->addSorting(new FieldSorting('autoIncrement', FieldSorting::DESCENDING));
				/** @var SicoCreditPlusSavedCartEntity $oSavedCart */
				$oSavedCart = $oSavedCartRepository->search($oSearchCriteria, $salesChannelContext->getContext())->first();
			}
			if ( $bForceSavedCart && $oSavedCart ) {
				$oCustomer = $oSavedCart->getCustomer();
				$oCurrency = $oSavedCart->getCurrency();
				$this->oLogger->log(false, 'Found saved cart with token', ['token' => $token, 'id' => $oSavedCart->getId(), 'customer_id' => $oSavedCart->getCustomerId()], $salesChannelContext->getSalesChannelId());
				if ( !$oCustomer || !$oCurrency || !$oSavedCart->getSalesChannel() || !$oSavedCart->getCountry() || !$oSavedCart->getShippingMethod() || !$oSavedCart->getPaymentMethod() ) {
					$this->oLogger->log(true, 'At least one information was missing',
						[
							'customer_id' => $oSavedCart->getCustomerId(),
							'customerNull' => ($oCustomer === null),
							'currency_id' => $oSavedCart->getCurrencyId(),
							'currencyNull' => ($oCurrency === null),
							'sales_channel_id' => $oSavedCart->getSalesChannelId(),
							'salesChannelNull' => ($oSavedCart->getSalesChannel() === null),
							'country_id' => $oSavedCart->getCountryId(),
							'countryNull' => ($oSavedCart->getCountry() === null),
							'shipping_method_id' => $oSavedCart->getShippingMethodId(),
							'shippingMethodNull' => ($oSavedCart->getShippingMethod() === null),
							'payment_method_id' => $oSavedCart->getPaymentMethodId(),
							'paymentMethodNull' => ($oSavedCart->getPaymentMethod() === null),
						],
						$salesChannelContext->getSalesChannelId()
					);
				}
				$salesChannelContext->assign(['customer' => $oCustomer, 'currency' => $oCurrency, 'token' => $oSavedCart->getToken()]);
				/** @var SavedCartSerializerInterface $oSavedCartSerializer */
				$oSavedCartSerializer = $this->container->get('sico_credit_plus.saved_cart_serializer');
				$oSavedCart->fillFromPayload($oSavedCartSerializer);
				$salesChannelContext->assign(['shippingMethod' => $oSavedCart->getShippingMethod(), 'shippingMethodId' => $oSavedCart->getShippingMethodId(), 'paymentMethod' => $oSavedCart->getPaymentMethod(), 'paymentMethodId' => $oSavedCart->getPaymentMethodId()]);
				if ( $oSavedCart->getDeliveries() && $oSavedCart->getDeliveries()->first() && ($oShippingLocation = $oSavedCart->getDeliveries()->first()->getLocation()) ) {
					$salesChannelContext->assign(['shippingLocation' => $oShippingLocation]);
				}
				$oOrder = $this->oCartService->createNew($oSavedCart->getToken());
				$oOrder->setLineItems($oSavedCart->getLineItems());
				$oOrder->setRuleIds($oSavedCart->getRuleIds());
				$oOrder->setPrice($oSavedCart->getPrice());
				$oOrder->setExtensions($oSavedCart->getExtensions());
				$oOrder->setDeliveries($oSavedCart->getDeliveries());
				$oOrder->setTransactions($oSavedCart->getTransactions());
			} elseif ( $bForceSavedCart && !$oSavedCart ) {
				return null;
			} else {
				$oCustomer = $salesChannelContext->getCustomer();
				$oCurrency = $salesChannelContext->getCurrency();
				$oOrder = $this->oCartPersister->load($token,$salesChannelContext);
			}
			// I  used this to transport certain information through the program
			$oOrder->getData()->set('currency',$oCurrency);
			$billingAddress = $oCustomer->getActiveBillingAddress();
			$shippingAddress = $oCustomer->getActiveShippingAddress();
			$billingAddress->setCustomer($oCustomer);
			$shippingAddress->setCustomer($oCustomer);

			/** @var EntityRepository $salutationRepository */
			$salutationRepository = $this->container->get(SalutationDefinition::ENTITY_NAME.'.repository');
			/** @var SalutationEntity $shippingSalutation */
			$shippingSalutation = $salutationRepository->search(new Criteria([$shippingAddress->getSalutationId()]),$salesChannelContext->getContext())->first();
			/** @var SalutationEntity $billingSalutation */
			$billingSalutation = $salutationRepository->search(new Criteria([$billingAddress->getSalutationId()]),$salesChannelContext->getContext())->first();
			$billingAddress->setSalutation($billingSalutation);
			$shippingAddress->setSalutation($shippingSalutation);
			$oOrder->getData()->set('billingAddress',$billingAddress);
			$oOrder->getData()->set('shippingAddress',$shippingAddress);
			return $oOrder;
		}catch(CartTokenNotFoundException $e){
			return null;
		}
	}

	/**
	 * This function is used to remove converted saved carts from the database.
	 * The error this should remove is the duplication of orders.
	 * @param string $sDealerOrderNumber Dealer order number saved on the cart
	 * @param SalesChannelContext $salesChannelContext Sales channel context
	 * @return bool True if deletion was successful or no cart was found
	 * @throws ContainerExceptionInterface
	 * @throws NotFoundExceptionInterface
	 */
	protected function deleteSavedCart(string $sDealerOrderNumber, SalesChannelContext $salesChannelContext): bool
	{
		/** @var EntityRepository $oSavedCartRepository */
		$oSavedCartRepository = $this->container->get(SicoCreditPlusSavedCartDefinition::ENTITY_NAME.'.repository');
		$this->oLogger->log(false, 'Searching for saved cart with params to delete it', ['don' => $sDealerOrderNumber], $salesChannelContext->getSalesChannelId());
		// This is ugly, because the same token may be used more than once, so we check for order number first

		$bResult = true;
		$oSearchCriteria = new Criteria();
		$oSearchCriteria->addFilter(new EqualsFilter('dealerOrderNumber', $sDealerOrderNumber));
		$oSavedCarts = $oSavedCartRepository->search($oSearchCriteria, $salesChannelContext->getContext());
		if ( $oSavedCarts->count() ) {
			/** @var SicoCreditPlusSavedCartCollection $oSavedCartsEntities */
			$oSavedCartsEntities = $oSavedCarts->getEntities();
			foreach ( $oSavedCartsEntities as $oSavedCartEntity ) {
				$this->oLogger->log(false, 'Deleting saved cart with dealer order number', ['don' => $sDealerOrderNumber, 'id' => $oSavedCartEntity->getId()], $salesChannelContext->getSalesChannelId());
				$oEntityWrittenEvent = $oSavedCartRepository->delete([['id' => $oSavedCartEntity->getId()]], $salesChannelContext->getContext());
				$aErrors = $oEntityWrittenEvent->getErrors();
				$bResult = $bResult && empty($aErrors);
			}
		}

		return $bResult;
	}

	/**
	 * @param string $sOrderTransactionId
	 * @param SalesChannelContext $salesChannelContext
	 * @return OrderEntity|null Order, if cporder (or fallback oxid) and don are correct, null otherwise
	 */
	protected function getOrder(string $sOrderTransactionId, SalesChannelContext $salesChannelContext): ?OrderEntity {
		try {
			/** @var EntityRepository $orderTransactionRepo */
			$orderTransactionRepo = $this->container->get(OrderTransactionDefinition::ENTITY_NAME . '.repository');
			$oCriteria = new Criteria([$sOrderTransactionId]);
			$oCriteria = $oCriteria->addAssociations([
				'order',
				'order.transactions',
				'order.transactions.stateMachineState',
				'order.transactions.paymentMethod',
				'order.addresses',
				'order.addresses.salutation',
				'order.addresses.order',
				'order.addresses.order.orderCustomer',
				'order.orderCustomer',
				'order.orderCustomer.customer',
				'order.lineItems',
				'order.lineItems.product',
				'order.lineItems.product.price',//$lineItem->getProduct()->getPrice()->getCurrencyPrice()->getGross()
				'order.lineItems.product.price.currentPrice',
				'order.deliveries',
				'order.deliveries.shippingMethod',
				'order.deliveries.shippingOrderAddress',
				'order.deliveries.shippingOrderAddress.country',
				'order.deliveries.shippingOrderAddress.countryState',
				'order.deliveries.positions',
				'order.deliveries.positions.orderLineItem',
                'order.stateMachineState',
                'order.currency'
			]);
			/** @var OrderTransactionCollection $oOrderTransactionCollection */
			$oOrderTransactionCollection= $orderTransactionRepo->search(
				$oCriteria, $salesChannelContext->getContext()
			)->getEntities();
			if($oOrderTransactionCollection->count() == 0){
				return null;
			}
			/** @var OrderTransactionEntity $oOrderTransactionEntity */
			$oOrderTransactionEntity = $oOrderTransactionCollection->get($sOrderTransactionId);
            return $oOrderTransactionEntity?->getOrder();

        } catch (NotFoundExceptionInterface|InvalidUuidException|ContainerExceptionInterface $e) {
			return null;
		}

	}

    /**
     * @param Cart|OrderEntity $oOrder
     * @param SalesChannelContext $context
     * @return string|null
     */
	protected function getTargetURL(OrderEntity|Cart $oOrder, SalesChannelContext $context ): ?string
    {
		/** @var EntityRepository $oCpPaymentRepo */
		$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
		/** @var CreditPlusPaymentEntity|null $oCpPaymentEntity */
		$oCpPaymentEntity = null;
		if($oOrder instanceof OrderEntity){
			$oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('orderId',$oOrder->getId())),$context->getContext())->getEntities();
			if ( $oCpPaymentColl->count() === 0 ) {
				$oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('orderNumber', $oOrder->getCustomFieldsValue('cpDealerOrderNumber'))),$context->getContext())->getEntities();
				if ( $oCpPaymentColl->count() > 0 ) {
					$oCpPaymentEntity = $oCpPaymentColl->first();
					$oCpPaymentRepo->update([['id' => $oCpPaymentEntity->getId(), 'orderId' => $oOrder->getId()]], $context->getContext());
				}
			} else {
				$oCpPaymentEntity = $oCpPaymentColl->first();
			}
		} elseif ( $oOrder instanceof Cart ) {
			$sDealerOrderNumber = $oOrder->getData()->get('cpDealerOrderNumber');
			if ( $sDealerOrderNumber ) {
				$oCpPaymentColl = $oCpPaymentRepo->search((new Criteria())->addFilter(new EqualsFilter('orderNumber', $sDealerOrderNumber), new EqualsFilter('orderId', null)),$context->getContext())->getEntities();
				$oCpPaymentEntity = $oCpPaymentColl->first();
			}
		}


		if ( $oCpPaymentEntity && $oCpPaymentEntity->getOrderFinish() ) {
			$sTargetURL = $oCpPaymentEntity->getOrderFinish();
		} else if ( $oCpPaymentEntity && $oCpPaymentEntity->getUrl() ) {
			$sTargetURL = $oCpPaymentEntity->getUrl();
		} else {
			$sTargetURL = $this->getTargetURLOthers($oOrder, $context, $oCpPaymentEntity);
		}

		return $sTargetURL;
	}

    /**
     * @param Cart|OrderEntity $oOrder
     * @param SalesChannelContext $context
     * @param CreditPlusPaymentEntity|null $oCpPaymentEntity
     * @return string|null
     */
	protected function getTargetURLOthers(OrderEntity|Cart $oOrder, SalesChannelContext $context, ?CreditPlusPaymentEntity $oCpPaymentEntity = NULL): ?string
    {

		try{
			/** @var EntityRepository $oCpPaymentRepo */
			$oCpPaymentRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
			$oWSApi = $this->getWebshopApi($context);
			$aInput = $this->getCreditPlusAmount($oOrder,$context->getSalesChannelId());
			$sDealerOrderNumber = $this->getDealerOrderNumber($oOrder, $context->getSalesChannelId(), $oCpPaymentEntity);
			$aInput = $this->getCreditPlusAmount($oOrder, $sDealerOrderNumber);
			$oCreditPlusMainData = $this->getCreditPlusMainData($context,$oOrder);
			$aCustomerAddress = $this->getCustomerAddress($oOrder);
			$aShippingAddress = $this->getShippingAddress($oOrder);
			$sRealTriggerURL = $this->getTriggerUrl($oOrder, $sDealerOrderNumber);
			if($oOrder instanceof Cart){
				$this->saveCartForTrigger($context, $oOrder, $sDealerOrderNumber);
			}
			$sRealGoodFinishUrl = '';
			$sGoodFinishURL = $this->getGoodFinishURL($oOrder,$sRealGoodFinishUrl, $context->getSalesChannelId(), $sDealerOrderNumber);
			$sStatusChangeURL = $this->getStatusChangeURL($oOrder,$context->getSalesChannelId(), $sDealerOrderNumber);
			$bInOrder = $this->isInorderMode($context->getSalesChannelId());
			$aShopIntegrationData = $this->getShopIntegrationData($oOrder, $sRealTriggerURL,$sGoodFinishURL,$bInOrder,$sStatusChangeURL);

			$oCreditOfferResponse = $oWSApi->createCreditOfferCPWebshop($aInput, $oCreditPlusMainData, $aCustomerAddress, $aShippingAddress, $aShopIntegrationData);

			// isset Response && Value is not empty string
			if (isset($oCreditOfferResponse->createCreditOffer->customerUrl) && $oCreditOfferResponse->createCreditOffer->customerUrl) {
				if(!isset($oCpPaymentEntity) ||!$oCpPaymentEntity){
					$aCreateData = [
						'genTimestamp' => time(),
						'url' => $oCreditOfferResponse->createCreditOffer->customerUrl,
						'backUrl' => $sRealGoodFinishUrl,
						'statusChangeUrl' => $sRealTriggerURL,
						];
					if($oOrder instanceof OrderEntity){
						$sOrderId = $oOrder->getId();
						$sToken = null;
						$aCreateData['orderId'] = $sOrderId;
						$aCreateData['orderNumber'] = $sDealerOrderNumber;
						$aCreateData['token'] = $sToken;
						$aCreateData['versionId'] = $oOrder->getVersionId();
					} else {
						$sOrderId = null;
						$sToken = $this->salesChannelContext->getToken();
						$aCreateData['orderId'] = $sOrderId;
						$aCreateData['orderNumber'] = $sDealerOrderNumber;
						$aCreateData['token'] = $sToken;
						$aCreateData['versionId'] = $context->getContext()->getVersionId();
					}
					$oCpPaymentRepo->create([
						$aCreateData
					],$context->getContext());


					$sUrl = $oCreditOfferResponse->createCreditOffer->customerUrl;
				} else {
					$iCpPaymentEntityId = $oCpPaymentEntity->getId();
					$sUrl = $oCpPaymentEntity->getUrl();
					$aUpdate = [];
					if ( !$sUrl ) {
						// If url is empty
						$sUrl = $oCreditOfferResponse->createCreditOffer->customerUrl;
						$aUpdate['url'] = $sUrl;
						$oCpPaymentEntity->setUrl($sUrl);
					}
					if ( !$oCpPaymentEntity->getStatusChangeUrl() ) {
						$aUpdate['statusChangeUrl'] = $sRealTriggerURL;
						$oCpPaymentEntity->setStatusChangeUrl($sRealTriggerURL);
					}
					if ( !$oCpPaymentEntity->getBackUrl() ) {
						$aUpdate['backUrl'] = $sRealGoodFinishUrl;
						$oCpPaymentEntity->setBackUrl($sRealGoodFinishUrl);
					}
					if ( !$sDealerOrderNumber ) {
						// If order number is empty
						if ( $oOrder instanceof OrderEntity ) {
							$aUpdate['orderNumber'] = $oOrder->getOrderNumber();
						} else {
							$aUpdate['orderNumber'] = $oOrder->getData()->get('cpDealerOrderNumber');
						}
					}
					if ( $aUpdate ) {
						// If there is anything to update
						$aUpdate['id'] = $iCpPaymentEntityId;
						$oCpPaymentRepo->update([$aUpdate], $context->getContext());
					}
				}
				return $sUrl;
			}

			if($this->isInorderMode($context->getSalesChannelId())){
				return $this->router->generate('frontend.creditplus.trigger', ['statusText' => 30308]);
			}
			else {
				return $this->router->generate('frontend.creditplus.trigger', ['statusText' => 30308]);
			}

		} catch( Exception $e){
			$sTrace = '';
			foreach($e->getTrace() as $trace){
				if(is_array($trace)){
					if(isset($trace['file']) && isset($trace['line'])){
						$sTrace.= $trace['file'].':'.$trace['line'].'||';
					}
				}
			}
			$this->oLogger->log(true,$e->getMessage(),[__FILE__,__LINE__,$e->getFile(),$e->getLine(),$sTrace]);
			return '';
		}
	}

	/**
	 * @param OrderEntity|Cart $oOrder
	 * @param string $sSalesChannelId
	 * @param CreditPlusPaymentEntity|null $oCpPaymentEntity
	 *
	 * @return string
	 */
	protected function getDealerOrderNumber($oOrder, string $sSalesChannelId, ?CreditPlusPaymentEntity $oCpPaymentEntity = null): string
	{
		if ( $oCpPaymentEntity && $oCpPaymentEntity->getOrderNumber() ) {
			return $oCpPaymentEntity->getOrderNumber();
		}
		$sDealerOrderNumber = '';
		if($oOrder instanceof OrderEntity){
			$sDealerOrderNumber .= $oOrder->getOrderNumber();
		} else {
			$sDealerOrderNumber = $this->oSession->get('cpDealerOrderNumber');
			if($sDealerOrderNumber == null){
				$sDealerOrderNumber = 'CP'.date('YmdHis');
				$this->oSession->set('cpDealerOrderNumber',$sDealerOrderNumber);
				$oOrder->getData()->set('cpDealerOrderNumber',$sDealerOrderNumber);
			}
		}

		//$oConfig = Shopware()->Config();
		$sTransactionNumber = $this->systemConfigService->get('SicoCreditPlus.config.sTransactionNumber', $sSalesChannelId);
		if ( $sTransactionNumber === 'cpnumber' ) {
			$sDealerOrderNumber = $this->oSession->get('cpDealerOrderNumber');
			if($sDealerOrderNumber == null){
				$sDealerOrderNumber = 'CP'.date('YmdHis');
				$this->oSession->set('cpDealerOrderNumber', $sDealerOrderNumber);
			}
		}
		return $sDealerOrderNumber;
	}

	/**
     * @param SalesChannelContext $context
	 * @return CreditPlusWebshopAPI
	 */
	protected function getWebshopApi(SalesChannelContext $context): CreditPlusWebshopAPI {
		return $this->getCreditPlusHelper()->getWebshopAPI($context->getSalesChannelId());
	}

    /**
     * @return SicoCreditPlusHelper|null
     */
	protected function getCreditPlusHelper(): ?SicoCreditPlusHelper
    {
		return $this->_oCreditPlusHelper;
	}

    /**
     * @param OrderEntity|Cart $oOrder
     * @param string $sDealerOrderNumber
     *
     * @return array
     */
    protected function getCreditPlusAmount( $oOrder, string $sDealerOrderNumber ): array
    {
        $aInput = [
            'agentId' => 'Webshop',
            'loanAmount' => 0.0,
            'dealerOrderNumber' => $sDealerOrderNumber,
        ];
        if ($oOrder instanceof OrderEntity){
            $aInput['loanAmount'] = $oOrder->getAmountTotal();
        } elseif ($oOrder instanceof Cart) {
            $aInput['loanAmount'] = $oOrder->getPrice()->getTotalPrice();
        }

        return $aInput;
    }

    /**
     * @param SalesChannelContext $context
     * @param Cart|OrderEntity $oOrder
     *
     * @return CreditPlusMainData
     * @throws Exception
     */
	protected function getCreditPlusMainData(SalesChannelContext $context, OrderEntity|Cart $oOrder ): CreditPlusMainData
    {
		$oCreditPlusMainData = new CreditPlusMainData();
		$dealerNumber = $this->systemConfigService->get('SicoCreditPlus.config.sCPDealer',$context->getSalesChannelId());
		$dealerName = $this->systemConfigService->get('SicoCreditPlus.config.sDealerName',$context->getSalesChannelId());
		$soapUser = $this->systemConfigService->get('SicoCreditPlus.config.sSoapUser',$context->getSalesChannelId());
		$soapPass = $this->systemConfigService->get('SicoCreditPlus.config.sSoapPass',$context->getSalesChannelId());
		if($soapPass == null){
			$soapPass = '';
		}
		$soapType = $this->systemConfigService->get('SicoCreditPlus.config.sSoapType',$context->getSalesChannelId());
		$partnerName = $this->systemConfigService->get('SicoCreditPlus.config.sPartnerName',$context->getSalesChannelId());
		if($partnerName == null){
			$partnerName = '';
		}
		$wsdl = $this->systemConfigService->get('SicoCreditPlus.config.sWSDL',$context->getSalesChannelId());
		$oFinancingArticle = $this->getCreditPlusHelper()->getFinancingArticleReference($context,$oOrder);

		$oCreditPlusMainData
			->setDealerNumber($dealerNumber)
			->setDealerName($dealerName)
			->setSoapUser($soapUser)
			->setSoapPass($soapPass)
			->setSoapType($soapType)
			->setPartnerName($partnerName)
			->setProductTypeID($oFinancingArticle->productTypeID)
			->setProductClassID($oFinancingArticle->productClassID)
			->setWSDL($wsdl);
		return $oCreditPlusMainData;
	}

	/**
	 * @param Cart|OrderEntity $oOrder
	 *
	 * @return array Billing address
	 */
	protected function getCustomerAddress(OrderEntity|Cart $oOrder ): array
    {
		if($oOrder instanceof OrderEntity){
			foreach($oOrder->getAddresses() as $address){
				if ($oOrder->getBillingAddressId() === $address->getId()){
					return $this->getAddress($address);
				}
			}
		} else {
			return $this->getAddress($oOrder->getData()->get('billingAddress'));
		}

		return [];
	}

	/**
	 * @param Cart|OrderEntity $oOrder
	 *
	 * @return array Shipping address
	 */
	protected function getShippingAddress(OrderEntity|Cart $oOrder ): array
    {
		if($oOrder instanceof OrderEntity){
			$oAddresses = $oOrder->getAddresses();

			if($oAddresses->count() > 1){
				foreach($oOrder->getAddresses() as $address){
					if ($oOrder->getBillingAddressId() !== $address->getId()){
						return $this->getAddress($address);
					}
				}
			}
			else{
				$oAddress = $oAddresses->first();
				return $this->getAddress($oAddress);
			}
		} else {
			return $this->getAddress($oOrder->getData()->get('shippingAddress'));
		}

		return [];
	}

	/**
	 * @param CustomerAddressEntity|OrderAddressEntity $oAddress
	 * @return array Address as required for this plugin
	 */
	protected function getAddress(CustomerAddressEntity|OrderAddressEntity $oAddress ): array
    {
		$iDefaultFinancingTime = 6;

		$sStreet = $oAddress->getStreet();
		$aStreet = explode(' ',$sStreet);
		$sStreetNo = array_pop($aStreet);
		if($oAddress instanceof OrderAddressEntity){
			$sEmail = $oAddress->getOrder()->getOrderCustomer()->getEmail();
		}
		else {
			$sEmail = $oAddress->getCustomer()->getEmail();
		}

		return array(
			'salutation' => ($oAddress->getSalutation()->getSalutationKey() == 'mr')?1:2,
			'salutation_string' => ($oAddress->getSalutation()->getSalutationKey() == 'mr')?'Herr':'Frau',
			'city' => $oAddress->getCity(),
			'postalCode' => $oAddress->getZipCode(),
			'street' => $sStreet,
			'Hausnummer' => $sStreetNo,
			'firstName' => $oAddress->getFirstName(),
			'Laufzeit' => $iDefaultFinancingTime,
			'lastName' => $oAddress->getLastName(),
			'email' => $sEmail
		);
	}

	/**
	 * @param OrderEntity|Cart $oOrder
	 * @param string $sRealGoodFinishUrl
	 * @param string $sSalesChannelId
	 * @param string $sDealerOrderNumber
	 *
	 * @return string
	 * @throws Exception
	 */
	protected function getGoodFinishURL($oOrder, &$sRealGoodFinishUrl, string $sSalesChannelId, string $sDealerOrderNumber = ''){
		$sRealGoodFinishUrl = $this->getStatusChangeURL($oOrder, $sSalesChannelId, $sDealerOrderNumber, '7');

		if($oOrder instanceof OrderEntity){
			$ordernumber = $oOrder->getOrderNumber();
		} else {
			$ordernumber = $oOrder->getData()->get('cpDealerOrderNumber');
			if(empty($ordernumber)){
				$ordernumber = $this->oSession->get('cpDealerOrderNumber');
			}
		}
		if(empty($ordernumber)){
			throw new Exception('Ordernumber empty');
		}
        return $this->generateUrl('frontend.creditplus.trigger.after-credit-form',['ordernumber' => $ordernumber,'route' => 1],UrlGeneratorInterface::ABSOLUTE_URL);
	}

	/**
	 * @param Cart|OrderEntity $oOrder Order object linked to the payment
	 * @param string $sSalesChannelId Sales Channel ID
	 * @param string $sDealerOrderNumber Dealer Order Number
	 * @param string $sStatus Defaults to 3 (trigger by CreditPlus), used with 7 on direct returns
	 * @return string
	 * @throws Exception
	 */
	protected function getStatusChangeURL($oOrder, string $sSalesChannelId, string $sDealerOrderNumber, string $sStatus = '3'){
		$sSaltValue = $this->systemConfigService->get('SicoCreditPlus.config.sSignatureSalt', $sSalesChannelId);
		$sURL = $this->getTriggerUrl($oOrder, $sDealerOrderNumber);
		$sCheckURL = $sURL.'&'.$sSaltValue.$sStatus;
		$sSignature = md5($sCheckURL);
		$sURL .= 'callback='.$sStatus.'&signature='.$sSignature;

		return $sURL;

	}

	protected function turnSlash2Minus($value): array|string
    {
		return str_replace('/','-',$value);
	}
	protected function turnMinus2Slash($value): array|string
    {
		return str_replace('-','/',$value);
	}

	/**
	 * @param OrderEntity|Cart $oOrder
	 * @param string $sDealerOrderNumber
	 *
	 * @return string
	 * @throws Exception
	 */
	protected function getTriggerUrl( $oOrder, string $sDealerOrderNumber = '' ) {
		$aData = array(
			'sRequestTime' => date('YmdHis'),
			'sAddressMD5' => $this->getAddressesMD5($oOrder)
		);
		$sData = urlencode($this->turnSlash2Minus(base64_encode(gzdeflate(serialize($aData)))));

		try{
			return $this->getTriggerUrlWithDataAndDon($sData, $sDealerOrderNumber);
		}catch(Exception $e){
			if(!$sDealerOrderNumber){
				$sDealerOrderNumber = 'Don Not Set';
			}
			$this->oLogger->log(true,'Could not generate TriggerURL: '.$e->getMessage(),[__FILE__,__LINE__,'don' => $sDealerOrderNumber]);
			throw $e;
		}
	}

	/**
	 * Returns md5 string for addresses
	 *
	 * @param Cart|OrderEntity $oOrder
	 * @return string md5(serialize(array('billing' => $aBillingAddress, 'shipping' => $aShippingAddress)))
	 */
	protected function getAddressesMD5(OrderEntity|Cart $oOrder ): string
    {
		$aCustomerAddress = $this->getCustomerAddress($oOrder);
		$aShippingAddress = $this->getShippingAddress($oOrder);
		$aMD5Array = array(
			'billing' => $aCustomerAddress,
			'shipping' => $aShippingAddress
		);
		return md5(serialize($aMD5Array));
	}

	protected function getTriggerUrlWithDataAndDon($sData, $sDON): string
    {
		$sTriggerURL = $this->router->generate('frontend.creditplus.trigger.dispatch',
			[
				'sData' => $sData,
				'sDon' => $sDON
			], UrlGeneratorInterface::ABSOLUTE_URL);

		if(strpos($sTriggerURL,'?') === false){
			$sTriggerURL .= '?';
		}else{
			if(substr($sTriggerURL,-1) === '?'){
				// Ends with ? - no need to change
			} elseif ( substr($sTriggerURL, -1) === '&' ) {
				// Ends with & - no need to change
			} else {
				$sTriggerURL .= '&';
			}
		}

		return $sTriggerURL;
	}

	/**
	 * @param Cart|OrderEntity $oOrder
	 * @param string $sTriggerURL URL used for status changes
	 * @param string $sGoodFinishURL URL used only on inorder finish to carry you to the order finished page
	 * @param bool $bInOrder True if order is created after payment ("Vor Bestellabschluss")
	 * @param string $sStatusChangeURL Trigger URL with status 3
	 * @return array
	 * @throws ContainerExceptionInterface
	 * @throws NotFoundExceptionInterface
	 */
	protected function getShopIntegrationData(OrderEntity|Cart $oOrder, string $sTriggerURL, string $sGoodFinishURL = '', bool $bInOrder = false, string $sStatusChangeURL = '' ): array
    {
		$aCart = array();
		//$oArticleRep = Shopware()->Models()->getRepository('\Shopware\Models\Article\Article');
		/** @var OrderLineItemEntity|LineItem $lineItem */
		foreach($oOrder->getLineItems() as $lineItem){
			$dUnitPrice = $dRetailPrice = $lineItem->getPrice()->getUnitPrice();
			if($lineItem instanceof OrderLineItemEntity){
				$bGood = $lineItem->getGood();
				if($bGood){
					$sEan = $lineItem->getProduct()->getEan();
					if(empty($sEan)){
						$sEan = $lineItem->getProduct()->getProductNumber();
					}

					$sProductName = $lineItem->getProduct()->getName();
				} else {
					$sProductName = $lineItem->getLabel();
					$aPayload = $lineItem->getPayload();
					if(isset($aPayload['promotionId'])){ // Promotions does not have a ean so i use the promotion id
						$sEan = $aPayload['promotionId'];
					} else {
						$sEan = $lineItem->getId(); // I use the line item id because i don't know which item i have
					}
				}
			} else { //  if($lineItem instanceof LineItem)
				$bGood = $lineItem->isGood();
				if($bGood){
					/** @var EntityRepository $oProductRepo */
					$oProductRepo = $this->container->get(ProductDefinition::ENTITY_NAME.'.repository');

					$oProductCollection = $oProductRepo->search(new Criteria([$lineItem->getReferencedId()]),$this->salesChannelContext->getContext());

					/** @var ProductEntity $oProduct */
					$oProduct = $oProductCollection->first();
					$sEan = $oProduct->getEan();
					if(empty($sEan)){
						$sEan = $oProduct->getProductNumber();
					}
					$sProductName = $oProduct->getName();
				}
				else {
					$sProductName = $lineItem->getLabel();
					$aPayload = $lineItem->getPayload();
					if(isset($aPayload['promotionId'])){
						// Promotions do not have an ean, so I use the promotion id instead
						$sEan = $aPayload['promotionId'];
					} else {
						// I use the line item id because I don't know which item I have
						$sEan = $lineItem->getId();
					}
				}

			}

			if(!$bGood){
				$oShoppingCartItem = new ShoppingCartItem(array(
					'amount' => $lineItem->getQuantity(),//$oDetail->getQuantity(),
					'description' => mb_substr($sProductName,0,50),//mb_substr($oDetail->getArticleName(), 0, 50),
					'ean' => substr($sEan,0,13),
					'listPrice' => $dRetailPrice,
					'promotion' => false,
					'sellingPrice' => $dUnitPrice
				));
				$aCart[] = $oShoppingCartItem;
				continue;
			}
			if($lineItem instanceof OrderLineItemEntity){
				$dRetailPrice = $lineItem->getProduct()->getPrice()->getCurrencyPrice($this->salesChannelContext->getCurrency()->getId())->getGross();
			}else {
				if($lineItem->isGood()){
					//Should never happen but to check it
					if(!isset($oProduct)){
						/** @var EntityRepository $oProductRepo */
						$oProductRepo = $this->container->get(ProductDefinition::ENTITY_NAME.'.repository');
						$oProductCollection = $oProductRepo->search(new Criteria([$lineItem->getReferencedId()]),$this->salesChannelContext->getContext());
						/** @var ProductEntity $oProduct */
						$oProduct = $oProductCollection->first();
					}

					$oRetailPrice = $lineItem->getPrice()->getListPrice();
					if($oRetailPrice) { // if pseudo price does not exits we use the unit Price again
						$dRetailPrice = $oRetailPrice->getPrice();
					} else {
						$dRetailPrice = $lineItem->getPrice()->getUnitPrice();
					}
				}
			}

			$oShoppingCartItem = new ShoppingCartItem(array(
				'amount' => $lineItem->getQuantity(),
				'description' => mb_substr($sProductName, 0, 50),
				'ean' => substr($sEan,0,13),
				'listPrice' => $dRetailPrice,
				'promotion' => false,
				'sellingPrice' => $dUnitPrice
			));
			if ( $oShoppingCartItem->getSellingPrice() < $oShoppingCartItem->getListPrice() ) {
				$oShoppingCartItem->setPromotion(true);
			}
			$aCart[] = $oShoppingCartItem;
		}

        return array(
            'triggerUrl' => $sTriggerURL,
            'finishedUrl' => $sGoodFinishURL,
            'inorder' => $bInOrder,
            'statusChangeUrl' => $sStatusChangeURL,
            'shoppingCart' => $aCart
        );
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $sOrderTransactionId
	 *
	 * @return RedirectResponse
	 * @throws Exception
	 */
    #[Route(path: '/payment/creditplus/{sOrderTransactionId}/cancelandreorder', name: 'frontend.creditplus.trigger.cancelandreorder', options: ['seo' => false], methods: ['GET'])]
    public function cancelAndReorderAction(Request $request, SalesChannelContext $context, string $sOrderTransactionId): RedirectResponse
    {
        $this->oSession = $request->getSession();
        $this->getCreditPlusHelper()->setSession($this->oSession);
		$oOrder = $this->getOrder($sOrderTransactionId,$context);
		$iPayment = $request->get('disablePayment');
		if($iPayment == 1){
			$request->getSession()->set('bSicoFinancingExcluded',true);
		}
		if ($oOrder != null){
			//storniert die alte Bestellung
			$this->_oCreditPlusHelper->setSalesChannelContext($context);
			$this->_oCreditPlusHelper->setPaymentStatus($oOrder,'creditplus_cancelled');
			//Konvertiert die Bestellung in einen Warenkorb um
			$oCart = $this->oOrderConverter->convertToCart($oOrder,$context->getContext());
			/*
			 * Diese Beiden sorgen wohl dafür, dass Shopware versucht die Bestellung upzudaten. Dies sorgt dafür,
			 * dass bei der zweiten Bestellung zu einem Fehler kommt.
			*/
			$oCart->removeExtension('originalId');
			$oCart->removeExtension('originalOrderNumber');
			// Hier wird der Warenkorb abgespeichert, damit er wieder gefunden werden kann.
			$this->oCartPersister->save($oCart,$context);

			$this->getCreditPlusHelper()->setOrderStatus($oOrder,'cancel');

			$finishUrl = $this->router->generate('frontend.checkout.cart.page',['token' => $oCart->getToken()]);

			return new RedirectResponse($finishUrl);
		}
		else {
			$url = $this->router->generate('frontend.creditplus.trigger',['statusText' => 30305]);
			return new RedirectResponse($url);
		}
	}

	protected function getOrderByOrderId($sOrderId, SalesChannelContext $context): OrderEntity
    {
		/** @var EntityRepository $orderRepo */
		$orderRepo = $this->container->get(OrderDefinition::ENTITY_NAME . '.repository');

		$oCriteria =new Criteria([$sOrderId]);
		$oCriteria = $oCriteria->addAssociations([

			'transactions',
			'transactions.stateMachineState',
			'transactions.paymentMethod',
			'addresses',
			'addresses.salutation',
			'addresses.order',
			'addresses.order.orderCustomer',
			'orderCustomer',
			'orderCustomer.customer',
			'lineItems',
			'lineItems.product',
			'lineItems.product.price',
			'lineItems.product.price.currentPrice',
            'stateMachineState',
            'currency'
		]);
		/** @var OrderEntity $oOrderEntity */
		$oOrderEntity = $orderRepo->search(
			$oCriteria,$context->getContext()
		)->get($sOrderId);
		return $oOrderEntity;
	}

	/**
	 * @param string $sTargetURL
	 *
	 * @return array
	 */
	protected function splitGetParamsForTemplate(string $sTargetURL ): array
    {
		//$sTargetURL = $oTemplate->getTemplateVars('sTargetURL');
		$aGetParams = array();
		if (str_contains($sTargetURL, '?')) {
			$aTargetURL = explode('?', $sTargetURL);
			if ( $aTargetURL[1] ) {
				$aParamBlocks = explode('&', $aTargetURL[1]);
				foreach ( $aParamBlocks as $sParamBlock ) {
					$aParam = explode('=', $sParamBlock);
					$aGetParams[urldecode($aParam[0])] = urldecode($aParam[1]);
				}
			}
		}
		//$oTemplate->assign('aGetParams', $aGetParams);

		return $aGetParams;
	}

    /**
     * @param string $sSalesChannelId
     * @param Cart|OrderEntity $oOrder
     * @return string
     */
    protected function getBackUrl(string $sSalesChannelId, Cart|OrderEntity $oOrder): string
    {
		if ($this->isInorderMode($sSalesChannelId) && ($oOrder instanceof Cart) ) {
			$sBackUrl = $this->router->generate('frontend.checkout.confirm.page');
		} else {
			$sBackUrl = $this->router->generate('frontend.creditplus.trigger',[
				'statusText' => 30303,
				'cporder' => $oOrder->getOrderNumber(),
				'don' => $oOrder->getTransactions()->first()->getId()
			]);
		}
		return $sBackUrl;
	}

	/**
	 * @var bool $_bIsInorderMode
	 */
	private ?bool $_bIsInorderMode = null;

    /**
     * Prüft, ob die aktuelle Einstellung "inorder" ist und speichert dies zwischen.
     * @param string $sSalesChannelId
     * @return bool|null
     */
	private function isInorderMode(string $sSalesChannelId): ?bool
    {
		if ( $this->_bIsInorderMode === null ) {
			$this->_bIsInorderMode = ($this->systemConfigService->get('SicoCreditPlus.config.sTransactionMode',$sSalesChannelId) == 'inorder');
		}
		return $this->_bIsInorderMode;
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param RequestDataBag $data
	 *
	 * @return Response
	 */
    #[Route(path: '/payment/creditplus/beforeOrder', name: 'frontend.creditplus.trigger.before-order', options: ['seo' => false], methods: ['GET'])]
    public function beforeOrderAction(RequestDataBag $data,SalesChannelContext $context,Request $request): Response
    {
		return $this->renderStorefront('@SicoCreditPlus/trigger/before_order.html.twig');
	}

	/**
	 * @param Request $request
	 * @param SalesChannelContext $context
	 * @param string $ordernumber
	 * @param int $route
	 * @return Response
	 * @throws ContainerExceptionInterface
	 * @throws NotFoundExceptionInterface
	 */
    #[Route(path: '/payment/creditplus/afterCreditForm/{ordernumber}/{route}', name: 'frontend.creditplus.trigger.after-credit-form', options: ['seo' => false], methods: ['GET'])]
    public function afterCreditForms(Request $request, SalesChannelContext $context, string $ordernumber, int $route): Response
    {
		$this->oSession = $request->getSession();
		$this->getCreditPlusHelper()->setSession($this->oSession);
		/** @var EntityRepository $oCpRepo */
		$oCpRepo = $this->container->get(CreditPlusPaymentDefinition::ENTITY_NAME.'.repository');
		/** @var CreditPlusPaymentEntity $oCpEntity */
		$oCpEntity = $oCpRepo->search((new Criteria())->addFilter(new EqualsFilter('orderNumber',$ordernumber)),$context->getContext())->first();

		if($oCpEntity === null){
			$this->oLogger->log(true,'Credit Plus Payment Entry Not Found',['file' => __FILE__,'line' => __LINE__,'ordernumber' => $ordernumber]);
			$sURL = $this->getForwardURL($context->getSalesChannelId(),array('index', null, null, array('statusText' => 30311)));
			return $this->renderStorefront('@SicoCreditPlus/trigger/afterCreditForms.html.twig',['url' => $sURL]);
		}

		if($route == 1){
			$sUrl = $oCpEntity->getBackUrl();
		} else if($route == 2){
			$callback = $request->get('callback');
			$signature = $request->get('signature');

			$sUrl = $oCpEntity->getStatusChangeUrl();
			$bFirst = true;
			if($callback){

				$sUrl .= 'callback='.$callback;
				$bFirst = false;
			}
			if($signature) {
				if(!$bFirst){
					$sUrl .= '&';
				}
				$sUrl .= 'signature='.$signature;
				$bFirst = false;
			}
		}

		return $this->renderStorefront('@SicoCreditPlus/trigger/afterCreditForms.html.twig',['url' => $sUrl]);
	}

	/**
	 * @param Cart|OrderEntity $oOrder
	 * @return bool true if cancelled, false if not
	 */
	protected function isCancelledOrder(OrderEntity|Cart $oOrder): bool
    {
		//Cart is not cancelable
		if($oOrder instanceof Cart){
			return false;
		}

		$oCancelledPaymentState = $this->_oCreditPlusHelper->getPaymentStateByName('cancelled');
		$oCancelledOrderState = $this->_oCreditPlusHelper->getOrderStateByName('cancelled');

		$sOrderStatusId = $oOrder->getStateMachineState()->getId();
		$aTransactions = $oOrder->getTransactions();
		if($sOrderStatusId == $oCancelledOrderState->getId()){
			return true;
		}

		foreach($aTransactions as $oTransaction) {
			if($oTransaction->getId() == $oCancelledPaymentState->getId()){
				return true;
			}
		}

		return false;
	}

	/**
	 * @param SalesChannelContext $oSalesChannelContext
	 * @param Cart $oOrder
	 * @param string $sOrderNumber
	 * @return SicoCreditPlusSavedCartEntity
	 * @throws ContainerExceptionInterface
	 * @throws NotFoundExceptionInterface
	 * @throws Exception
	 */
	protected function saveCartForTrigger(SalesChannelContext $oSalesChannelContext, Cart $oOrder, string $sOrderNumber): SicoCreditPlusSavedCartEntity
    {
		/** @var EntityRepository $oSavedCartRepo */
		$oSavedCartRepo = $this->container->get(SicoCreditPlusSavedCartDefinition::ENTITY_NAME.'.repository');
		/** @var SavedCartSerializerInterface $oSavedCartSerializer */
		$oSavedCartSerializer = $this->container->get('sico_credit_plus.saved_cart_serializer');
		$oSearchCriteria = new Criteria();
		$oSearchCriteria->addFilter(new EqualsFilter('dealerOrderNumber', $sOrderNumber));
		/** @var SicoCreditPlusSavedCartCollection $oSavedCartCollection */
		$oSavedCartCollection = $oSavedCartRepo->search($oSearchCriteria, $oSalesChannelContext->getContext());
		if ( $oSavedCartCollection->count() ) {
			/** @var SicoCreditPlusSavedCartEntity $oSavedCart */
			$oSavedCart = $oSavedCartCollection->first();
			$oSavedCart->createFromCart($oSalesChannelContext, $oOrder, $oSavedCartSerializer);
			$aData = $oSavedCart->getVars();
			$oSavedCartRepo->update([$aData], $oSalesChannelContext->getContext());
		} else {
			$oSavedCart = new SicoCreditPlusSavedCartEntity();
			$oSavedCart->createFromCart($oSalesChannelContext, $oOrder, $oSavedCartSerializer);
			if ( !$oSavedCart->getDealerOrderNumber() ) {
				$oSavedCart->setDealerOrderNumber($sOrderNumber);
			}
			$aData = $oSavedCart->getVars();
			$this->oLogger->log(false, 'Attempting to create saved cart', ['cart' => $aData], $oSalesChannelContext->getSalesChannelId());
			$oSavedCartRepo->create([$aData], $oSalesChannelContext->getContext());
		}
		return $oSavedCart;
	}

    /**
     * @param Cart|OrderEntity $oOrder
     * @return bool
     */
	protected function needsToBePaid(OrderEntity|Cart $oOrder): bool
    {
		//Cart must always be paid(Otherwise you are a thief ;-) )
		if($oOrder instanceof Cart){
			return true;
		}
		if ( $this->isInorderMode($this->salesChannelContext->getSalesChannelId()) ) {
			// Status 24 may still need ID verification
			$oContract = $this->_oCreditPlusHelper->getContractData($this->salesChannelContext->getSalesChannelId(), $oOrder);
			if ( $oContract && !$oContract->finallyApproved ) {
				return true;
			}
		}



		$oOpenPaymentState = $this->_oCreditPlusHelper->getPaymentStateByName('open');
		$oBorrowingRequestShownPaymentState = $this->_oCreditPlusHelper->getPaymentStateByName('borrowing_request_shown');
		$oOpenOrderState = $this->_oCreditPlusHelper->getOrderStateByName('open');
		$oAcceptedPaymentState = $this->_oCreditPlusHelper->getPaymentStateByName('creditplus_accepted');
		$oApprovedPaymentState = $this->_oCreditPlusHelper->getPaymentStateByName('creditplus_approved');

		$sOrderStatusId = $oOrder->getStateMachineState()->getId();
		$aTransactions = $oOrder->getTransactions();


		$ret = true;
		$ret = $ret && $sOrderStatusId == $oOpenOrderState->getId();

		foreach($aTransactions as $oTransaction) {
			$ret = $ret && $oTransaction->getStateId() == $oOpenPaymentState->getId() ||
				$oTransaction->getStateId() == $oBorrowingRequestShownPaymentState->getId() ||
				$oTransaction->getStateId() == $oAcceptedPaymentState->getId() ||
				$oTransaction->getStateId() == $oApprovedPaymentState->getId();
		}
		return $ret;
	}
}
