<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 22.04.2025
 * @time 15:02
 */

namespace SicoCreditPlus\Core\Checkout\Payment\SalesChannel;

use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Checkout\Payment\SalesChannel\AbstractPaymentMethodRoute;
use Shopware\Core\Checkout\Payment\SalesChannel\PaymentMethodRouteResponse;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Service\CreditPlusPayment;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Attribute\Route;

#[Route(defaults: ['_routeScope' => ['store-api']])]
class SicoCreditPlusPaymentMethodRouteDecorator extends AbstractPaymentMethodRoute
{
	/** @var AbstractPaymentMethodRoute|null Set on constructor with parent object */
	protected ?AbstractPaymentMethodRoute $decorator = null;

	/** @var SicoCreditPlusHelper|null Set on constructor */
	private ?SicoCreditPlusHelper $creditPlusHelper = null;

	/** @var CartService|null Set on constructor */
	protected ?CartService $cartService = null;

	/** @var SystemConfigService|null Set on constructor */
	protected ?SystemConfigService $systemConfigService = null;

	/** @var SalesChannelRepository */
	private SalesChannelRepository $productRepo;

	/**
	 * @param AbstractPaymentMethodRoute $decorator Original payment method route being decorated through services.xml
	 * @param CartService $cartService The service to access Cart objects
	 * @param SystemConfigService $systemConfigService The system config service
	 * @param SicoCreditPlusHelper $creditPlusHelper The helper to access CreditPlus API
	 * @param SalesChannelRepository $productRepo The product repository
	 */
	public function __construct(
		AbstractPaymentMethodRoute $decorator,
		CartService $cartService,
		SystemConfigService $systemConfigService,
		SicoCreditPlusHelper $creditPlusHelper,
		SalesChannelRepository $productRepo
	) {
		$this->decorator = $decorator;
		if ( !$this->cartService ) {
			$this->cartService = $cartService;
		}
		if ( !$this->creditPlusHelper ) {
			$this->creditPlusHelper = $creditPlusHelper;
		}
		$this->systemConfigService = $systemConfigService;
		$this->productRepo = $productRepo;
	}

	public function getDecorated(): AbstractPaymentMethodRoute
	{
		return $this->decorator;
	}

	#[Route(path: '/store-api/payment-method', name: 'store-api.payment.method', defaults: ['_entity' => 'payment_method'], methods: ['GET', 'POST'])]
	public function load(Request $request, SalesChannelContext $context, Criteria $criteria): PaymentMethodRouteResponse
	{
		$this->creditPlusHelper->setSalesChannelContext($context);
		$oCart = $this->cartService->getCart($context->getToken(), $context);

		if ( !$this->isCreditPlusPaymentMethodAvailable($oCart, $context, $request) ) {
			$criteria->addFilter(new NotFilter('AND', [
				new EqualsFilter('handlerIdentifier', CreditPlusPayment::class),
			]));
		}

		return $this->getDecorated()->load($request, $context, $criteria);
	}

	public function isCreditPlusPaymentMethodAvailable(Cart $oCart, SalesChannelContext $oSalesChannelContext, Request $oRequest): bool
	{
		$aArticleNumbers = [];
		/** @var LineItem $oLineItem */
		foreach($oCart->getLineItems() as $oLineItem){
			if ( $oLineItem->getType() !== LineItem::PRODUCT_LINE_ITEM_TYPE ) { continue; }
			$aArticleNumbers[] = $oLineItem->getReferencedId();
		}
		$aArticles = [];
		if ( $aArticleNumbers ) {
			/** @var ProductCollection $aArticles */
			$aArticles =  $this->productRepo->search(new Criteria($aArticleNumbers), $oSalesChannelContext)->getEntities();
		}
		if ( !$aArticles ) {
			return false;
		}

		//$oBasket = $event->getPage()->
		try {
			$oReference = $this->creditPlusHelper->getFinancingArticleReference($oSalesChannelContext,null,$oCart->getLineItems(),$oCart->getPrice()->getTotalPrice());
		} catch ( \Exception $oEx ) {
			$oReference = null;
		}
        $bSicoFinancingExcluded = false;
        if($oRequest->hasSession()){
            if($oRequest?->getSession()->has('bSicoFinancingExcluded')){
                $bSicoFinancingExcluded = $oRequest->getSession()?->get('bSicoFinancingExcluded',false);
            }
        }

		if ( $oReference == null || $bSicoFinancingExcluded ) {
			return false;
		}
		return true;
	}
}
