<?php declare(strict_types=1);

namespace SicoCreditPlus\Migration;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception as DBALException;
use Shopware\Core\Framework\Migration\MigrationStep;
use Shopware\Core\Framework\Uuid\Uuid;

class Migration1717228161PaymentInfoDropPrevention extends MigrationStep
{
    public function getCreationTimestamp(): int
    {
        return 1717728161;
    }

    /**
     * Adds fields to bind the CreditPlus payment data to the correct order version.
     * Scans CreditPlus logs to recreate deleted entries.
     * Adds field to product table.
     *
     * @param Connection $connection
     * @return void
     * @throws DBALException
     */
    public function update(Connection $connection): void
    {
        $sql = 'SHOW COLUMNS FROM sico_credit_plus_payment';
        $resultStatement = $connection->executeQuery($sql);
        $fields = $resultStatement->fetchAllAssociative();
        $bHasColumn = false;
        foreach ($fields as $field) {
            if ($field['Field'] === 'order_version_id') {
                $bHasColumn = true;
            }
        }
        // Add new field to store the version id
        try {
            if (!$bHasColumn) {
                $connection->executeStatement("
					ALTER TABLE sico_credit_plus_payment ADD COLUMN order_version_id BINARY(16) NULL DEFAULT version_id AFTER order_id;
				");
            }
        } catch ( \Exception $oException ) {
            try {
                if (!$bHasColumn) {
                    // MySQL 5.7 does not allow to add a default value of another column, create a trigger instead
                    $connection->executeStatement("
						ALTER TABLE sico_credit_plus_payment ADD COLUMN order_version_id BINARY(16) NULL DEFAULT NULL AFTER order_id;
					");
					$connection->executeStatement("
						CREATE TRIGGER sico_credit_plus_payment_before_insert BEFORE INSERT ON sico_credit_plus_payment FOR EACH ROW
						BEGIN
							IF NEW.order_version_id IS NULL THEN
								SET NEW.order_version_id = NEW.version_id;
							END IF;
						END;
					");
				}
            } catch ( \Exception $oInnerException ) {
                // Maybe the trigger failed to be created, this is as far as we can go.
                if ( $oInnerException instanceof DBALException ) {
                    // This is a Doctrine exception, rethrow it
                    throw $oInnerException;
                }
            }
            // If field already exists, ignore
        }
        $connection->executeStatement("
			UPDATE sico_credit_plus_payment LEFT JOIN `order` ON sico_credit_plus_payment.order_id = `order`.id AND sico_credit_plus_payment.version_id = `order`.version_id SET sico_credit_plus_payment.order_version_id = `order`.version_id WHERE NOT (sico_credit_plus_payment.order_id IS NULL);
		");
        // Remove orphans left over from previous copies before foreign key fails
        $connection->executeStatement("
			DELETE FROM sico_credit_plus_payment WHERE order_version_id IS NULL;
		");
        // Switch from single reference field to multi-reference field
        try {
            $connection->executeStatement("
				ALTER TABLE sico_credit_plus_payment DROP FOREIGN KEY `sico_credit_plus_payment_ibfk_1`;
			");
        } catch (\Exception $e) {
            // Ignore if foreign key does not exist
        }
        try {
            $connection->executeStatement("
				ALTER TABLE sico_credit_plus_payment DROP INDEX `sico_credit_plus_payment_ibfk_1`;
			");
        } catch (\Exception $e) {
            // Ignore if index does not exist
            // Might have left through the previous statement
        }
        // This is the new foreign key with version_id
        $connection->executeStatement("
			ALTER TABLE sico_credit_plus_payment ADD FOREIGN KEY `sico_credit_plus_payment_ibfk_1` (order_id, order_version_id) REFERENCES `order` (id, version_id) ON DELETE CASCADE;
		");
        // Funny field required by the Shopware API to address data on foreign tables
        $sql = 'SHOW COLUMNS FROM `order`';
        $resultStatement = $connection->executeQuery($sql);
        $fields = $resultStatement->fetchAllAssociative();
        $bHasColumn = false;
        foreach($fields as $field){
            if($field['Field'] === 'creditPlusPayment'){
                $bHasColumn = true;
            }
        }
        try {
            if(!$bHasColumn) {
                $connection->executeStatement("
					ALTER TABLE `order` ADD COLUMN creditPlusPayment BINARY(16) NULL DEFAULT id;
				");
            }
        } catch ( \Exception $oException ) {
            try {
                if (!$bHasColumn) {
                    // Same as above: MySQL 5.7 does not support default values from other fields
                    $connection->executeStatement("
						ALTER TABLE `order` ADD COLUMN creditPlusPayment BINARY(16) NULL DEFAULT NULL;
					");
					$connection->executeStatement("
						CREATE TRIGGER order_before_insert_creditplus BEFORE INSERT ON `order` FOR EACH ROW
						BEGIN
							IF NEW.creditPlusPayment IS NULL THEN
								SET NEW.creditPlusPayment = NEW.id;
							END IF;
						END;
					");
                }
            } catch ( \Exception $oInnerException ) {
                // Maybe the trigger failed to be created, this is as far as we can go.
                if ( $oInnerException instanceof DBALException ) {
                    throw $oInnerException;
                }
            }
            // Ignore if field already exists
        }
        // Fill the new field with data
        $connection->executeStatement("
			UPDATE `order` SET creditPlusPayment = id WHERE creditPlusPayment IS NULL;
		");
		/**
		 * Remaining part moved to LostPaymentDataRestorerHandler in Version 1.4.2
		 * @see \SicoCreditPlus\ScheduledTasks\LostPaymentDataRestorerHandler::handleRestore()
		 */
    }

    public function updateDestructive(Connection $connection): void
    {
        // No destructive updates exists
    }
}
