<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 29.12.2023
 * @time 16:10
 */

namespace SicoCreditPlus\ScheduledTasks;

use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Dbal\QueryBuilder;
use Shopware\Core\Content\ProductStream\Service\ProductStreamBuilder;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Dbal\CriteriaQueryBuilder;
use Shopware\Core\Framework\DataAbstractionLayer\EntityDefinition;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\SearchRequestException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Parser\QueryStringParser;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\Context\AbstractSalesChannelContextFactory;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupCollection;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupEntity;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;

#[AsMessageHandler(handles: GroupMapperTask::class)]
class GroupMapperHandler extends \Shopware\Core\Framework\MessageQueue\ScheduledTask\ScheduledTaskHandler
{
	protected ?EntityRepository $oSicoProductGroupRepository = null;
	protected ?EntityRepository $oSicoProductGroupProductRepository = null;
	/**
	 * @var SalesChannelRepository|null Some kind of "not Entity"-Repository is loaded here
	 */
	protected ?SalesChannelRepository $oProductRepository = null;
	protected ?EntityRepository $oSalesChannelRepository = null;
	protected ?AbstractSalesChannelContextFactory $oSalesChannelContextFactory = null;
	protected ?EntityDefinition $oProductDefinition = null;
	protected ?Connection $oConnection = null;
	protected ?CriteriaQueryBuilder $oCriteriaQueryBuilder = null;

	public function __construct(EntityRepository $scheduledTaskRepository, ?LoggerInterface $exceptionLogger = null, EntityRepository $oSicoProductGroupRepository = null, EntityRepository $oSicoProductGroupProductRepository = null, EntityRepository $oSalesChannelRepository = null, SalesChannelRepository $oProductRepository = null, EntityDefinition $oProductDefinition = null, AbstractSalesChannelContextFactory $oSalesChannelContextFactory = null, Connection $oConnection = null, CriteriaQueryBuilder $oCriteriaQueryBuilder = null)
	{
		parent::__construct($scheduledTaskRepository, $exceptionLogger);
		if ( !$this->oSicoProductGroupRepository ) {
			$this->oSicoProductGroupRepository = $oSicoProductGroupRepository;
		}
		if ( !$this->oSicoProductGroupProductRepository ) {
			$this->oSicoProductGroupProductRepository = $oSicoProductGroupProductRepository;
		}
		if ( !$this->oProductRepository ) {
			$this->oProductRepository = $oProductRepository;
		}
		if ( !$this->oSalesChannelRepository ) {
			$this->oSalesChannelRepository = $oSalesChannelRepository;
		}
		if ( !$this->oProductDefinition ) {
			$this->oProductDefinition = $oProductDefinition;
		}
		if ( !$this->oSalesChannelContextFactory ) {
			$this->oSalesChannelContextFactory = $oSalesChannelContextFactory;
		}
		if ( !$this->oConnection ) {
			$this->oConnection = $oConnection;
		}
		if ( !$this->oCriteriaQueryBuilder ) {
			$this->oCriteriaQueryBuilder = $oCriteriaQueryBuilder;
		}
	}

	/**
	 * @inheritDoc
	 */
	public static function getHandledMessages(): iterable
	{
		return [ GroupMapperTask::class ];
	}

	public function run(): void
	{
		$oContext = Context::createDefaultContext();
		$oProductGroupCriteria = new Criteria();
		$oProductGroupCriteria->addAssociation('product');
		$oResult = $this->oSicoProductGroupRepository->search($oProductGroupCriteria, $oContext);

		$aSalesChannelContexts = [];
		$oSalesChannelCriteria = new Criteria();
		$oSalesChannelCriteria->addAssociation('type');
		/** @var SalesChannelEntity[] $aSalesChannels */
		$aSalesChannels = $this->oSalesChannelRepository->search($oSalesChannelCriteria, $oContext)->getElements();
		foreach ( $aSalesChannels as $oSalesChannel ) {
			$oType = $oSalesChannel->getType();
			// Safety feature: If type is null, the next line would fail
			$bFeed = $oType === null;
			// Every sales channel with a rocket is assumed to be a feed
			$bFeed = $bFeed || $oSalesChannel->getType()->getIconName() == 'regular-rocket';
			if ( $bFeed ) {
				continue;
			}
			$sSalesChannelID = $oSalesChannel->getId();
			$oSalesChannelContext = $this->oSalesChannelContextFactory->create(Uuid::randomHex(), $sSalesChannelID);
			$aSalesChannelContexts[$sSalesChannelID] = $oSalesChannelContext;
		}

		/** @var SicoProductgroupCollection|SicoProductgroupEntity[] $aProductGroups */
		$aProductGroups = $oResult->getElements();
		foreach ( $aProductGroups as $oProductGroup ) {

			/** @see ProductStreamBuilder::buildFilters() */
			$oDynamicProductGroup = $oProductGroup->getDynamicProductGroup();
			if ( !$oDynamicProductGroup ) {
				// Don't touch specifically assigned groups
				continue;
			}
			$aRawFilters = $oDynamicProductGroup->getApiFilter();
			$aFilters = [];
			$oException = new SearchRequestException();
			foreach ($aRawFilters as $aRawFilter) {
				$aFilters[] = QueryStringParser::fromArray($this->oProductDefinition, $aRawFilter, $oException, '');
			}

			$oCriteria = new Criteria();
			$oCriteria->addFilter(...$aFilters);

			$this->handleBigAssign($oProductGroup, $oCriteria, $aSalesChannelContexts);
		}
	}

	/**
	 * This is done by raw SQL because having correct results for 61.000 products in 27 seconds
	 * beats having an out of memory exception in 2 seconds.
	 *
	 * @param SicoProductgroupEntity $oProductGroup Productgroup being reassigned articles
	 * @param Criteria $oCriteria Criteria filled by dynamic product group (product stream)
	 * @param SalesChannelContext[] $aSalesChannelContexts SalesChannelContext objects for each sales channel
	 * @return void
	 */
	private function handleBigAssign(SicoProductgroupEntity $oProductGroup, Criteria $oCriteria, array $aSalesChannelContexts): void
	{
		// Build SELECT queries with Doctrine query builder
		$aQueries = [];
		$aParams = [];
		foreach ( $aSalesChannelContexts as $oSalesChannelContext ) {
			$oQuery = new QueryBuilder($this->oConnection);
			$oQueryBuilder = $this->oCriteriaQueryBuilder->build($oQuery, $this->oProductDefinition, $oCriteria, $oSalesChannelContext->getContext());
			$oQueryBuilder->select('product.id');
			$sSQL = $oQueryBuilder->getSQL();
			$aCurrentParams = $oQueryBuilder->getParameters();
			if ( $sSQL ) {
				$aQueries[] = $sSQL;
				$aParams[] = $aCurrentParams;
			}
		}
		$sQuery = implode(' UNION ', $aQueries);

		// Drop temporary table beforehand
		$sDropQuery = 'DROP TEMPORARY TABLE IF EXISTS sico_tmp_group_map;';
		$this->oConnection->executeStatement($sDropQuery);

		// Create table based on SELECT queries
		$sCreateQuery = 'CREATE TEMPORARY TABLE sico_tmp_group_map AS '.$sQuery.';';
		$aCreateParams = [];
		$aParamTypes = [];
		foreach ( $aParams as $aParam ) {
			foreach ( $aParam as $sKey => $oValue ) {
				$aCreateParams[$sKey] = $oValue;
				if ( is_array($oValue) ) {
					$aParamTypes[$sKey] = ArrayParameterType::STRING;
				}
			}
		}
		$this->oConnection->executeStatement($sCreateQuery, $aCreateParams, $aParamTypes);

		// Add index for column to speed up DELETE and INSERT JOIN
		$sForeignKeyQuery = 'ALTER TABLE sico_tmp_group_map ADD PRIMARY KEY (id);';
		$this->oConnection->executeStatement($sForeignKeyQuery);

		// Delete unmapped items first
		$sDeleteQuery = 'DELETE pga FROM sico_productgroup_s_articles pga LEFT JOIN sico_tmp_group_map tgm ON pga.product_id = tgm.id WHERE pga.productgroup_id = :prodgroup_id AND tgm.id IS NULL;';
		$aDeleteParams = ['prodgroup_id' => hex2bin($oProductGroup->getId())];
		$this->oConnection->executeStatement($sDeleteQuery, $aDeleteParams);

		// Insert new mapped products
		$sInsertQuery = 'INSERT INTO sico_productgroup_s_articles (productgroup_id, product_id, created_at, updated_at, product_version_id) SELECT :prodgroup_id, tgm.id, :created_at, :updated_at, :version_id FROM sico_tmp_group_map tgm LEFT JOIN sico_productgroup_s_articles pga ON tgm.id = pga.product_id AND pga.productgroup_id = :prodgroup_id WHERE pga.product_id IS NULL;';
		$aInsertParams = [
			'prodgroup_id' => hex2bin($oProductGroup->getId()),
			'created_at' => date('Y-m-d H:i:s'),
			'updated_at' => date('Y-m-d H:i:s'),
			'version_id' => $aCreateParams['version'],
		];

		$this->oConnection->executeStatement($sInsertQuery, $aInsertParams);

		$this->oConnection->executeStatement($sDropQuery);
	}
}
