<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 29.12.2023
 * @time 16:10
 */

namespace SicoCreditPlus\ScheduledTasks;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Uuid\Uuid;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;

#[AsMessageHandler(handles: LostPaymentDataRestorerTask::class)]
class LostPaymentDataRestorerHandler extends \Shopware\Core\Framework\MessageQueue\ScheduledTask\ScheduledTaskHandler
{
	protected ?SicoCreditPlusLogger $oLogger = null;
	protected ?Connection $oConnection = null;

	public function __construct(EntityRepository $scheduledTaskRepository, ?LoggerInterface $exceptionLogger = null, SicoCreditPlusLogger $oLogger = null, Connection $oConnection = null)
	{
		parent::__construct($scheduledTaskRepository, $exceptionLogger);
		if ( !$this->oLogger ) {
			$this->oLogger = $oLogger;
		}
		if ( !$this->oConnection ) {
			$this->oConnection = $oConnection;
		}
	}

	public function run(): void
	{
		$this->handleRestore();
	}

	private function handleRestore(): void
	{
		if ( !$this->oConnection ) {
			$this->oLogger->error('No database connection available for restoring lost payment data', []);
			return;
		}
		// Create new half-dummy entries for those we lost with pre-order (first select) and post-order (second select)
		$oRes = $this->oConnection->executeQuery("
SELECT o.id, o.version_id, o.order_number, o.custom_fields, o.sales_channel_id
FROM `order` o 
    LEFT JOIN sico_credit_plus_payment scpp ON o.id = scpp.order_id 
WHERE o.custom_fields LIKE '%cpDealerOrderNumber%' AND scpp.id IS NULL
UNION
SELECT o.id, o.version_id, o.order_number, o.custom_fields, o.sales_channel_id
FROM `order` o
	 INNER JOIN order_transaction ot ON o.id = ot.order_id AND o.version_id = ot.order_version_id
	 INNER JOIN payment_method pm ON ot.payment_method_id = pm.id
	 LEFT JOIN sico_credit_plus_payment scpp ON o.id = scpp.order_id
WHERE
    ( (o.custom_fields IS NULL) OR (NOT (o.custom_fields LIKE '%cpDealerOrderNumber%')) ) AND
    pm.handler_identifier = :cpPaymentIdentifier AND
    scpp.id IS NULL
GROUP BY o.id, o.version_id, o.order_number, o.custom_fields
LIMIT 200
;", ['cpPaymentIdentifier' => 'SicoCreditPlus\\Service\\CreditPlusPayment']);
		if ($oRes->rowCount() > 0) {
			$sInsertStatement = "INSERT INTO sico_credit_plus_payment (id, version_id, order_id, url, gen_timestamp, orderNumber, created_at, updated_at, token, order_finish, back_url, status_change_url) VALUES (:id, :versionId, :orderId, :url, :genTimestamp, :orderNumber, NOW(), NOW(), :token, :orderFinish, :backUrl, :statusChangeUrl);";
			$sUpdateStatement = "UPDATE `order` SET custom_fields = :customFields WHERE id = UNHEX(:orderId) AND version_id = UNHEX(:orderVersionId);";
			while ($aRow = $oRes->fetchAssociative()) {
				$aCustomFields = [];
				if ($aRow['custom_fields']) {
					$aCustomFields = json_decode($aRow['custom_fields'], true);
				}
				if (!isset($aCustomFields['cpDealerOrderNumber'])) {
					// Fake post-order as pre-order
					$aCustomFields['cpDealerOrderNumber'] = (string)$aRow['order_number'];
				}
				if (isset($aCustomFields['cpDealerOrderNumber']) && ($sDealerOrderNumber = $aCustomFields['cpDealerOrderNumber'])) {
					// Search for URL data in logs, if available
					$sURL = $sOrderFinish = $sBackURL = $sStatusChangeURL = '';
					$sBaseDir = dirname(__FILE__, 6);
					// Find log entry with statusChangedNotificationUrl from creation request (sending side) as var_export
					$sLogFileContent = exec("grep -rC 0 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \"'statusChangedNotificationUrl'\"");
					if ( !trim($sLogFileContent) ) {
						$sDealerOrderNumberFromLog = exec("grep -rC 3 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \"getContractsResponse\" | grep \"dealerOrderNumber\"");
						if (preg_match('#<dealerOrderNumber>([^<]+)</dealerOrderNumber>#', $sDealerOrderNumberFromLog, $aMatchesDealerOrderNumber)) {
							if ( $aMatchesDealerOrderNumber ) {
								$sDealerOrderNumber = $aMatchesDealerOrderNumber[1];
								$aCustomFields['cpDealerOrderNumber'] = $sDealerOrderNumber;
								$aRow['custom_fields'] = json_encode($aCustomFields);
								$this->oConnection->executeStatement($sUpdateStatement, ['customFields' => $aRow['custom_fields'], 'orderId' => bin2hex($aRow['id']), 'orderVersionId' => bin2hex($aRow['version_id'])]);
								$this->oLogger->info('Updated order ' . $aRow['order_number'] . ' with recovered dealer order number ' . $sDealerOrderNumber, [], bin2hex($aRow['sales_channel_id']));
								$sLogFileContent = exec("grep -rC 0 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \"'statusChangedNotificationUrl'\"");
							}
						}
					}
					$aMatches = [];
					if (preg_match("#'statusChangedNotificationUrl' => '([^']+)'#", $sLogFileContent, $aMatches)) {
						// The database field is named differently
						$aStatusChangeURL = explode('?', $aMatches[1]);
						$sStatusChangeURL = $aStatusChangeURL[0] ?? '';
						if ($sStatusChangeURL) {
							$sStatusChangeURL .= '?';
						}
						$sBackURL = (string)$aMatches[1];
					}
					// Find log entry with customerUrl from creation response (receiving side) as XML
					$sLogFileContent = exec("grep -rC 0 '$sDealerOrderNumber' $sBaseDir/var/log/ | grep \":createCreditOfferResponse\" | grep -v \"printDebugCode\"");
					$aMatches = [];
					if (preg_match('#<customerUrl>([^<]+)</customerUrl>#', $sLogFileContent, $aMatches)) {
						$sURL = html_entity_decode($aMatches[1]);
						$sOrderFinish = $sURL;
					}
					// Create a new payment entry
					$aNewPaymentData = [
						'id' => Uuid::randomBytes(),
						'versionId' => $aRow['version_id'],
						'orderId' => $aRow['id'],
						'orderVersionId' => $aRow['version_id'],
						'url' => $sURL,
						'genTimestamp' => time(),
						'orderNumber' => $aCustomFields['cpDealerOrderNumber'],
						'token' => 'dummyTokenRebuilt' . substr(md5('SicoCreditPlus' . microtime(true)), 4, 6),
						'orderFinish' => $sOrderFinish,
						'backUrl' => $sBackURL,
						'statusChangeUrl' => $sStatusChangeURL
					];
					$this->oConnection->executeStatement($sInsertStatement, $aNewPaymentData);
					$this->oLogger->info('Restored lost payment data for order ' . $aRow['order_number'] . ' with dealer order number ' . $aCustomFields['cpDealerOrderNumber'], [], bin2hex($aRow['sales_channel_id']));
				}
			}
		}
	}
}
