<?php
/**
 * Created by PhpStorm.
 * @author mihovil.bubnjar
 * @date 29.12.2023
 * @time 16:10
 */

namespace SicoCreditPlus\ScheduledTasks;

use DateTime;
use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\Currency\CurrencyEntity;
use Shopware\Core\System\SalesChannel\Context\AbstractSalesChannelContextFactory;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use SicoCreditPlus\Components\SicoCreditPlusLogger;
use SicoCreditPlus\Core\Content\SicoCreditPlusCalculatedRate\SicoCreditPlusCalculatedRateEntity;
use SicoCreditPlus\Lib\CreditPlusObjects\WebshopRateTableMonthRow;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;

#[AsMessageHandler(handles: RateCalculatorTask::class)]
class RateCalculatorHandler extends \Shopware\Core\Framework\MessageQueue\ScheduledTask\ScheduledTaskHandler
{
	/**
	 * @var SalesChannelRepository|null Some kind of "not Entity"-Repository is loaded here
	 */
	protected ?SalesChannelRepository $oProductRepository = null;
	protected ?EntityRepository $oSalesChannelRepository = null;
	protected ?AbstractSalesChannelContextFactory $oSalesChannelContextFactory = null;
	protected ?SicoCreditPlusHelper $oCreditPlusHelper = null;
	protected ?SicoCreditPlusLogger $oCreditPlusLogger = null;
	protected ?EntityRepository $oCalculatedRateRepository = null;
	protected ?EntityRepository $oFinancingGroupsRepository = null;
	protected ?Connection $oConnection = null;

	public function __construct(
		EntityRepository $scheduledTaskRepository,
		?LoggerInterface $exceptionLogger = null,
		EntityRepository $oSalesChannelRepository = null,
		SalesChannelRepository $oProductRepository = null,
		AbstractSalesChannelContextFactory $oSalesChannelContextFactory = null,
		SicoCreditPlusLogger $oCreditPlusLogger = null,
		SicoCreditPlusHelper $oCreditPlusHelper = null,
		EntityRepository $oCalculatedRateRepository = null,
		EntityRepository $oFinancingGroupsRepository = null,
		Connection $oConnection = null
	)
	{
		parent::__construct($scheduledTaskRepository, $exceptionLogger);
		if ( !$this->oProductRepository ) {
			$this->oProductRepository = $oProductRepository;
		}
		if ( !$this->oSalesChannelRepository ) {
			$this->oSalesChannelRepository = $oSalesChannelRepository;
		}
		if ( !$this->oSalesChannelContextFactory ) {
			$this->oSalesChannelContextFactory = $oSalesChannelContextFactory;
		}
		if ( !$this->oCreditPlusLogger ) {
			$this->oCreditPlusLogger = $oCreditPlusLogger;
		}
		if ( !$this->oCreditPlusHelper ) {
			$oCreditPlusHelper->setCacheFiles(true);
			$this->oCreditPlusHelper = $oCreditPlusHelper;
		}
		if ( !$this->oCalculatedRateRepository ) {
			$this->oCalculatedRateRepository = $oCalculatedRateRepository;
		}
		if ( !$this->oFinancingGroupsRepository ) {
			$this->oFinancingGroupsRepository = $oFinancingGroupsRepository;
		}
		if ( !$this->oConnection ) {
			$this->oConnection = $oConnection;
		}
	}

	/**
	 * @inheritDoc
	 */
	public static function getHandledMessages(): iterable
	{
		return [ RateCalculatorTask::class ];
	}

	public function run(): void
	{
		$oContext = Context::createDefaultContext();

		$aSalesChannelContexts = [];
		$oSalesChannelCriteria = new Criteria();
		$oSalesChannelCriteria->addAssociation('type');
		/** @var SalesChannelEntity[] $aSalesChannels */
		$aSalesChannels = $this->oSalesChannelRepository->search($oSalesChannelCriteria, $oContext)->getElements();
		foreach ( $aSalesChannels as $oSalesChannel ) {
			$oType = $oSalesChannel->getType();
			// Safety feature: If type is null, the next line would fail
			$bFeed = $oType === null;
			// Every sales channel with a rocket is assumed to be a feed
			$bFeed = $bFeed || $oType->getIconName() == 'regular-rocket';
			if ( $bFeed ) {
				continue;
			}
			$sSalesChannelID = $oSalesChannel->getId();
			$oSalesChannelContext = $this->oSalesChannelContextFactory->create(Uuid::randomHex(), $sSalesChannelID);

			// Find out which sales channels are linked to a financing group
			$oFinancingGroupsCriteria = new Criteria();
			$oFinancingGroupsCriteria->addAssociation('prodgroupSalesChannel');
			$oFinancingGroupsCriteria->addAssociation('prodgroupSalesChannel.salesChannel');
			$oFinancingGroupsCriteria->addFilter(new EqualsFilter('prodgroupSalesChannel.salesChannelId', $sSalesChannelID));
			$aFinancingGroups = $this->oFinancingGroupsRepository->search($oFinancingGroupsCriteria, $oSalesChannelContext->getContext());
			if  ( !$aFinancingGroups || !$aFinancingGroups->count() ) {
				// Don't calculate rates for this sales channel
				continue;
			}

			$this->oCreditPlusHelper->resetWebshopAPI();
			$oWSApi = $this->oCreditPlusHelper->getWebshopAPI($oSalesChannelContext->getSalesChannelId());
			if ( $oWSApi->isApiCallable() ) {
				// Only calculate currently callable contexts
				$aSalesChannelContexts[$sSalesChannelID] = $oSalesChannelContext;
			} else {
				$this->oCreditPlusLogger->error('API is not callable for sales channel "'.$oSalesChannelContext->getSalesChannel()->getName().'". Skipping this sales channel.', [], $sSalesChannelID);
			}
		}
		// If a product has a sicoCreditPlusCalculatedRates value which it should not have (e.g. inherited from parent product), reset it to the expected value
		$this->oConnection->executeStatement("UPDATE product SET sicoCreditPlusCalculatedRates = product.id WHERE (NOT (sicoCreditPlusCalculatedRates = product.id))");
		foreach ( $aSalesChannelContexts as $oSalesChannelContext ) {
			$oQuery = $this->oConnection->executeQuery("
				SELECT product.id
				FROM product
				INNER JOIN product_visibility pv ON product.id = pv.product_id AND product.version_id = pv.product_version_id AND pv.sales_channel_id = :salesChannelId
				LEFT JOIN sico_credit_plus_calculated_rate scpcr ON scpcr.product_id = product.id AND scpcr.product_version_id = product.version_id AND scpcr.sico_sales_channel_id = :salesChannelId
				WHERE
				    (
				        	(scpcr.sico_calculation_finished = 0)
				   		OR
				    		(scpcr.id IS NULL)
					)
					AND
						(pv.visibility >= 10)
					AND
						(product.active = 1)
				GROUP BY product.id
				UNION
				SELECT product.id
				FROM product
				INNER JOIN product_visibility pv ON product.id = pv.product_id AND product.version_id = pv.product_version_id AND pv.sales_channel_id = :salesChannelId
				INNER JOIN sico_credit_plus_calculated_rate scpcr ON scpcr.product_id = product.id AND scpcr.product_version_id = product.version_id
				WHERE (pv.visibility >= 10) AND (product.active = 1)
				GROUP BY product.id
				HAVING COUNT(scpcr.sico_sales_channel_id = :salesChannelId) = 0
			", ['salesChannelId' => hex2bin($oSalesChannelContext->getSalesChannelId())]);
			$aProductIds = [];
			if ( $oQuery && $oQuery->rowCount() ) {
				$iProductCount = 0;
				while ( $iProductCount < 100 ) {
					$aRow = $oQuery->fetchAssociative();
					if ( !$aRow ) {
						break;
					}
					$aProductIds[] = bin2hex($aRow['id']);
					$iProductCount++;
				}
			}
			if ( !$aProductIds ) {
				// Found no products to update, continue with next sales channel
				continue;
			}
			$oProductCriteria = new Criteria($aProductIds);
			$oProductCriteria->addAssociation('tax');
			$oProductCriteria->addAssociation('sicoCreditPlusCalculatedRates');
			$oProductCriteria->addAssociation('sicoCreditPlusCalculatedRates.sicoSalesChannel');
			$oProductCriteria->addAssociation('children');
			$oProductCriteria->addAssociation('children.tax');
			$oProductCriteria->addAssociation('children.sicoCreditPlusCalculatedRates');
			$oProductCriteria->addAssociation('children.sicoCreditPlusCalculatedRates.sicoSalesChannel');

			$this->oCreditPlusHelper->setSalesChannelContext($oSalesChannelContext);
			$oProductsToIndex = $this->oProductRepository->search($oProductCriteria, $oSalesChannelContext);
			/** @var ProductEntity[] $aProductsToIndex */
			$aProductsToIndex = $oProductsToIndex->getElements();
			//$this->oCreditPlusLogger->info('RateCalculatorHandler: Starting rate calculation for sales channel.', ['SalesChannel.id' => $oSalesChannelContext->getSalesChannelId(), 'ProductsToIndex.count' => count($aProductsToIndex)], $oSalesChannelContext->getSalesChannelId());
			$oCurrency = $oSalesChannelContext->getCurrency();
			$aVisitedProducts = [];
			foreach ( $aProductsToIndex as $oProductToIndex ) {
				if ( in_array($oProductToIndex->getId(), $aVisitedProducts) ) {
					continue;
				}
				$aVisitedProducts[] = $oProductToIndex->getId();
				$this->calculateRatesForProduct($oProductToIndex, $oSalesChannelContext, $oCurrency);
				// Find non-mapped variants as well
				if ( ($aChildren = $oProductToIndex->getChildren()) !== null ) {
					foreach ( $aChildren as $oChildProduct ) {
						$this->calculateRatesForProduct($oChildProduct, $oSalesChannelContext, $oCurrency);
					}
				}
			}
		}
	}

	/**
	 * @param ProductEntity $oProductToIndex
	 * @param SalesChannelContext $oSalesChannelContext
	 * @param CurrencyEntity $oCurrency
	 * @return void
	 */
	protected function calculateRatesForProduct(ProductEntity $oProductToIndex, SalesChannelContext $oSalesChannelContext, CurrencyEntity $oCurrency): void
	{
		$oWSApi = $this->oCreditPlusHelper->getWebshopAPI($oSalesChannelContext->getSalesChannelId());
		$oWSCurrency = $this->oCreditPlusHelper->getWebshopCurrency($oCurrency);
		$oCalculatedRate = null;
		/** @var SicoCreditPlusCalculatedRateEntity[] $aCalculatedRates */
		if ( $aCalculatedRates = $oProductToIndex->get('sicoCreditPlusCalculatedRates') ) {
			foreach ( $aCalculatedRates as $oFoundCalculatedRate ) {
				if ( $oFoundCalculatedRate->getSicoSalesChannelId() === $oSalesChannelContext->getSalesChannelId() ) {
					$oCalculatedRate = $oFoundCalculatedRate;
					$this->oCreditPlusLogger->info('Found a calculated rate and updating it.', ['SicoCalculatedRate.id' => $oCalculatedRate->getId()], $oSalesChannelContext->getSalesChannelId());
				}
			}
		}
		if ( !$oCalculatedRate ) {
			$oCalculatedRate = new SicoCreditPlusCalculatedRateEntity();
			$oCalculatedRate->setId(Uuid::randomHex());
			$oCalculatedRate->setProductId($oProductToIndex->getId());
			$oCalculatedRate->setSicoProduct($oProductToIndex);
			$oCalculatedRate->setSicoSalesChannelId($oSalesChannelContext->getSalesChannelId());
			$oCalculatedRate->setSicoSalesChannel($oSalesChannelContext->getSalesChannel());
			$this->oCreditPlusLogger->info('Creating a new calculated rate.', ['SicoCalculatedRate.id' => null, 'Product.id' => $oProductToIndex->getId()], $oSalesChannelContext->getSalesChannelId());
		}
		$aFinancingMonths = $this->oCreditPlusHelper->getFinancingMonths($oSalesChannelContext, $oCurrency, $oProductToIndex);
		/** @var WebshopRateTableMonthRow $oShortestMonths */
		$oShortestMonths = null;
		/** @var WebshopRateTableMonthRow $oLongestMonths */
		$oLongestMonths = null;
		/** @var WebshopRateTableMonthRow $oAbsoluteCheapest */
		$oAbsoluteCheapest = null;
		foreach ( $aFinancingMonths as $oMonthRow ) {
			if ( !$oShortestMonths || ($oShortestMonths->months > $oMonthRow->months) ) {
				$oShortestMonths = $oMonthRow;
			}
			if ( !$oLongestMonths || ($oLongestMonths->months < $oMonthRow->months) ) {
				$oLongestMonths = $oMonthRow;
			}
			if ( !$oAbsoluteCheapest || ($oWSApi->retrieveFloatPriceFromFormattedPrice($oAbsoluteCheapest->monthlyRate, $oWSCurrency) > $oWSApi->retrieveFloatPriceFromFormattedPrice($oMonthRow->monthlyRate, $oWSCurrency)) ) {
				$oAbsoluteCheapest = $oMonthRow;
			}
		}
		// Comes one, come all - like to a circus :)
		if ( $oShortestMonths ) {
			$oCalculatedRate->setSicoMinMonths((int)$oShortestMonths->months);
			$oCalculatedRate->setSicoMinMonthsRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oShortestMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoMinMonthsInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oShortestMonths->interestRate, $oWSCurrency));
			$oCalculatedRate->setSicoMaxMonths((int)$oLongestMonths->months);
			$oCalculatedRate->setSicoMaxMonthsRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oLongestMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoMaxMonthsInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oLongestMonths->interestRate, $oWSCurrency));
			$oCalculatedRate->setSicoAbsoluteMinRateMonths((int)$oAbsoluteCheapest->months);
			$oCalculatedRate->setSicoAbsoluteMinRateRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oAbsoluteCheapest->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoAbsoluteMinRateInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oAbsoluteCheapest->interestRate, $oWSCurrency));
		} else {
			$oCalculatedRate->setSicoMinMonths(-1);
			$oCalculatedRate->setSicoMinMonthsRate(-1);
			$oCalculatedRate->setSicoMinMonthsInterestRate(-1);
			$oCalculatedRate->setSicoMaxMonths(-1);
			$oCalculatedRate->setSicoMaxMonthsRate(-1);
			$oCalculatedRate->setSicoMaxMonthsInterestRate(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateMonths(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateRate(-1);
			$oCalculatedRate->setSicoAbsoluteMinRateInterestRate(-1);
		}

		// With the realistic values above, some smaller items may not be listed as "can be financed" in a filter.
		// So these values are there to make them seem like they can be financed. They still have to cost at least 6 cents.
		$aFictionalFinancingMonths = $this->oCreditPlusHelper->getFinancingMonths($oSalesChannelContext, $oCurrency, $oProductToIndex, 0.01);
		/** @var WebshopRateTableMonthRow $oFictionalMonths */
		$oFictionalMonths = null;
		foreach ( $aFictionalFinancingMonths as $oMonthRow ) {
			if ( ($oFictionalMonths === null) || ($oWSApi->retrieveFloatPriceFromFormattedPrice($oFictionalMonths->monthlyRate, $oWSCurrency) > $oWSApi->retrieveFloatPriceFromFormattedPrice($oMonthRow->monthlyRate, $oWSCurrency)) ) {
				$oFictionalMonths = $oMonthRow;
			}
		}
		if ( $oFictionalMonths ) {
			$oCalculatedRate->setSicoFictionalMinRateMonths((int)$oFictionalMonths->months);
			$oCalculatedRate->setSicoFictionalMinRateRate($oWSApi->retrieveFloatPriceFromFormattedPrice($oFictionalMonths->monthlyRate, $oWSCurrency));
			$oCalculatedRate->setSicoFictionalMinRateInterestRate($oWSApi->retrieveFloatFromFormattedInterestRate($oFictionalMonths->interestRate, $oWSCurrency));
		} else {
			$oCalculatedRate->setSicoFictionalMinRateMonths(-1);
			$oCalculatedRate->setSicoFictionalMinRateRate(-1);
			$oCalculatedRate->setSicoFictionalMinRateInterestRate(-1);
		}
		$oCalculatedRate->setSicoCurrency($oCurrency);
		$oCalculatedRate->setSicoCurrencyId($oCurrency->getId());
		$oCalculatedRate->setSicoCalculationFinished();
		$aUpsertData = [
			'productId' => $oCalculatedRate->getProductId(),
			'productVersionId' => $oProductToIndex->getVersionId(),
			'sicoSalesChannelId' => $oCalculatedRate->getSicoSalesChannelId(),
			'sicoMinMonths' => $oCalculatedRate->getSicoMinMonths(),
			'sicoMinMonthsInterestRate' => $oCalculatedRate->getSicoMinMonthsInterestRate(),
			'sicoMinMonthsRate' => $oCalculatedRate->getSicoMinMonthsRate(),
			'sicoMaxMonths' => $oCalculatedRate->getSicoMaxMonths(),
			'sicoMaxMonthsInterestRate' => $oCalculatedRate->getSicoMaxMonthsInterestRate(),
			'sicoMaxMonthsRate' => $oCalculatedRate->getSicoMaxMonthsRate(),
			'sicoAbsoluteMinRateMonths' => $oCalculatedRate->getSicoAbsoluteMinRateMonths(),
			'sicoAbsoluteMinRateInterestRate' => $oCalculatedRate->getSicoAbsoluteMinRateInterestRate(),
			'sicoAbsoluteMinRateRate' => $oCalculatedRate->getSicoAbsoluteMinRateRate(),
			'sicoFictionalMinRateMonths' => $oCalculatedRate->getSicoFictionalMinRateMonths(),
			'sicoFictionalMinRateInterestRate' => $oCalculatedRate->getSicoFictionalMinRateInterestRate(),
			'sicoFictionalMinRateRate' => $oCalculatedRate->getSicoFictionalMinRateRate(),
			'sicoCurrencyId' => $oCalculatedRate->getSicoCurrencyId(),
			'sicoCalculationFinished' => $oCalculatedRate->getSicoCalculationFinished(),
			'created_at' => new DateTime(),
			'updated_at' => new DateTime()
		];
		if ( $oCalculatedRate->getId() ) {
			$aUpsertData['id'] = $oCalculatedRate->getId();
		}
		$this->oCalculatedRateRepository->upsert([
			$aUpsertData
		], $oSalesChannelContext->getContext());
	}
}
