<?php declare(strict_types=1);

namespace SicoCreditPlus\Service;

use DateTime;
use Shopware\Core\Content\Media\Aggregate\MediaFolder\MediaFolderCollection;
use Shopware\Core\Content\Media\Aggregate\MediaFolder\MediaFolderDefinition;
use Shopware\Core\Content\Media\Core\Application\AbstractMediaUrlGenerator;
use Shopware\Core\Content\Media\File\FileSaver;
use Shopware\Core\Content\Media\File\MediaFile;
use Shopware\Core\Content\Media\MediaCollection;
use Shopware\Core\Content\Media\MediaDefinition;
use Shopware\Core\Content\Media\MediaEntity;
use Shopware\Core\Content\Media\Thumbnail\ThumbnailService;
use Shopware\Core\Framework\Api\Context\AdminApiSource;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionCollection;
use SicoCreditPlus\Core\Content\SicoOfferedOption\SicoOfferedOptionDefinition;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupCollection;
use SicoCreditPlus\Core\Content\SicoProductgroup\SicoProductgroupDefinition;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;
use Symfony\Component\Filesystem\Filesystem;
use function pathinfo;

class PluginActivateService
{
    protected ThumbnailService $oThumbnailService;
    protected ContainerInterface $container;
    protected AbstractMediaUrlGenerator $oUrlGenerator6;
    protected FileSaver $fileSaver;

    public function __construct(ThumbnailService          $oThumbnailService,
                                ContainerInterface        $container,
                                AbstractMediaUrlGenerator $oUrlGenerator6,
                                FileSaver $fileSaver)
    {
        $this->oThumbnailService = $oThumbnailService;
        $this->container = $container;
        $this->oUrlGenerator6 = $oUrlGenerator6;
        $this->fileSaver = $fileSaver;
    }

    public function pluginActivate(Context $context): void
    {
        $this->setPaymentMethodIsActive($context);
        $this->addDefaultDataToDatabase($context);

        try {
            /** @var \Shopware\Core\Framework\DataAbstractionLayer\Indexing\EntityIndexerRegistry $registry
             * @noinspection PhpFullyQualifiedNameUsageInspection
             */
            $registry = $this->container->get('Shopware\Core\Framework\DataAbstractionLayer\Indexing\EntityIndexerRegistry');
            $registry->sendIndexingMessage(['product.indexer']);
        } catch (ServiceNotFoundException $oEx) {
            // Do nothing, this is probably already handled by the Migration script
        }
    }

    private function setPaymentMethodIsActive(Context $context): void
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');
        $paymentMethodId = $this->getPaymentMethodId($context);
        // Payment does not even exist, so nothing to (de-)activate here
        if (!$paymentMethodId) {
            return;
        }
            $sMediaId = $this->addIconToPaymentMethod($context);


        $paymentMethod = [
            'id' => $paymentMethodId,
            'active' => true
        ];

        if (isset($sMediaId)) {
            $paymentMethod['mediaId'] = $sMediaId;
        }

        $paymentRepository->update([$paymentMethod], $context);
    }

    /**
     * @throws \Exception
     */
    private function addIconToPaymentMethod(Context $context) {

        /** @var EntityRepository $mediaRepo */
        $mediaRepo = $this->container->get(MediaDefinition::ENTITY_NAME.'.repository');

        $oCriteria = new Criteria();
        $oCriteria->addFilter(new EqualsFilter('fileName','creditplus_logo_small'));
        $oCriteria->addAssociations([
            'mediaFolder',
            'mediaFolder.configuration',
            'mediaFolder.configuration.mediaThumbnailSizes'
        ]);

        /** @var MediaEntity $mediaEntity */
        $mediaEntity = $mediaRepo->search($oCriteria, $context)->first();
        $projectDir = $this->container->getParameter('kernel.project_dir').'/';
        $thumbnailFile = null;
        if(file_exists($projectDir.'custom/static-plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png')){
            $thumbnailFile = $projectDir.'custom/static-plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png';
        }
        if(file_exists($projectDir.'custom/plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png')){
            $thumbnailFile = $projectDir.'custom/plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png';
        }
        if($thumbnailFile) {
            $oMedia = new MediaFile(
                $thumbnailFile,
                'image/png',
                'png',
                filesize($thumbnailFile)
            );//custom/plugins/SicoCreditPlus/src/Resources/public/img/creditplus_logo_small.png
        }
        else{
            throw new \Exception('creditplus_logo_small.png in custom/plugins/SicoCreditPlus/src/Resources/public/img/ or in custom/static-plugins/SicoCreditPlus/src/Resources/public/img/ not found');
        }
        if($mediaEntity == null){
            //$sMediaEntityId = $this->oMediaService->saveMediaFile($oMedia,'creditplus_logo_small',$context,'theme');
            $newMediaId = Uuid::randomHex();
            $sFolderId = $this->getMediaDefaultFolderId('theme',$context);
            $sUserId = null;
            if ( ($contextSource = $context->getSource()) && ($contextSource instanceof AdminApiSource) ) {
                $sUserId = $contextSource->getUserId();
            }
            $data =	[
                'id' => $newMediaId,
                'private' => false,
                'fileName' => 'creditplus_logo_small',
                'fileExtension' => $oMedia->getFileExtension(),
                'mimeType' => $oMedia->getMimeType(),
                'fileSize' => $oMedia->getFileSize(),
                'uploadedAt' => new DateTime(),
                'mediaFolderId' => $sFolderId,
                'userId' => $sUserId,
            ];
            $context->scope(Context::SYSTEM_SCOPE, function (Context $context) use ($mediaRepo,$data): void {
                $mediaRepo->create([$data], $context);
            });
            $oCriteria = new Criteria([$newMediaId]);
            $oCriteria->addAssociations([
                'mediaFolder',
                'mediaFolder.configuration',
                'mediaFolder.configuration.mediaThumbnailSizes'
            ]);
            $oMediaEntity = $mediaRepo->search($oCriteria, $context)->get($newMediaId);
        } else {
            $oMediaEntity = $mediaEntity;
        }

        $this->copyFile($oMedia, $oMediaEntity, $context);

        //Hier muss ich den Eintrag nochmal anpassen damit das Bild auch im Frontend angezeigt wird. Weil private = 1 zeigt er nicht an
        $sFolderId = $this->getMediaDefaultFolderId('theme',$context);
        $mediaRepo->update([
            ['id' => $oMediaEntity->getId(), 'mediaFolderId' => $sFolderId,'private' => false]
        ],$context);

        if($oMediaEntity != null){
            $oMediaCollection = new MediaCollection();
            $oMediaCollection->add($oMediaEntity);
            $this->oThumbnailService->generate($oMediaCollection,$context);
            return $oMediaEntity->getId();
        }
        return null;
    }
    private function getPaymentMethodId(Context $context): ?string
    {
        /** @var EntityRepository $paymentRepository */
        $paymentRepository = $this->container->get('payment_method.repository');

        // Fetch ID for update
        $paymentCriteria = (new Criteria())->addFilter(new EqualsFilter('handlerIdentifier', CreditPlusPayment::class));
        $paymentIds = $paymentRepository->searchIds($paymentCriteria, $context);
        if ($paymentIds->getTotal() === 0) {
            return null;
        }
        return $paymentIds->getIds()[0];
    }

    private function addDefaultDataToDatabase($context): void
    {
        /** @var EntityRepository $sicoProductgroupRepository */
        $sicoProductgroupRepository = $this->container->get(SicoProductgroupDefinition::ENTITY_NAME . '.repository');

        $sProductgroupUUID = Uuid::randomHex();
        if ($sicoProductgroupRepository != null) {
            /** @var SicoProductgroupCollection $collection */
            $collection = $sicoProductgroupRepository->search(
                new Criteria(),
                $context
            )->getEntities();

            if ($collection->count() <= 0) {
                $sicoProductgroupRepository->create([
                    ['id' => $sProductgroupUUID, 'sicoName' => 'Finanzierbare Produkte', 'sicoProductTypeId' => 56, 'sicoProductClassId' => 11]
                ], $context);
            } else {
                $sProductgroupUUID = $collection->first()->getId();
            }
        }

        /** @var EntityRepository $sicoOfferedOptionRepository */
        $sicoOfferedOptionRepository = $this->container->get(SicoOfferedOptionDefinition::ENTITY_NAME . '.repository');
        if ($sicoOfferedOptionRepository != null) {
            /** @var SicoOfferedOptionCollection $collection */
            $collection = $sicoOfferedOptionRepository->search(new Criteria(), $context)->getEntities();
            if ($collection->count() <= 0) {
                $aDefaultMonths = array(
                    6 => 4.99,
                    8 => 4.99,
                    10 => 4.99,
                    12 => 4.99,
                    18 => 4.99,
                    20 => 5.99,
                    24 => 5.99,
                    30 => 5.99,
                    36 => 5.99,
                    42 => 5.99,
                    48 => 5.99,
                    54 => 5.99,
                    60 => 5.99,
                    66 => 5.99,
                    72 => 5.99
                );
                $newData = [];
                foreach ($aDefaultMonths as $iMonth => $fRate) {
                    $newData[] = ['sicoProdgroupId' => $sProductgroupUUID, 'sicoMonths' => $iMonth, 'sicoInterest' => $fRate, 'sicoProdcode' => 'Standard', 'sicoActive' => true, 'sicoRatefactor' => -1];
                }

                $sicoOfferedOptionRepository->create($newData, $context);
            }
        }
    }


    private function getMediaDefaultFolderId(string $folder, Context $context): ?string
    {
        /** @var EntityRepository $mediaFolderRepository */
        $mediaFolderRepository = $this->container->get(MediaFolderDefinition::ENTITY_NAME.'.repository');
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('media_folder.defaultFolder.entity', $folder));
        $criteria->addAssociation('defaultFolder');
        $criteria->setLimit(1);
        /** @var MediaFolderCollection $defaultFolder */
        $defaultFolder = $mediaFolderRepository->search($criteria, $context);
        $defaultFolderId = null;
        if ($defaultFolder->count() === 1) {
            $defaultFolderId = $defaultFolder->first()->getId();
        }

        return $defaultFolderId;
    }

    private function copyFile(MediaFile $oMediaFile, MediaEntity $oMediaEntity, Context $context): void
    {
        if ($oMediaEntity->getPath() == null) {
            $oMediaEntity->setPath('media/' . $oMediaEntity->getId());
        }
        $path = $oMediaEntity->getPath();

        $oSourceFile = $oMediaFile->getFileName();
        $projectDir = $this->container->getParameter('kernel.project_dir').'/';

        //Wenn dieser Ordner existiert, dann befindet sich der aufrufende auf dem root, ansonsten müsste er im public sitzen
        if (file_exists($projectDir . 'public')) {
            $path = $projectDir . 'public/' . $path;
        }

        if (!file_exists($path)) {
            $pathParts = explode('/', $path);
            array_pop($pathParts);
            $dirPath = implode('/', $pathParts);
            $filesystem = new Filesystem();
            $filesystem->mkdir($dirPath);
            file_put_contents($path, 'aaa');//According to the php doc the copy function does not create a file
            copy($oSourceFile, $path);
        }
        $this->fileSaver->persistFileToMedia(
            $oMediaFile,
            pathinfo($oSourceFile, PATHINFO_FILENAME),
            $oMediaEntity->getId(),
            $context
        );
    }

}
