<?php
/**
 * Created by PhpStorm.
 * User: Philipp.Holzmann
 * Date: 30.01.2020
 * Time: 11:51
 */

namespace SicoCreditPlus\Subscriber;
use Shopware\Core\Checkout\Cart\CartPersister;
use Shopware\Core\Checkout\Cart\AbstractCartPersister;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Checkout\Cart\SalesChannel\CartService;
use Shopware\Core\Checkout\Payment\PaymentMethodEntity;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextService;
use Shopware\Core\System\SalesChannel\Event\SalesChannelContextSwitchEvent;
use Shopware\Storefront\Page\Checkout\Cart\CheckoutCartPageLoadedEvent;
use SicoCreditPlus\Errors\CreditPlusNotAvailableError;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use SicoCreditPlus\Components\SicoCreditPlusHelper;
use Symfony\Component\HttpFoundation\Exception\SessionNotFoundException;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\Translation\TranslatorInterface;
use Shopware\Core\System\Snippet\SnippetService;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Struct\ArrayEntity;

class CheckoutCartSubscriber implements EventSubscriberInterface{

	/** @var SicoCreditPlusHelper */
	private $creditPlusHelper;

	/** @var SnippetService  */
	private $translator;

	/** @var SystemConfigService  */
	private $systemConfigService;

	/** @var EntityRepository  */
	private $currencyRepo;

	/** @var EntityRepository */
	private $productRepo;

	/** @var EntityRepository  */
	private $paymentRepo;

	/** @var CartService  */
	private $cartService;

	/** @var CartPersister  */
	private $cartPersister;

	/** @var RequestStack  */
	private $requestStack;

	public function __construct(
		SicoCreditPlusHelper $creditPlusHelper,
		TranslatorInterface $translator,
		SystemConfigService $systemConfigService,
        EntityRepository $currencyRepo,
        EntityRepository $productRepo,
        EntityRepository $paymentRepo,
		CartService $cartService,
        AbstractCartPersister $cartPersister,
		RequestStack $requestStack
	)
	{
		$this->creditPlusHelper = $creditPlusHelper;
		$this->translator = $translator;
		$this->systemConfigService = $systemConfigService;
		$this->currencyRepo = $currencyRepo;
		$this->productRepo = $productRepo;
		$this->paymentRepo = $paymentRepo;
		$this->cartService = $cartService;
		$this->cartPersister = $cartPersister;
		$this->requestStack = $requestStack;
	}

	/**
	 * Returns an array of event names this subscriber wants to listen to.
	 *
	 * The array keys are event names and the value can be:
	 *
	 *  * The method name to call (priority defaults to 0)
	 *  * An array composed of the method name to call and the priority
	 *  * An array of arrays composed of the method names to call and respective
	 *    priorities, or 0 if unset
	 *
	 * For instance:
	 *
	 *  * array('eventName' => 'methodName')
	 *  * array('eventName' => array('methodName', $priority))
	 *  * array('eventName' => array(array('methodName1', $priority), array('methodName2')))
	 *
	 * @return array The event names to listen to
	 */
	public static function getSubscribedEvents()
	{
		return [
			CheckoutCartPageLoadedEvent::class => 'checkoutConfirmPageLoaded',
			SalesChannelContextSwitchEvent::class => 'salesChannelContextSwitched'
		];
	}

	public function salesChannelContextSwitched(SalesChannelContextSwitchEvent $oEvent): void
	{
		$oData = $oEvent->getRequestDataBag();
		$sPaymentMethod = $oData->get(SalesChannelContextService::PAYMENT_METHOD_ID);
		if ( $sPaymentMethod ) {
			/** @var PaymentMethodEntity $cpPayment */
			$cpPayment = $this->paymentRepo->search(
				(new Criteria())->addFilter(
					new EqualsFilter('handlerIdentifier', 'SicoCreditPlus\Service\CreditPlusPayment'))
				,$oEvent->getContext())->first();
			if ( $cpPayment ) {
				if ( $sPaymentMethod !== $cpPayment->getId() ) {
					// Moved away from payment method
					$oRequest = $this->requestStack->getMainRequest();
					try {
						if ( $oRequest && ($oSession = $oRequest->getSession()) ) {
							$oSession->remove('bSicoFinancingExcluded');
						}
					} catch ( SessionNotFoundException $oSNFE ) {
						// Session not found: Doesn't matter
					}
				}
			}
		}
	}

	public function checkoutConfirmPageLoaded(CheckoutCartPageLoadedEvent $event): void {
		$this->creditPlusHelper->setSalesChannelContext($event->getSalesChannelContext());
		$oCart = $event->getPage()->getCart();
		//Es kommt zu einem Fehler, wenn der Warenkorb leer ist
		if($oCart->getLineItems()->count() <= 0){
			/*
			 * Wenn zu dem Warenkorb von SicoTrigger->cancelAndReorder weitergeleitet wird, wird der alte
			 * Token mitgeliefert, um die zum Warenkorb konvertierte Bestellung wiederzufinden
			 */
			$oldToken = $event->getRequest()->get('token');
			if(!empty($oldToken)){
				$oCart = $this->cartService->getCart($oldToken,$event->getSalesChannelContext());
				//Setze den alten Warenkorb mit dem neuen token und speichere ihn ab damit das System diesen auch kennt
				$oCart->setToken($event->getSalesChannelContext()->getToken());
				$this->cartPersister->save($oCart,$event->getSalesChannelContext());
				$event->getPage()->setCart($oCart);
			} else {
				try {
					$event->getRequest()->getSession()->set('bSicoFinancingExcluded', false);
				} catch ( SessionNotFoundException $oSNFE ) {
					// Session not found: Doesn't matter
				}
				//Der Warenkorb ist einfach nur leer.
				return;
			}
		}

		$aArticleNumbers = array();
		/** @var LineItem $oLineItem */
		foreach($oCart->getLineItems() as $oLineItem){
			if ( $oLineItem->getType() !== LineItem::PRODUCT_LINE_ITEM_TYPE ) { continue; }
			$aArticleNumbers[] = $oLineItem->getReferencedId();
		}
		$aArticles = [];
		if ( $aArticleNumbers ) {
			/** @var ProductCollection $aArticles */
			$aArticles =  $this->productRepo->search(new Criteria($aArticleNumbers),$event->getContext())->getEntities();
		}

		//$oBasket = $event->getPage()->
		$oCurrency = $event->getSalesChannelContext()->getCurrency();
		$sCurrency = $event->getContext()->getCurrencyId();
		if($oCurrency == null){
			$oCurrency = $this->currencyRepo->search(new Criteria([$sCurrency]),$event->getContext())->get($sCurrency);
		}
		$oWSCurrency = $this->creditPlusHelper->getWebshopCurrency($oCurrency);

		$oReference = $this->creditPlusHelper->getFinancingArticleReference($event->getSalesChannelContext(),null,$oCart->getLineItems(),$oCart->getPrice()->getTotalPrice());
		if ( !$aArticles ) {
			$oReference = null;
		}
		$bSicoFinancingExcluded = $event->getRequest()->getSession()->get('bSicoFinancingExcluded',false);
        if($oReference == null || $bSicoFinancingExcluded){
			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
				'bShowBasket' => false
			]);
			if($bSicoFinancingExcluded){
				$event->getPage()->getCart()->addErrors(new CreditPlusNotAvailableError('Credit Plus not available Error'));
			}
			$event->getPage()->addExtension('SicoCreditPlus',$entity);
			return;
		}
		if ( $oReference->financingMonths ) {
			$aMonths = $oReference->financingMonths;
		} else {
			$aMonths = $this->creditPlusHelper->getFinancingMonths($event->getSalesChannelContext(),$oCurrency, $aArticles->get($oReference->id),null,$oCart->getPrice()->getTotalPrice());
		}
		$dCheapestInterestRate = $this->creditPlusHelper->getCheapestInterestRate($aMonths);
		$oCheapestMonth = $this->creditPlusHelper->getCheapestRate($aMonths);
		$sMinimalOrderSum = number_format(floatval($this->systemConfigService->get('SicoCreditPlus.config.sMinBasketPrice',$event->getSalesChannelContext()->getSalesChannelId())), $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator);
		$aReplacements = array(
			($oWSCurrency->side == 'left')?$oWSCurrency->sign.' '.$sMinimalOrderSum:$sMinimalOrderSum.' '.$oWSCurrency->sign,
			number_format($dCheapestInterestRate, $oWSCurrency->decimals, $oWSCurrency->decimalSeparator, $oWSCurrency->thousandSeparator)
		);
		$sFinancingDescription = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingDescription');

		$aMarker = array(
			'###MINDESTBESTELLWERT###',
			'###JAHRESZINS###'
		);
		$sFinancingDescription = str_replace($aMarker, $aReplacements, $sFinancingDescription);

		$sCheapestRateText = $this->translator->trans('sicoCreditplus.tabs.DetailsFinancingCheapestRate');
		$aCheapestMarker = array(
			'###PRICE###',
			'###MONTHS###',
			'###LOGOLINK###'
		);
		$sBasePath = $event->getSalesChannelContext()->getSalesChannel()->getDomains()->first()->getUrl();
		if ( substr($sBasePath, -1) !== '/' ) {
			$sBasePath .= '/';
		}
		$aCheapestReplacements = array(
			$oCheapestMonth->monthlyRate,
			$oCheapestMonth->months,
			$sBasePath.'bundles/sicocreditplus/img/creditplus_logo_small.png'
		);
		$sCheapestRateText = str_replace($aCheapestMarker, $aCheapestReplacements, $sCheapestRateText);

		$bShowPAngVText = ($this->translator->trans('sicoCreditplus.tabs.sccppangvtext_text') != '');
		$bShowBasketSetting = intval($this->systemConfigService->get('SicoCreditPlus.config.bShowBasket',$event->getSalesChannelContext()->getSalesChannelId()));
		$bPaymentExcluded = (!$this->creditPlusHelper->isPaymentExcluded($event->getSalesChannelContext(), $oCart, $aMonths));
		$bShowBasket = $bShowBasketSetting && $bPaymentExcluded;
		if($bShowBasket){
			$i = 0;
			$aSicoLowestRates = [];
			if(count($aMonths) >= 3) {
				foreach($aMonths as $oMonth){
					if($i >= (count($aMonths) - 3)){
						$aSicoLowestRates = array_merge($aSicoLowestRates,[$oMonth]);
					}
					$i++;
				}
			} else {
				$aSicoLowestRates = $aMonths;
			}
			$entity = new ArrayEntity([
				'aSicoRateTable' => $aMonths,
				'sSicoFinancingDescription'=> $sFinancingDescription,
				'sSicoCheapestRateText' => $aMonths?$sCheapestRateText:'',
				'bShowPAngVText' => $bShowPAngVText,
				'bShowBasket' => $bShowBasket,
				'aSicoLowestRates' => $aSicoLowestRates
			]);
		}
		else{
			$entity = new ArrayEntity([
				'aSicoRateTable' => array(),
				'sSicoFinancingDescription'=> '',
				'sSicoCheapestRateText' => '',
				'bShowPAngVText' => false,
				'bShowBasket' => $bShowBasket
			]);
		}

		/** @var PaymentMethodEntity $cpPayment */
		$cpPayment = $this->paymentRepo->search(
			(new Criteria())->addFilter(
				new EqualsFilter('handlerIdentifier', 'SicoCreditPlus\Service\CreditPlusPayment'))
			,$event->getContext())->first();
		$entity->assign(['cpPaymentId' => $cpPayment->getId()]);
		$event->getPage()->addExtension('SicoCreditPlus',$entity);
	}
}
